<?php

namespace Colorizer;

add_action('admin_init', function() {
    if( isset($_GET['colorizer_setup_demo']) && current_user_can('administrator') ) {
        $status = setup_demo();
        
        if( is_wp_error($status) ) {
            wp_die( $status->get_error_message() );
        }
    }
}, 100);


function allow_svg_mime($mimes) {
    $mimes['svg'] = 'image/svg+xml';
    return $mimes;
}

function setup_demo() {
    // Creating media files
    $images = [
        'front' => 'front.jpeg', 
        'back' => 'back.jpeg',
        
        'logo_colorizer' => 'logo1.svg',
        'logo_noname' => 'logo2.svg',
        'logo_cloud' => 'logo3.svg',
        
        'front_woman' => 'T-shirt-front-W.png',
        'back_woman' => 'T-shirt-back-W.png',
        'front_man' => 'T-shirt-front-M.png',
        'back_man' => 'T-shirt-back-M.png',
        
        'background' => 'background.jpg'
    ];
    
    // We need to enable upload of SVG files once
    add_filter('upload_mimes', 'Colorizer\allow_svg_mime', 100);
    add_filter('mime_types', 'Colorizer\allow_svg_mime', 100);
    
    $media = [];
    foreach($images as $name => $file) {
        $image_id = add_demo_image_to_media_library($file);
        
        if( is_wp_error($image_id) ) return $image_id;
        
        $media[$name] = $image_id;
    }
    
    // Remove the filter just in case
    remove_filter('upload_mimes', 'Colorizer\allow_svg_mime', 100);
    remove_filter('mime_types', 'Colorizer\allow_svg_mime', 100);
    
    // Create a product
    $product_id = create_variable_product('Colorizer demo product');
    if( is_wp_error($product_id) ) return $product_id;
    
    
    // Creating attributes
    $attributes = [
        'Colorizer Demo Color' => [
          ['name' => 'Red',     'color' => '#c14545'],
          ['name' => 'Orange',  'color' => '#fba53a'],
          ['name' => 'Blue',    'color' => '#18a3e7'],
          ['name' => 'Green',   'color' => '#0ee147'],
          ['name' => 'Black',   'color' => '#000000'],
          ['name' => 'White',   'color' => '#FFFFFF'],
          ['name' => 'Yellow',  'color' => '#feff6c'],
          ['name' => 'Pink',    'color' => '#f993f4'],
        ],
        'Colorizer Demo Logo color' => [
          ['name' => 'Red',     'color' => '#c14545'],
          ['name' => 'Orange',  'color' => '#fba53a'],
          ['name' => 'Blue',    'color' => '#18a3e7'],
          ['name' => 'Green',   'color' => '#0ee147'],
          ['name' => 'Black',   'color' => '#000000'],
          ['name' => 'White',   'color' => '#FFFFFF'],
          ['name' => 'Yellow',  'color' => '#feff6c'],
          ['name' => 'Pink',    'color' => '#f993f4'],
        ],
        'Colorizer Demo Logo' => [
          ['name' => 'Colorizer'],
          ['name' => 'Cloud'],
          ['name' => 'NoName']
        ],
        'Colorizer Demo Model' => [
          ['name' => 'Woman'],
          ['name' => 'Man']
        ]
    ];
    
    $created_attributes = [];
    foreach($attributes as $attribute_name => $terms) {
        $attribute_and_terms = add_attribute_with_values($attribute_name, $terms);
        
        if( is_wp_error($attribute_and_terms) ) return $attribute_and_terms;
        
        $created_attributes[$attribute_and_terms['attribute']->get_name()] = $attribute_and_terms;
    }
    
    $product = wc_get_product($product_id);
    $product->set_image_id($media['front']);
    $product->set_gallery_image_ids( [$media['back']] );
    $product->set_stock_status('instock');
    $product->set_stock_quantity(100);
    $product->set_catalog_visibility('hidden');
    $info = "<h1>IMPORTANT</h1>If this demo doesn't work it is possible that your theme doesn't use the standard WooCommerce gallery. You can check this by using Storefront theme - it should always work with it.";
    $product->set_short_description($info);
    $product->set_description($info);
    
    $attributes = [];
    foreach($created_attributes as $attribute_and_terms) {
        $attributes[] = $attribute_and_terms['attribute']; 
    }
    $product->set_attributes($attributes);
    $product->set_default_attributes([
        'pa_colorizer-demo-logo' => 'cloud',
        'pa_colorizer-demo-color' => 'blue',
        'pa_colorizer-demo-logo-color' => 'white',
        'pa_colorizer-demo-model' => 'man'
    ]);
    
    $variation_post = array( // Setup the post data for the variation
            'post_title'  => 'Colorizer demo variation',
            'post_status' => 'publish',
            'post_parent' =>  $product_id,
            'post_type'   => 'product_variation',
    );

    $variation_id = wp_insert_post($variation_post);
    
    if( is_wp_error($variation_id) ) return $variation_id;
    $variation = wc_get_product($variation_id);
    $variation->set_regular_price('100');
    $variation->set_stock_status('instock');
    $variation->set_stock_quantity(100);
    $variation->save();

    $product->save();
    
    // Colorizer data
    $file = plugin_dir_path(__FILE__) . 'assets/colorizer-export.json';
    $template = file_get_contents($file);
    
    $replacements_array = [];
    // First, images with urls
    foreach($media as $name => $attachment_id) {
        $replacements_array[$name] = wp_get_attachment_url($attachment_id);
    }
    foreach($created_attributes as $attribute_and_terms) {
        $attribute = $attribute_and_terms['attribute'];
        $name = str_replace('pa_', '', $attribute->get_name());
        $replacements_array[$name] = $attribute->get_id();
        $all_ids = [];
        foreach($attribute_and_terms['terms'] as $term) {
            $all_ids[] = $term['id'];
            $replacements_array[$name . '-' . $term['slug']] = $term['id'];
        }
        $replacements_array[$name . '-all'] = implode(', ', $all_ids);
    }

    foreach($replacements_array as $find => $replace) {
        $template = str_replace('{'.$find.'}', $replace, $template);
    }
    update_post_meta($product_id, 'colorizer_data', $template);
    
    wp_safe_redirect( $product->get_permalink() );
}

function create_variable_product($name) {
    $post = array( // Set up the basic post data to insert for our product
        'post_author'  => 1,
        'post_status'  => 'draft',
        'post_title'   => $name,
        'post_type'    => 'product'
    );

    $post_id = wp_insert_post($post, true); // Insert the post returning the new post id

    if ( is_wp_error($post_id) ) {
        return $post_id;
    }

    update_post_meta( $post_id, '_visibility', 'private');
    wp_set_object_terms($post_id, 'variable', 'product_type');
    return $post_id;
}

function add_attribute_with_values($name, $terms) {
    $taxonomy_slug = wc_sanitize_taxonomy_name( $name );
    $taxonomy_name = wc_attribute_taxonomy_name( $taxonomy_slug );
    $already_exists = taxonomy_exists( $taxonomy_name );
    
    $taxonomy_id = 0;
    if( !$already_exists ) {
        $id = wc_create_attribute(['name' => $name]);
        
        if( is_wp_error($id) ) return $id;
        
        $taxonomy_name = wc_attribute_taxonomy_name( $taxonomy_slug );
    
		$taxonomy_data                  = array(
    		'hierarchical'          => false,
    		'update_count_callback' => '_update_post_term_count',
    		'labels'                => array(
				'name'              => sprintf( _x( 'Product %s', 'Product Attribute', 'woocommerce' ), $name ),
				'singular_name'     => $name,
				'search_items'      => sprintf( __( 'Search %s', 'woocommerce' ), $name ),
				'all_items'         => sprintf( __( 'All %s', 'woocommerce' ), $name ),
				'parent_item'       => sprintf( __( 'Parent %s', 'woocommerce' ), $name ),
				'parent_item_colon' => sprintf( __( 'Parent %s:', 'woocommerce' ), $name ),
				'edit_item'         => sprintf( __( 'Edit %s', 'woocommerce' ), $name ),
				'update_item'       => sprintf( __( 'Update %s', 'woocommerce' ), $name ),
				'add_new_item'      => sprintf( __( 'Add new %s', 'woocommerce' ), $name ),
				'new_item_name'     => sprintf( __( 'New %s', 'woocommerce' ), $name ),
				'not_found'         => sprintf( __( 'No &quot;%s&quot; found', 'woocommerce' ), $name ),
    		),
    		'show_ui'            => true,
    		'show_in_quick_edit' => false,
    		'show_in_menu'       => false,
    		'meta_box_cb'        => false,
    		'query_var'          => true,
    		'rewrite'            => false,
    		'sort'               => false,
    		'public'             => true,
    		'show_in_nav_menus'  => true,
    		'capabilities'       => array(
    			'manage_terms' => 'manage_product_terms',
    			'edit_terms'   => 'edit_product_terms',
    			'delete_terms' => 'delete_product_terms',
    			'assign_terms' => 'assign_product_terms',
    		),
    	);
    
    	$result = register_taxonomy( $taxonomy_name, apply_filters( "woocommerce_taxonomy_objects_{$taxonomy_name}", array( 'product' ) ), apply_filters( "woocommerce_taxonomy_args_{$taxonomy_name}", $taxonomy_data ) );
        if( is_wp_error($result) ) {
            return $result;
        }
    }
    $taxonomy_id = wc_attribute_taxonomy_id_by_name($taxonomy_name);

    /*
        Example term in $terms:
        [
            name => "name",
            color => "#000000"
        ]
    */
    $created_terms = [];
    $terms_array = [];
    foreach( $terms as $term ) {
        $slug = sanitize_title($term['name']);
        $term_array = term_exists($slug, $taxonomy_name);
        
        if( !$term_array ) {
            $inserted = wp_insert_term($term['name'], $taxonomy_name);
            
            if( is_wp_error($inserted) ) return $inserted; 
            
            $term_array = $inserted;
        }
        
        $term_id = $term_array['term_id'];

        if( !empty($term['color']) ) update_term_meta($term_id, 'colorizer_color', $term['color']);
        $created_terms[] = $slug;
        $terms_array[] = [
            'slug' => $slug,
            'id' => $term_id
        ];
    }
    
    $attr_object = new \WC_Product_Attribute();
    $attr_object->set_id($taxonomy_id);
    $attr_object->set_name($taxonomy_name);
    $attr_object->set_options($created_terms);
    $attr_object->set_visible(true);
    $attr_object->set_variation(true);
    
    return array( 'attribute' => $attr_object, 'terms' => $terms_array);
}

function add_demo_image_to_media_library($name) {
    $file = plugin_dir_path(__FILE__) . 'assets/images/demo/'. $name;
    $filename = basename($file);
    
    $upload_file = wp_upload_bits($filename, null, file_get_contents($file));
    
    if ($upload_file['error']) return new \WP_Error($upload_file['error']);
    
    $wp_filetype = wp_check_filetype($filename, null );
    $attachment = array(
        'post_mime_type' => $wp_filetype['type'],
    	'post_parent' => 0,
    	'post_title' => preg_replace('/\.[^.]+$/', '', $filename),
    	'post_content' => '',
    	'post_status' => 'inherit'
    );
    $attachment_id = wp_insert_attachment( $attachment, $upload_file['file'], 0 );
    	
    if ( is_wp_error($attachment_id) ) return $attachment_id;
    	
    require_once(ABSPATH . "wp-admin" . '/includes/image.php');
    $attachment_data = wp_generate_attachment_metadata( $attachment_id, $upload_file['file'] );
    wp_update_attachment_metadata( $attachment_id,  $attachment_data );
    return $attachment_id;
}
