/* global colorizer */
/* global fabric */
/* global jQuery */
/* global URL */
/* global navigator */
/* global HTMLCanvasElement */

'use strict'
jQuery(function($) {

    // Helper function
    function hexToRGB(hex, alpha) {
        var r = parseInt(hex.slice(1, 3), 16),
            g = parseInt(hex.slice(3, 5), 16),
            b = parseInt(hex.slice(5, 7), 16);

        if (alpha) {
            return "rgba(" + r + ", " + g + ", " + b + ", " + alpha + ")";
        }
        else {
            return "rgb(" + r + ", " + g + ", " + b + ")";
        }
    }

    const $form = $('.variations_form');
    const attributes = colorizer.attributes;
    const models = colorizer.models ? JSON.parse(colorizer.models) : null;
    const thumbnail_size = colorizer.thumbnail_size || {width: 200, height: 200, crop: 1};
    const ieOrEdge = document.documentMode || /Edge/.test(navigator.userAgent);
    if (!$form.length || !models || !models.length) return;

    if ( !ieOrEdge && fabric.enableGLFiltering &&  fabric.isWebglSupported && fabric.isWebglSupported(fabric.textureSize) ) fabric.filterBackend = new fabric.WebglFilterBackend({ tileSize: fabric.textureSize });
    else fabric.filterBackend = new fabric.Canvas2dFilterBackend();
    // Ok, there is something to generate

    //Simple hack, we can't allow changing src of image - it causes flickering
    function try_to_replace_wc_set_variation_attr() {
        if ($.fn.wc_set_variation_attr) {
            $.fn._wc_set_variation_attr = $.fn.wc_set_variation_attr;
            $.fn.wc_set_variation_attr = function(attr, value) {
                if (this.attr('block_changing_src')) return;

                this._wc_set_variation_attr(attr, value);
            }
        }
        else {
            setTimeout(try_to_replace_wc_set_variation_attr, 100);
        }
    }
    try_to_replace_wc_set_variation_attr();

    // Add hidden canvases
    var canvases = [];
    for (var i = 0; i < models.length; i++) {
        var $canvas = $('<canvas style="display: none" id="colorizer_canvas' + i + '"></canvas>');
        $('body').append($canvas);
        canvases.push(new fabric.StaticCanvas($canvas.get(0), { renderOnAddRemove: false, stateful: false, enableRetinaScaling: false }));
    }

    // Helper funcstions
    function getAttributeColor(attribute_id, value_id) {
        for (const attribute of attributes) {
            if (attribute.id === attribute_id) {
                for (const value of attribute.values) {
                    if (value.id === value_id && value.color) return value.color;
                }
            }
        }
        return null;
    }

    function getAttributePattern(attribute_id, value_id) {
        for (const attribute of attributes) {
            if (attribute.id === attribute_id) {
                for (const value of attribute.values) {
                    if (value.id === value_id && value.pattern) return value.pattern;
                }
            }
        }
        return null;
    }

    function get_chosen_attributes() {
        var chosen = [];
        for (var attribute of attributes) {
            var value_chosen = jQuery( document.getElementById(attribute.slug) ).val(); // There was a bug when an ID had % in it
            var value_chosen_id = null;
            for (var value of attribute.values) {
                // There might be a pa_ sometimes
                var slug = value.slug.replace(/^pa_/, '');
                var slug_chosen = value_chosen.replace(/^pa_/, '');
                if ( slug === slug_chosen ) {
                    value_chosen_id = value.id
                }
            }
            chosen.push([attribute.id, value_chosen_id]);
        }
        return chosen;
    }

    function update_images(attributes_chosen) {
        let every_attribute_set = true;
        for (let [attribute, value] of attributes_chosen) {
            if (value === null) every_attribute_set = false;
        }

        function update_image_for_model(i, model) {
            const canvas = canvases[i];
            canvas.clear();

            // Count for how many images we have to wait for
            let images_to_add = 0,
                images_added = 0,
                finished_adding_all = false,
                images = [];

            function if_last_set_as_image() {
                if (finished_adding_all && images_added === images_to_add) { // Added all other images
                    // Add all
                    // We have to reverse it, because the first images are added at the begining
                    images.reverse().forEach(function(img) {
                        if (img) canvas.add(img);
                    });
                    console.log(`For model ${i}: Images to add: ${images_added}/${images_to_add}`);
                    replace_image(i, canvas, every_attribute_set);
                }
            }

            // Check for background
            if (model.background) {
                images_to_add++;
                fabric.util.loadImage(model.background, function(img) {
                    var oImg = new fabric.Image(img);
                    canvas.setHeight(oImg.height);
                    canvas.setWidth(oImg.width);
                    images_added++;
                    canvas.setBackgroundImage(oImg, canvas.renderAll.bind(canvas), {
                        top: 0,
                        left: 0,
                        opacity: 1
                    });
                    if_last_set_as_image();
                }, null, {
                    crossOrigin: 'anonymous',
                    top: 0,
                    left: 0,
                });
            }

            function update_image_with_filters(layer, part, image_index, backgroundRGBA, colorFilter, pattern) {
                fabric.util.loadImage(part.src, function(img) {
                    var image = new fabric.Image(img);
                    image.set({
                        'left': part.x,
                        'top': part.y,
                        'scaleX': part.scaleX ? part.scaleX : 1,
                        'scaleY': part.scaleY ? part.scaleY : 1,
                        'angle': part.rotation,
                        'opacity': layer.opacity / 100,
                    });
                    if (backgroundRGBA) {
                        image.set('backgroundColor', backgroundRGBA);
                    }
                    if (colorFilter) {
                        image.filters.push(colorFilter);
                    }
                    if (pattern) {
                        image.filters.push(new fabric.Image.filters.BlendImage({
                            image: pattern,
                            mode: 'multiply',
                            alpha: 0,
                        }));
                    }

                    image.applyFilters();
                    images_added++;
                    images[image_index] = image;
                    if_last_set_as_image();
                }, null, {
                    crossOrigin: 'anonymous'
                });
            }

            for (const layer of model.layers) {
                if (layer.hidden) continue;

                // Set optional color filter
                let colorFilter = null;
                let patternFilter = null;
                let backgroundRGBA = null;
                if (layer.colorizedBy || layer.background) {
                    // Check if this attr is selected
                    for (const [attribute, value] of attributes_chosen) {
                        if (attribute === layer.colorizedBy) {
                            const color = getAttributeColor(attribute, value);
                            if (color) {
                                colorFilter = new fabric.Image.filters.BlendColor({
                                    color: color,
                                    mode: 'tint',
                                    alpha: layer.colorOpacity / 100
                                });
                            }
                            const pattern = getAttributePattern(attribute, value);
                            if (pattern) {
                                patternFilter = new Promise(function(resolve) {
                                    fabric.Image.fromURL(pattern, function(img) {
                                        resolve(img);
                                    });
                                });
                            }
                        }
                        if (attribute === layer.background) {
                            const background = getAttributeColor(attribute, value);
                            if (background) backgroundRGBA = hexToRGB(background, layer.backgroundOpacity / 100);
                        }
                    }
                }

                for (const part of layer.parts) {
                    if (part.hidden) continue;

                    // Check if is hidden because of attributes
                    if (part.hiddenWhen.length) {
                        if (!every_attribute_set) {
                            continue; // Check next part
                        }

                        const attributes_to_hide = {};
                        for (let [attribute, value] of part.hiddenWhen) {
                            attributes_to_hide[attribute] = value;
                        }

                        const hide = attributes_chosen.every(function([attr, val]) {
                            return attributes_to_hide[attr] && attributes_to_hide[attr].includes(val)
                        });
                        if (hide) continue; // Check next part
                    }

                    // Part should be visible
                    images_to_add++;

                    (function(image_index) {
                        if (patternFilter instanceof Promise) {
                            patternFilter.then(function(filter) {
                                update_image_with_filters(layer, part, image_index, backgroundRGBA, colorFilter, filter);
                            });
                        } else {
                            update_image_with_filters(layer, part, image_index, backgroundRGBA, colorFilter, false);
                        }
                    })(images_to_add);
                }
            }

            finished_adding_all = true;
            // Images might have been already added
            if_last_set_as_image();
        }

        let i = 0;
        for (const model of models) {
            update_image_for_model(i, model);
            i++;
        }
    }

    var $images = $('.woocommerce-product-gallery figure a img');
    if( $images.length === 0 ) {
        $images = $('.woocommerce-product-gallery img.attachment-woocommerce_thumbnail');
    }
    $images.attr('block_changing_src', true); // Disable WooCommerce changing src

    function replace_image(index, canvas, all_variations_chosen) {
        var $image = $images.eq(index),
            $tooltips = $('.woocommerce-product-gallery .flex-control-thumbs img'),
            $tooltip = $tooltips.eq(index),
            image = $image.get(0),
            tooltip = $tooltip.get(0);

        if( !image ) return;

        if( !all_variations_chosen ) {
            const parentSrc = $image.parent().attr('href');
            image.removeAttribute('srcset');
            image.setAttribute('data-large_image', parentSrc);
            image.setAttribute('data-src', parentSrc);
            image.src = parentSrc;
            if( tooltip ) {
                tooltip.src = parentSrc;
            }
            const zoom = $image.parent().next().get(0);
            if( zoom ) {
                zoom.src = parentSrc;
            }
            return;
        }

        function assignImgAttr(img) {
            image.src = img;
            image.setAttribute('data-large_image', img);
            image.setAttribute('data-src', img);
            image.width = canvas.width;
            image.height = canvas.height;
            image.removeAttribute('srcset');
            image.setAttribute('data-large_image_height', canvas.height);
            image.setAttribute('data-large_image_width', canvas.width);

            setTimeout(function() {
                const zoom = $image.parent().next().get(0);
                if( zoom ) {
                    zoom.src = img;
                    zoom.width = canvas.width;
                    zoom.height = canvas.height;
                }
            }, 200);

            if( tooltip ) {
                tooltip.src = img;
                tooltip.setAttribute('block_changing_src', true);
            }
        }

        canvas.renderAll();

        if( URL && URL.createObjectURL && HTMLCanvasElement.prototype.toBlob ) {
            canvas.lowerCanvasEl.toBlob(function(blob){
                const img = URL.createObjectURL(blob);

                assignImgAttr(img);
            }, 'image/png', 0.8)
        } else {
            const img = canvas.lowerCanvasEl.toDataURL('image/png', 0.8);
            assignImgAttr(img);
        }
    }

    function debounce(fn, delay) {
        var timer = null;
        return function() {
            var context = this,
                args = arguments;
            clearTimeout(timer);
            timer = setTimeout(function() {
                fn.apply(context, args);
            }, delay);
        };
    };

    // Generating new images on variation change
    const debounced_update_images = debounce(update_images, 100);
    $form.on('change', function(e) {
        debounced_update_images(get_chosen_attributes());
    });

    // Saving image as variation
    const $colorizer_thumbnail_input = $('<input type="hidden" name="colorizer_thumbnail">');
    $form.append($colorizer_thumbnail_input);
    $form.on('submit', function(e) {
        const canvas = canvases[0];
        if( canvas && canvas.getObjects().length ) {
            const multiplier = thumbnail_size.width/canvas.width;
            const url = canvas.toDataURL({
                format:'jpeg',
                quality: 0.8,
                multiplier: multiplier
            });
            $colorizer_thumbnail_input.val(url);
        }
    });
});
