/* global Vue, fabric */

'use strict'

Vue.component('resizable-canvas', {
    props: {
        width: Number,
        height: Number,
    },
    data: function () {
        return {
            // Display
            top: 0,
            left: 0,
            dragging: false,
            initialMouseX: 0,
            initialMouseY: 0,
            fabric: null,
            scale: 100,
            canvasElements: {},
            active: null,
            model: {}
        }
    },
    template: `
    <div @mousedown="startDrag"
         @mouseup="stopDrag"
         @mouseleave="stopDrag"
         @mousemove="drag"
         @wheel.prevent="enlarge"
         class="canvas-wrapper">
      <span class="scale-indicator">{{ scale }}%</span>
      <canvas
        ref="canvas"
        :height="height"
        :width="width"></canvas>
    </div>
  `,
    methods: {
        /* CANVAS SIZE AND POSITION */
        // Dragging
        startDrag: function (e) {
            // Clicking on canvas shouldn't move it
            if(e.target.tagName == 'CANVAS') return;

            this.dragging = true;
            this.initialMouseX = parseInt(e.clientX);
            this.initialMouseY = parseInt(e.clientY);
        },
        stopDrag: function (e) {
            if( !this.dragging ) return;

            this.dragging = false;
            this.left += e.clientX - this.initialMouseX;
            this.top += e.clientY - this.initialMouseY;
            this.moveCanvas();
        },
        drag: function (e) {
            if (!this.dragging) return false;

            const dx = e.clientX - this.initialMouseX,
                  dy = e.clientY - this.initialMouseY;
            this.moveCanvas(dy, dx);
            return false;
        },
        // Resizing
        resize: function () {
            // TODO: Debounce
            var wrapperBounding = this.$el.getBoundingClientRect(),
                canvasBounding = this.$refs.canvas.getBoundingClientRect();
            this.top = (wrapperBounding.height - canvasBounding.height) / 2;
            this.left = (wrapperBounding.width - canvasBounding.width) / 2;
            this.moveCanvas();
        },
        // Moving
        moveCanvas: function(dy = 0, dx = 0) {
            const topString = (this.top+dy) + 'px',
                  leftString = (this.left+dx) + 'px';

            this.$refs.canvas.style.top = topString;
            this.fabric.upperCanvasEl.style.top = topString;
            this.$refs.canvas.style.left = leftString;
            this.fabric.upperCanvasEl.style.left = leftString;
        },
        // Zooming
        enlarge: function(e) {
	        const delta = Math.max(-1, Math.min(1, (e.wheelDelta || -e.detail)));
            this.scale = Math.max(5, this.scale+delta*5); // 5%

            const scale = this.scale/100,
                  transformation = `scale(${scale})`;
            this.$refs.canvas.style.transform = transformation;
            this.fabric.upperCanvasEl.style.transform = transformation;
        },
        // Centering
        centerCanvas: function(e) {
          // First scale
          const backgroundWidth = this.fabric.getWidth(),
                backgroundHeight = this.fabric.getHeight();
          
          const horizontalScale = this.$el.getBoundingClientRect().width/backgroundWidth;
          const verticalScale = this.$el.getBoundingClientRect().height/backgroundHeight;
          // Scale = 5*n
          this.scale = Math.floor(100*Math.min(1,Math.min(horizontalScale, verticalScale))/5)*5;

          const scale = this.scale/100,
                transformation = `scale(${scale})`;
          this.$refs.canvas.style.transform = transformation;
          this.fabric.upperCanvasEl.style.transform = transformation;
          
          // Then center
          const wrapperBounding = this.$el.getBoundingClientRect();
          
          this.top = (wrapperBounding.height - backgroundHeight - 2) / 2;
          this.left = (wrapperBounding.width - backgroundWidth - 2) / 2;
          this.moveCanvas();
        },


        // FABRIC
        moveActive: function([top, left], step = 1) {
          if( this.active ) {
            this.active.set({
              'left': this.active.left + top*step,
              'top': this.active.top + left*step
            });
            this.active.dirty = true;
          }
        },
        changeBackground: function(url) {
          const canvas = this.fabric;
          const self = this;
          fabric.Image.fromURL(url, function(oImg) {
              canvas.setHeight(oImg.height);
              canvas.setWidth(oImg.width);
              canvas.setBackgroundImage(oImg, canvas.renderAll.bind(canvas), {
                  top: 0,
                  left: 0
              });
              self.centerCanvas();
          });
        },
        addPart: function(layerId, partId, setAsActive = false) {
          if( !this.canvasElements[layerId]  ) this.canvasElements[layerId] = {};

          const part = this.getPart(layerId, partId);
          const canvas = this.fabric;
          const self = this;
          const layer = this.model.getLayer(layerId);
          fabric.Image.fromURL(part.src, function(element) {
            element.set({
                left: part.x,
                top: part.y,
                angle: part.rotation,
                scaleX: part.scaleX,
                scaleY: part.scaleY,
                visible: part.visible && layer.visible,
                selectable: part.visible && layer.visible,
                opacity: part.active ? 1 : 0.5,
            });
            element.partId = partId;
            element.layerId = layerId;
            canvas.add(element);
            self.canvasElements[layerId][partId] = element;
            if(setAsActive) self.selectPart(layerId, partId);
            else canvas.renderAll();
          });
        },
        selectPart: function(layerId, partId) {
          const element = this.getCanvasElement(layerId, partId);
          const layer = this.model.getLayer(layerId);

          // Hide previous active element
          if( this.active ) {
            this.active.set('selectable', false).set('opacity', 0.5);
          }
          if( element && layer.visible && element.visible ) {
            element.set('selectable', true).set('opacity', 1);
            this.fabric.setActiveObject(element);
            this.active = element;
          }
          this.fabric.renderAll();
        },
        deletePart: function(layerId, partId) {
          const part = this.getCanvasElement(layerId, partId);
          if( part ) {
            if( this.active == part ) {
              this.active = null;
              this.fabric.discardActiveObject();
            }
            delete this.canvasElements[layerId][partId];
            this.fabric.remove(part);
          }
        },
        deleteLayer: function(layerId) {
          if( this.canvasElements[layerId] ) {
            for( let partId in this.canvasElements[layerId] ) {
              if( this.canvasElements[layerId].hasOwnProperty(partId) ) this.deletePart(layerId, partId);
            }
            delete this.canvasElements[layerId];
          }
          this.fabric.renderAll();
        },
        changeImage: function(layerId, partId) {
          const element = this.getCanvasElement(layerId, partId),
                part = this.getPart(layerId, partId);
          // We can't create an image without a proper src, because it will not be selectable later
          // Possible bug in fabric?
          if( !element ) {
            this.addPart(layerId, partId, part.active);
            return;
          }

          const canvas = this.fabric;
          element.setSrc(part.src, (_) => canvas.renderAll() );
          canvas.renderAll();
        },
        hideOrShowPart: function(layerId, partId) {
          const part = this.getPart(layerId, partId);
          const element = this.getCanvasElement(layerId, partId);
          if( element ) {
            element.set('visible', part.visible)
                   .set('selectable', part.visible);
            if( !part.visible && this.active == element ) {
              this.active = null;
              this.fabric.discardActiveObject();
            }
            // There is another active element
            if( part.visible && this.active != element) {
              element.set('opacity', 0.5);
            }
            this.fabric.renderAll();
          }
        },
        hideOrShowLayer: function(layerId) {
          const layer = this.model.getLayer(layerId);
          const elements = this.canvasElements[layerId];
          if( elements ) {
            for( let partId in elements ) {
              if( !elements.hasOwnProperty(partId) ) continue;

              let element = elements[partId];
              if( element == this.active ) {
                this.fabric.discardActiveObject();
                this.active = null;
              }
              element.set('visible', layer.visible)
                     .set('selectable', false);
            }
            this.fabric.renderAll();
          }
        },
        getCanvasElement: function(layerId, partId) {
          if( !this.canvasElements[layerId] || !this.canvasElements[layerId][partId])
            return null

          return this.canvasElements[layerId][partId];
        },
        getPart: function(layerId, partId) {
          return this.model.getLayer(layerId).getPart(partId);
        },
        setup: function(model) {
          const canvas = this.fabric;
          this.model = model;

          // Editor hooks
          // this.model.eventBus.$on('partAdded', this.addPart);
          this.model.eventBus.$on('backgroundChanged', this.changeBackground);
          this.model.eventBus.$on('partDeleted',  this.deletePart);
          this.model.eventBus.$on('imageChanged', this.changeImage);
          this.model.eventBus.$on('partSelected', this.selectPart);
          this.model.eventBus.$on('layerDeleted', this.deleteLayer);
          this.model.eventBus.$on('partVisibilityChanged', this.hideOrShowPart);
          this.model.eventBus.$on('layerVisibilityChanged', this.hideOrShowLayer);

          canvas.clear();
          this.canvasElements = {};
          if( this.model.background ) this.changeBackground(this.model.background);
          for(let layer of this.model.layers) {
            for(let part of layer.parts) {
              if( part.src ) this.addPart(layer.id, part.id, false);
            }
          }
        }
    },
    beforeDestroy: function () {
        window.removeEventListener('resize', this.resize)
    },
    mounted: function () {
        // Setup fabric
        const canvas = this.fabric = new fabric.Canvas(this.$refs.canvas, {selection: false, transparentCorners: false});
        fabric.Object.prototype.set({
            transparentCorners: false,
            borderColor: '#000',
            cornerColor: '#000',
            cornerStrokeColor: '#FFF',
            borderOpacityWhenMoving: 1
        });

        const self = this;
        canvas.on('selection:cleared', function(e){
          if( self.active ) canvas.setActiveObject(self.active);
        });
        canvas.on('object:modified', function(e) {
          const target = e.target;
          const part = self.getPart(target.layerId, target.partId);
          part.x = target.left;
          part.y = target.top;
          part.rotation = target.angle;
          part.scaleX = target.scaleX;
          part.scaleY = target.scaleY;
        });

        this.centerCanvas();
        window.addEventListener('resize', this.resize);
    }
})
