'use strict'

/* global Vue */
/* global EditorModel */

Vue.directive('sortable', {
  inserted: function (el, binding) {
    new Sortable(el, binding.value || {})
  }
})

Vue.component('colorizer-editor', {
    data: function() {
        return {
            model: new EditorModel()
        }
    },
    template: `
        <div class="colorizer-editor">
            <resizable-canvas ref="canvas" :width="600" :height="500"></resizable-canvas>
            <div class="panel">
                <div class="background">
                    <button type="button" @click="updateBackground">Select background</button>
                </div>
                <ul class="layers" v-sortable="{handle: '.layer-bar', onEnd: sorting }">
                    <layer v-for="layer in model.layers"
                           :item="layer"
                           :key="layer.id"
                           :model="layer"
                           @partSelected="(layerId, partId) => model.selectPart(layerId, partId)"
                           @deleteLayer="(id) => model.deleteLayer(id)"></layer>
                </ul>
                <button type="button" class="add-new" @click="(id) => model.addLayer(id)" ref="newLayer">Add a layer</button>
                <span style="float: right">
                  <button type="button" class="safe" @click="downloadExport">Export</button>
                  <label type="button" class="button dangerous">
                      Import
                      <input class="hidden" type="file" @change="importFromFile" accept=".json">
                  </label>
                </span>
            </div>
        </div>
    `,
    methods: {
      sorting: function(event) {
        this.model.moveLayer(event.oldIndex, event.newIndex)
      },
      updateBackground: function() {
        imageSelect((url) => this.model.changeBackground(url));
      },
      downloadExport: function() {
        function download(filename, text) {
            var element = document.createElement('a');
            element.setAttribute('href', 'data:text/plain;charset=utf-8,' + encodeURIComponent(text));
            element.setAttribute('download', filename);

            element.style.display = 'none';
            document.body.appendChild(element);

            element.click();

            document.body.removeChild(element);
        }
        download('colorizer-export.json', this.model.toJson());
      },
      importFromFile: function(e) {
        const input = e.target;
        const fileReader = new FileReader();
        const self = this;
        fileReader.readAsText(input.files[0]);
        fileReader.onloadend = function(event){
            const text = event.target.result;
            try {
                const parsed = JSON.parse(text);
                const model = EditorModel.fromObject(parsed);
                self.model = model;
                self.$refs.canvas.setup(model);
                self.$emit('modelChanged', model);
            } catch(e) {
                console.log(e);
                alert("Invalid import file!");
            }
        }
      },
      setModel: function(model) {
          this.model = model;
          this.$refs.canvas.setup(this.model);
      }
    }
})