<?php 
/**
 * WooCommerce Core Functions
 *
 * General core functions available on both the front-end and admin.
 *
 * @package /inc/integrations/woocommerce
 * @version 1.0
 */

 if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if( ! function_exists( 'pls_woocommerce_setup' ) ) :
	function pls_woocommerce_setup() {
		
		/* Enable product gallery lightbox */
		if( pls_get_option( 'product-gallery-lightbox', 1 ) ){
			add_theme_support( 'wc-product-gallery-lightbox' );
		}
				
		/* Force displaying variation attributes in the product name (in cart/minicart/checkout) */
		add_filter( 'woocommerce_product_variation_title_include_attributes', '__return_false' );
		
		/* (Optional) Force displaying product variation attributes as separated formatted metadata (in cart/minicart/checkout) */
		add_filter( 'woocommerce_is_attribute_in_product_name', '__return_false' );
	}
	add_action( 'init', 'pls_woocommerce_setup', 15 );
endif;

/* hide default wishlist & compare button on product archive page */
add_filter( 'woosw_button_position_archive', '__return_false' );
add_filter( 'woosc_button_position_archive', '__return_false' );

/* hide default wishlist & compare button on single product page */
add_filter( 'woosw_button_position_single', '__return_false' );
add_filter( 'woosc_button_position_single', '__return_false' );

/**
 * Get My Account Menu
 */
function pls_get_myaccount_menu() {
	$user_roles = array();
	if( is_user_logged_in() ){
		$user_info = get_userdata( get_current_user_id() );
		$user_roles = $user_info->roles;
	}
	$orders  = get_option( 'woocommerce_myaccount_orders_endpoint', 'orders' );
	$account_page_url = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) );
	if ( substr( $account_page_url, - 1, 1 ) != '/' ) {
		$account_page_url .= '/';
	}
	$orders_url   		= $account_page_url . $orders;
	$dashboard_url		= apply_filters('pls_myaccount_dashboard_url', $account_page_url );
	$orders_url			= apply_filters('pls_myaccount_orders_url', $orders_url  );

	$compare_url		= apply_filters('pls_myaccount_compare_url', '');
	$tracking_pageid	= pls_get_option('order-tracking-page', '');
	$order_tracking_url	= apply_filters('pls_myaccount_order_tracking_url', ( ! empty ( $tracking_pageid ) ) ? get_permalink( $tracking_pageid ) : '' );
	$logout_url			= apply_filters('pls_myaccount_logout_url', wc_logout_url() );
	$user_data 			= wp_get_current_user();
	$current_user 		= apply_filters('pls_myaccount_username',$user_data->user_login );	

	$myaccount_menu = array();
	$myaccount_menu['profile'] = array( 
										'link'		=> $dashboard_url,
										'label'	=> esc_html__( 'My Profile', 'modave' ),
								);
	$myaccount_menu['orders'] = array( 
										'link'		=> $orders_url,
										'label'	=> esc_html__( 'My Orders', 'modave' ),
								);
	if( ! empty ( $tracking_pageid ) ):
	$myaccount_menu['order-tracking'] = array( 
										'link'		=> $order_tracking_url,
										'label'	=> esc_html__( 'Order Tracking', 'modave' ),
								);
	 endif;
	
	if ( class_exists( 'WPCleverWoosw' ) ) {
		if ( WPCleverWoosw::get_setting( 'page_myaccount', 'yes' ) === 'yes' ) {
			$wishlist_url 	= $account_page_url.'wishlist';
			$wishlist_url	= apply_filters('pls_myaccount_wishlist_url', $wishlist_url );
			$myaccount_menu['wishlist'] = array( 
										'link'		=> $wishlist_url,
										'label'	=> esc_html__( 'Wishlist', 'modave' ),
								);
		}
	}
	
	if ( class_exists( 'WPCleverWoosc' ) ) {
		if ( WPCleverWoosc::get_setting( 'page_myaccount', 'yes' ) === 'yes' ) {
			$compare_url 	= $account_page_url.'compare';
			$compare_url	= apply_filters('pls_compare_url', $compare_url );
			$myaccount_menu['compare'] = array( 
										'link'		=> $compare_url,
										'label'	=> esc_html__( 'Compare', 'modave' ),
								);
		}
	}
	
	 $myaccount_menu['logout'] = array( 
										'link'		=> $logout_url,
										'label'		=> esc_html__( 'Logout', 'modave' ),
								);
	return apply_filters( 'pls_myaccount_menu', $myaccount_menu );
}

if( ! function_exists( 'pls_manage_woocommerce_hooks' ) ) {
	function pls_manage_woocommerce_hooks() {
		
		$tabs_location = pls_get_post_meta( 'single_product_tabs_location' );
		if( ! $tabs_location || 'default' == $tabs_location ) {
			$tabs_location = pls_get_option( 'single-product-tabs-location', 'after-summary' );
		}
		
		$bought_together_location = pls_get_post_meta( 'product_bought_together_location' );
		if( ! $bought_together_location || 'default' == $bought_together_location ) {
			$bought_together_location = pls_get_option( 'product-bought-together-location', 'summary-bottom' );
		}
		
		/* Shop Loop Categories */
		if( pls_get_option( 'shop-page-categories', 0 ) && pls_is_catalog() ) {
			add_action( 'pls_site_main_container_top', 'pls_shop_page_categories', 10 );
		}
		
		// Remove Product Header
		if( ! pls_get_option( 'products-header', 1 ) ) {
			remove_action( 'woocommerce_before_shop_loop', 'pls_woocommerce_before_shop_loop', 20 );
		}
		
		// Remove Product Sorting
		if( ! pls_get_option( 'products-sorting', 1 ) ) {
			remove_action( 'pls_woocommerce_shop_loop_header_right', 'woocommerce_catalog_ordering', 30 );
		}
		
		// Remove Shop page products rating
		if( ! pls_get_option( 'product-rating', 0 ) ) {
			remove_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_rating', 5	 );
		}

		// Remove Shop page products price
		if( ! pls_get_option( 'product-price', 1 ) ) {
			remove_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_price', 15 );
		}
		
		// Enable catalog mode
		if( pls_get_option( 'catalog-mode', 0 ) ) {
			remove_action( 'pls_wc_product_loop_cart_button', 'woocommerce_template_loop_add_to_cart', 10 );
			remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_add_to_cart', 10 );
			remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart', 30 );
		}
		
		if( ! pls_get_option( 'product-cart-button', 1 ) ) {	
			remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_add_to_cart', 10 );
		}
		
		$product_style 	= pls_get_loop_prop( 'product-style' );
		if ( pls_get_option( 'product-quick-shop', 0 ) && in_array( $product_style, ['product-style-1', 'product-style-2', 'product-style-5'] ) ) { 
			add_filter( 'woocommerce_product_add_to_cart_text', 'pls_quick_shop_button_text', 10 );
		}
		
		if( 'grid-list' == pls_get_loop_prop( 'products_view' ) ) {
			// Add Short Description in List View
			if( pls_get_option( 'product-short-description', 1 ) ) {
				add_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_single_excerpt', 15 );
			}
		}		
		
		add_filter( 'woocommerce_output_related_products_args', 'pls_related_products_args' );
		
		add_filter( 'woocommerce_upsell_display_args', 'pls_related_products_args' );
		
		if ( ! is_user_logged_in() && pls_get_option( 'login-to-see-price', 0 ) ) {
			add_filter( 'woocommerce_get_price_html', 'pls_login_to_see_prices' );  
			add_filter( 'woocommerce_loop_add_to_cart_link', '__return_false' );  

			//Remove Add to Cart Button
        	remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart', 30 );
		}
		
		// Remove product rating
		if( ! pls_get_option( 'single-product-rating', 1 ) ) {
			remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_rating', 8 );
		}
		
		// Remove product short description
		if( ! pls_get_option( 'single-product-short-description', 1 ) ) {
			remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_excerpt', 20 );
		}
		
		// Add Stock Progressbar After Product Image in Hotdeal
		if( 'horizontal' == pls_get_loop_prop( 'product_view_mode' ) ) {
			remove_action( 'woocommerce_before_shop_loop_item_title', 'pls_wc_product_sale_countdown', 15 );
			add_action( 'woocommerce_after_shop_loop_item', 'pls_wc_product_sale_countdown', 20 );
		}
		
		// Get Product Content Layout
		$product_content_layout = pls_get_product_content_layout();
		
		// Product Content Layout
		if ( 'style-2' == $product_content_layout ) {
			// Remove in Old Layout
			remove_action( 'woocommerce_after_single_product_summary', 'woocommerce_output_product_data_tabs', 10 );			
			remove_action( 'woocommerce_single_product_summary', 'pls_wc_single_product_delivery_return', 36 );
			remove_action( 'woocommerce_single_product_summary', 'pls_wc_single_product_ask_question', 36 );
			remove_action( 'woocommerce_single_product_summary', 'pls_wc_single_product_share', 36 );
			remove_action( 'woocommerce_single_product_summary', 'pls_wc_single_product_estimated_delivery', 37 );
			remove_action( 'woocommerce_single_product_summary', 'pls_wc_single_product_return_refund', 37 );
			remove_action( 'woocommerce_single_product_summary', 'pls_wc_single_product_store_location', 37 );
			remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_meta', 40 );
			remove_action( 'woocommerce_single_product_summary', 'pls_wc_single_product_trust_badge', 42 );
			
			// Add in New Layout
			add_action( 'pls_wc_after_single_product_summary', 'pls_wc_single_product_delivery_return', 5 );
			add_action( 'pls_wc_after_single_product_summary', 'pls_wc_single_product_ask_question', 10 );
			add_action( 'pls_wc_after_single_product_summary', 'pls_wc_single_product_share', 15 );
			add_action( 'pls_wc_after_single_product_summary', 'pls_wc_single_product_estimated_delivery', 20 );
			add_action( 'pls_wc_after_single_product_summary', 'pls_wc_single_product_return_refund', 20 );
			add_action( 'pls_wc_after_single_product_summary', 'pls_wc_single_product_store_location', 20 );
			add_action( 'pls_wc_after_single_product_summary', 'woocommerce_template_single_meta', 25 );
			add_action( 'pls_wc_after_single_product_summary', 'pls_wc_single_product_trust_badge', 30 );
			add_action( 'pls_wc_after_single_product_summary', 'woocommerce_output_product_data_tabs', 35 );
		}
				
		// Product bought together location
		if( $bought_together_location == 'summary-bottom' ) {
			add_action( 'woocommerce_single_product_summary', 'pls_bought_together_products', 41 );
		}elseif( $bought_together_location == 'gallery-bottom' ) {			
			add_action( 'woocommerce_before_single_product_summary', 'pls_bought_together_products', 30 );
		}
		
		// Remove product meta
		if( ! pls_get_option( 'single-product-meta', 1 ) ) {
			remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_meta', 40 );
		}
		
		// Disable product tabs title option
		if( ! pls_get_option( 'single-product-tabs-titles', 0 ) ) {
			add_filter( 'woocommerce_product_description_heading', '__return_false', 20 );
			add_filter( 'woocommerce_product_additional_information_heading', '__return_false', 20 );
		}
				
		// Product tabs location
		if( $tabs_location == 'summary-bottom' ) {
			remove_action( 'woocommerce_after_single_product_summary', 'woocommerce_output_product_data_tabs', 10 );
			add_action( 'woocommerce_single_product_summary', 'woocommerce_output_product_data_tabs', 57 );
		}
		
		//Remove yith compare button in loop yith_woocompare_compare_button
		if( class_exists( 'YITH_Woocompare' ) ){
			global $yith_woocompare;
			$yith_woocompare_obj = $yith_woocompare->obj;
			if ( get_option('yith_woocompare_compare_button_in_products_list') == 'yes' ) {
				remove_action( 'woocommerce_after_shop_loop_item', array( $yith_woocompare_obj, 'add_compare_link' ), 20 );
			}
		}
		
		if( is_product() ) {
			
			if( 'default' == pls_get_option( 'layout-upsells-related-recently', 'default' ) ) {
			
				// Add UpSell Products
				if( pls_get_option( 'upsells-products', 1 ) ) {
					add_action( 'woocommerce_after_single_product_summary', 'woocommerce_upsell_display', 15 );
				}
				
				// Add Related Products
				if( pls_get_option( 'related-products', 1 ) ) {
					add_action( 'woocommerce_after_single_product_summary', 'woocommerce_output_related_products', 20 );
				}
				
				// Add Recently Viewed Products
				if( pls_get_option( 'recently-viewed-products', 0 ) ) {
					add_action( 'woocommerce_after_single_product_summary', 'pls_output_recently_viewed_products', 25 );
				}
			} else {
				
				// Add Tabs UpSell/Related/Recently Products
				add_action( 'woocommerce_after_single_product_summary', 'pls_single_products_tabs', 15 );
				
			}
		
			add_filter( 'woocommerce_product_tabs', 'pls_product_tabs', 90 );
			
			add_filter( 'woocommerce_output_related_products_args', 'pls_related_products_args' );
			
			add_filter( 'woocommerce_upsell_display_args', 'pls_related_products_args' );	
		}	
	}
	add_action( 'wp', 'pls_manage_woocommerce_hooks', 1000 );	
}

/**
 * Update ordering options.
 */
function pls_wc_catalog_orderby( $options ) {
	$options = array (
		'menu_order' => esc_html__( 'Default', 'modave' ),
		'popularity' => esc_html__( 'Popularity', 'modave' ),
		'rating'     => esc_html__( 'Average rating', 'modave' ),
		'date'       => esc_html__( 'Latest', 'modave' ),
		'price'      => esc_html__( 'Price: low to high', 'modave' ),
		'price-desc' => esc_html__( 'Price: high to low', 'modave' ),
	);

	return $options;
}
add_filter( 'woocommerce_catalog_orderby', 'pls_wc_catalog_orderby' );

/*The social nextend social login/register*/
if ( ! function_exists( 'pls_social_nextend_social_login' ) ) {
    function pls_social_nextend_social_login() {
		if ( ! defined( 'NSL_PRO_PATH' ) ) {
			if ( class_exists('NextendSocialLogin') ) {
				echo '<div class="social-login-separator"><span>'. esc_html__( 'Or login with', 'modave' ) .'</span></div>';
				echo do_shortcode('[nextend_social_login]');
			}
		}
    }
    add_action( 'woocommerce_login_form_end', 'pls_social_nextend_social_login', 10 );
} 
if ( ! function_exists( 'pls_social_nextend_social_register' ) ) {
    function pls_social_nextend_social_register() {
		if ( ! defined( 'NSL_PRO_PATH' ) ) {
			if ( class_exists( 'NextendSocialLogin' ) ) {
				echo '<div class="social-login-separator"><span>'. esc_html__('Or connect with', 'modave') .'</span></div>';
				echo do_shortcode('[nextend_social_login]');
			}
		}
    }
    add_action( 'woocommerce_register_form_end', 'pls_social_nextend_social_register', 10 );
}

/*
 * Remove Product gallery Lightbox link
 */
function pls_wc_remove_link_on_thumbnails( $html ) {
	return strip_tags( $html,'<div><img>' );
}
add_filter( 'woocommerce_single_product_image_thumbnail_html', 'pls_wc_remove_link_on_thumbnails' );

/*
 * Swap "Regular price" and "Sale price"
 */
function pls_sale_regular_price( $price, $regular_price, $sale_price ) {	
	$price = '<ins>' . ( is_numeric( $sale_price ) ? wc_price( $sale_price ) : $sale_price ) . '</ins> <del aria-hidden="true">' . ( is_numeric( $regular_price ) ? wc_price( $regular_price ) : $regular_price ) . '</del>';
	
    return $price;
}
add_filter( 'woocommerce_format_sale_price', 'pls_sale_regular_price', 10, 3 );

/*
 * Product Summury Price Classs
 */
function pls_woocommerce_product_price_class( $class ) {
	
	global $product;

	if ( $product ) {
		$class .= ' pls-product-price';
	}
	
	return $class;
}
add_filter( 'woocommerce_product_price_class', 'pls_woocommerce_product_price_class' );

if ( ! function_exists( 'pls_quick_shop_button_text' ) ) {
	function pls_quick_shop_button_text( $btn_text ){
		global $product;		
		if ( $product ) {
			if ( $product->is_type( 'variable' ) ) {
				$quick_shop_text	= apply_filters( 'pls_quick_shop_text', esc_html__( 'Quick Add', 'modave' ) );
				$btn_text = $product->is_purchasable() ? $quick_shop_text : esc_html__( 'Read more', 'modave' );
			}
		}		
		return $btn_text;
	}
}

if ( ! function_exists( 'pls_login_to_see_prices' ) ) {
	function pls_login_to_see_prices() {
		if( is_user_logged_in() ) {
			return;
		}
		$login_to_prices_text = apply_filters( 'pls_login_to_prices_text', esc_html__('Login to see price','modave'));
		$account_page_id 	= get_option( 'woocommerce_myaccount_page_id' );
		$account_page_url 	= !empty( $account_page_id ) ? get_permalink( $account_page_id ) : '#';
		return '<a href="'.esc_url( $account_page_url ).'" class="pls-login-to-see-prices pls-login-btn">' . $login_to_prices_text . '</a>';
	}
}

if ( ! function_exists( 'pls_show_login_to_price' ) ) {
	function pls_show_login_to_price(){
		if( ! is_user_logged_in() && pls_get_option( 'login-to-see-price', 0 ) ) {
			return false;
		}
		return true;
	}
}

if ( ! function_exists( 'pls_enable_variation_price_change' ) ){
	function pls_enable_variation_price_change(){
		$status = true;
		return apply_filters('pls_enable_variation_price_change', $status );
	}
}

if ( ! function_exists( 'pls_get_recently_viewed_products' ) ){
	function pls_get_recently_viewed_products(){
		$viewed_products = ! empty( $_COOKIE['woocommerce_recently_viewed'] ) ? (array) explode( '|', wp_unslash( $_COOKIE['woocommerce_recently_viewed'] ) ) : array(); // @codingStandardsIgnoreLine
		$viewed_products = array_reverse( array_filter( array_map( 'absint', $viewed_products ) ) );
		if ( empty( $viewed_products ) ) {
			return array();
		}
		return $viewed_products;
	}
}

if ( ! function_exists( 'pls_wpcpq_price_quantity_title' ) ) {
	/**
	 * Add WPC Price Quantity(Combo Sale) Title
	 */
    function pls_wpcpq_price_quantity_title() {
		if ( class_exists( 'WPCleverWpcpq' ) ) {
			echo '<div class="wpcpq-title"><h5>'. esc_html__( 'Combo', 'modave' ) .'</h5></div>';
		}
    }
}
add_action( 'wpcpq_table_before', 'pls_wpcpq_price_quantity_title', 10 );

/**
 * pls product tabs
 */
if ( !function_exists( 'pls_product_tabs' ) ){
	/**
	 * pls product tabs
	 */
	function pls_product_tabs( $tabs ){
		global $post;
		$product_id 				= $post->ID;
		$prefix 					= PLS_PREFIX;		
		$additional_information 	= pls_get_option( 'product-additional-information-tab', 1 );
		$review_tab 				= pls_get_option( 'product-review-tab', 1 );
		$bought_together 			= pls_get_option( 'single-product-bought-together', 1 );
		$bought_together_location 	= pls_get_post_meta( 'product_bought_together_location' );
		if( ! $bought_together_location || 'default' == $bought_together_location ){
			$bought_together_location = pls_get_option( 'product-bought-together-location', 'summary-bottom' );
		}
		$bought_together_txt = pls_get_option( 'product-bought-together-title', 'Frequently Bought Together' );
		if( ! $review_tab ){
			unset( $tabs['reviews'] ); 
		}
		if( ! $additional_information ){
			unset( $tabs['additional_information'] ); 
		}
		if( $bought_together && $bought_together_location == 'in-tab' ){
			$pids = get_post_meta( $product_id, $prefix.'product_ids', true );
            if ( !empty($pids) ) {
                $tabs['bought_together'] = array(
                    'title' => $bought_together_txt,
                    'priority' => 1,
                    'callback' => 'pls_bought_together_products'
                );
            }
		}
		$enable_custom_tab = get_post_meta( $product_id, $prefix.'enable_custom_tab', true );
		$product_custom_tab_heading = get_post_meta( $product_id, $prefix.'product_custom_tab_heading', true );
		if ($enable_custom_tab && !empty($product_custom_tab_heading) ) {
			$tabs['pls_custom_tab'] = array(
				'title' => $product_custom_tab_heading,
				'priority' => 40,
				'callback' => 'pls_custom_tab'
			);
		}
		return $tabs;
	}
}

if ( ! function_exists( 'pls_custom_tab' ) ) {
	/**
	 * PLS Product Custom Tab
	 */
	function pls_custom_tab() {
		global $product;
		$prefix = PLS_PREFIX;
		$product_id = $product->get_id();
		$product_custom_tab_content = get_post_meta( $product_id,$prefix.'product_custom_tab_content', true );
		echo do_shortcode($product_custom_tab_content);
		
	}
}

if ( ! function_exists( 'pls_bought_together_products' ) ) {
	/**
	 * Bought Together Products
	 */
	function pls_bought_together_products() {
		$bought_together = pls_get_option( 'single-product-bought-together', 1 );
		if(!$bought_together){
			return;
		}
		if ( is_singular( 'product' ) ) {
			global $product;
			
			$prefix 			= PLS_PREFIX;
			$product_id 		= $product->get_id();
			$together_products 	= get_post_meta( $product_id, $prefix.'product_ids', true );
			if( empty( $together_products ) ){ return; }
			$together_products = array_merge( array( $product_id ), $together_products );
			
			$args = apply_filters( 'woocommerce_bought_together_products_args', array(
				'post_type'            	=> array( 'product', 'product_variation' ),
				'ignore_sticky_posts'  	=> 1,
				'no_found_rows'        	=> 1,
				'posts_per_page'       	=> -1,
				'orderby' 				=> 'post__in',
				'post__in' 				=> $together_products
			) );
			
			$products 		= new WP_Query( $args );
			$total_price 	= 0;
			$count 			= 0;
			$i 				= 1;
			$max_disply_products = apply_filters( 'pls_display_bought_together_products', 3 );
			$bought_together_txt = pls_get_option( 'product-bought-together-title', 'Frequently Bought Together' );
			if ( $products->have_posts() ) : 
			$args['products'] 				= $products;
			$args['max_disply_products'] 	= $max_disply_products;
			?>
			<div class="pls-wc-message"></div>
			<div class="pls-bought-together-products">
				<h3 class="pls-fbt-title">
					<?php echo apply_filters('pls_bought_together_title', $bought_together_txt ); ?>
				</h3>
				<?php wc_get_template( 'single-product/content-bought-together.php', $args ); ?>
			</div>
			<?php endif;
			wp_reset_postdata();
		}
	}
}

if ( ! function_exists( 'pls_all_add_to_cart' ) ) {
	/**
	 * All Add To Cart Products
	 */
	function pls_all_add_to_cart() {
		
		// phpcs:disable WordPress.Security.NonceVerification.Missing
		$product_id        = apply_filters( 'pls_add_to_cart_product_id', absint( $_POST['product_id'] ) );
		$quantity          = empty( $_POST['quantity'] ) ? 1 : wc_stock_amount( wp_unslash( $_POST['quantity'] ) );
		$variation_id      = empty( $_POST['variation_id'] ) ? 0 : sanitize_text_field( $_POST['variation_id'] );
		$variation         = empty( $_POST['variation'] ) ? array() : $_POST['variation'];
		$passed_validation = apply_filters( 'pls_add_to_cart_validation', true, $product_id, $quantity );
		$product_status    = get_post_status( $product_id );


		if ( $passed_validation && WC()->cart->add_to_cart( $product_id, $quantity, $variation_id, $variation ) && 'publish' === $product_status ) {

			do_action( 'woocommerce_ajax_added_to_cart', $product_id );
			
			if ( 'yes' === get_option( 'woocommerce_cart_redirect_after_add' ) ) {
				wc_add_to_cart_message( array( $product_id => $quantity ), true );
			}

			// Return fragments
			WC_AJAX::get_refreshed_fragments();

		} else {

			// If there was an error adding to the cart, redirect to the product page to show any errors.
			$data = array(
				'error'       => true,
				'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id )
			);

			wp_send_json( $data );

		}
		die();
	}
	add_action( 'wp_ajax_nopriv_pls_all_add_to_cart',  'pls_all_add_to_cart' );
	add_action( 'wp_ajax_pls_all_add_to_cart',  'pls_all_add_to_cart' );
}

if ( ! function_exists( 'pls_ajax_add_to_cart' ) ) {
	/**
	 * Ajax Add to Cart
	 */
	function pls_ajax_add_to_cart(){
		check_ajax_referer( 'pls-add-to-cart', 'nonce' );
		// Get messages
		ob_start();
		wc_print_notices();
		$notices = ob_get_clean();
		
		// Get fragments
		// Get mini cart
		ob_start();

		woocommerce_mini_cart();

		$mini_cart = ob_get_clean();
		
		// Fragments and mini cart are returned
		$data = array(
			'notices' => $notices,
			'fragments' => apply_filters( 'woocommerce_add_to_cart_fragments', array(
						'div.widget_shopping_cart_content' => '<div class="widget_shopping_cart_content">' . $mini_cart . '</div>'
					)
				),
			'cart_hash' => apply_filters( 'woocommerce_add_to_cart_hash', WC()->cart->get_cart_for_session() ? md5( json_encode( WC()->cart->get_cart_for_session() ) ) : '', WC()->cart->get_cart_for_session() )
		);
		wp_send_json( $data );		
		die();
	}
	add_action( 'wp_ajax_pls_ajax_add_to_cart', 'pls_ajax_add_to_cart' );
	add_action( 'wp_ajax_nopriv_pls_ajax_add_to_cart', 'pls_ajax_add_to_cart' );
}

if ( ! function_exists( 'pls_quick_shop_add_to_cart' ) ) {
	/**
	 * Ajax Add to Cart
	 */
	function pls_quick_shop_add_to_cart(){
		
		check_ajax_referer( 'pls-quick-shop', 'nonce' );	
		
		if( isset( $_REQUEST['product_id'] ) ) {
			$product_id = sanitize_text_field( (int) $_REQUEST['product_id'] );
		}
		
		$product    = wc_get_product( $product_id );
		
		if ( ! $product ) {
            return;
        }
		$post = get_post( $product_id );	
		
		setup_postdata( $post );
		ob_start(); ?>
		<div class="woocommerce">
			<div class="pls-quick-shop-content summary entry-summary">
				<div class="pls-qs-product-info ">
					<div class="pls-qs-product-image">
						<?php echo woocommerce_get_product_thumbnail( 'woocommerce_gallery_thumbnail' ); ?>
					</div>
					<div class="pls-qs-product-content">
						<h4 class="pls-qs-product-title">
							<a href="<?php echo esc_url( $product->get_permalink() ); ?>"><?php echo esc_html( $product->get_name() ); ?></a>
						</h4>
						<div class="pls-product-price-info">
							<p class="pls-product-price price">
							<?php echo wp_kses_post( $product->get_price_html() ); ?>						
							</p>
							<?php pls_wc_product_widget_sale_label()?>						
						</div>
						
					</div>
				</div>
				<div class="pls-quick-shop-form">
					<?php woocommerce_template_single_add_to_cart(); ?>
				</div>
			</div>
		</div>
		<?php echo ob_get_clean();
		wp_reset_postdata();
		die();
	}
	add_action( 'wp_ajax_pls_quick_shop_add_to_cart', 'pls_quick_shop_add_to_cart' );
	add_action( 'wp_ajax_nopriv_pls_quick_shop_add_to_cart', 'pls_quick_shop_add_to_cart' );
}

if ( ! function_exists( 'pls_get_products_view' ) ){
	/**
	 * Get Product View
	 */
	function pls_get_products_view(){
		
		$product_view = pls_get_option( 'products-view', 'grid-view' );
		if( isset( $_GET['view'] ) ){
			return sanitize_text_field( $_GET['view'] );
		}
		return $product_view;
	}
}

/**
 * WC Resize Review Avatar
 */
function pls_wc_review_gravatar_size(){
	return '52';
}
add_filter( 'woocommerce_review_gravatar_size', 'pls_wc_review_gravatar_size' );

if ( ! function_exists( 'pls_get_products_columns' ) ){
	/**
	 * Get Product View
	 */
	function pls_get_products_columns(){
		
		$product_col = pls_get_loop_prop( 'products-columns' );
		if( isset( $_GET['view'] ) && pls_is_catalog() ){
			if( 'grid-two-col' == $_GET['view'] ){
				$product_col = 2;
			}
			if( 'grid-three-col' == $_GET['view'] ){
				$product_col = 3;
			}
			if( 'grid-four-col' == $_GET['view'] ){
				$product_col = 4;
			}
			if( 'grid-five-col' == $_GET['view'] ){
				$product_col = 5;
			}
		}
		return $product_col;
	}
}

if ( ! function_exists( 'pls_related_products_args' ) ){
	/**
	 * Related Product args
	 */
	function pls_related_products_args($args){		
		$args['posts_per_page'] = pls_get_option( 'related-upsells-products', 6 );
		return $args;
	}
}

if ( !function_exists( 'pls_get_shop_viewnumbers' ) ){
	/*
	 * Number of Product View
	 */
	function pls_get_shop_viewnumbers(){
		$show_numbers = pls_get_option( 'products-per-page-number', '6, 9, 12, 24, 36, 48' );
		$show_numbers = explode( ',', $show_numbers );
		$show_numbers = array_map( 'trim', $show_numbers );
		return $show_numbers;
	}
}

if ( ! function_exists( 'pls_set_recently_viewed_products' ) ){
	/**
	 * Track Recently Viewed Products 
	 */
	function pls_set_recently_viewed_products() {
		if ( ! is_singular( 'product' ) ) {
			return;
		}

		global $post;

		if ( empty( $_COOKIE['woocommerce_recently_viewed'] ) ) { // @codingStandardsIgnoreLine.
			$viewed_products = array();
		} else {
			$viewed_products = wp_parse_id_list( (array) explode( '|', wp_unslash( $_COOKIE['woocommerce_recently_viewed'] ) ) ); // @codingStandardsIgnoreLine.
		}

		// Unset if already in viewed products list.
		$keys = array_flip( $viewed_products );

		if ( isset( $keys[ $post->ID ] ) ) {
			unset( $viewed_products[ $keys[ $post->ID ] ] );
		}

		$viewed_products[] = $post->ID;

		if ( count( $viewed_products ) > 15 ) {
			array_shift( $viewed_products );
		}

		// Store for session only.
		wc_setcookie( 'woocommerce_recently_viewed', implode( '|', $viewed_products ) );
	}
	add_action( 'template_redirect', 'pls_set_recently_viewed_products', 20 );
}

if ( ! function_exists( 'pls_update_cart_widget_quantity' ) ){
	/** 
	 * MINI Cart Quantity Update
	 */
	function pls_update_cart_widget_quantity(){
		check_ajax_referer( 'pls_nonce', 'nonce' );
		$cart_item_key 	= sanitize_text_field( $_POST['cart_item_key'] );
		$quantity 		= (int) sanitize_text_field( $_POST['quantity'] );
		if( !empty( $cart_item_key ) ){
			$cart =  WC()->cart->get_cart();
			if( isset($cart[$cart_item_key]) ){
				$quantity = apply_filters( 'woocommerce_stock_amount_cart_item', wc_stock_amount( preg_replace( '/[^0-9\.]/', '', $quantity ) ), $cart_item_key );
				if( !($quantity === '' || $quantity === $cart[$cart_item_key]['quantity']) ){
					if( !($cart[$cart_item_key]['data']->is_sold_individually() && $quantity > 1) ){
						WC()->cart->set_quantity( $cart_item_key, $quantity, false );
						$cart_updated = apply_filters( 'woocommerce_update_cart_action_cart_updated', true );
						if( $cart_updated ){
							WC()->cart->calculate_totals();
						}
					}
				}
			}
			WC_AJAX::get_refreshed_fragments();
		}	
	}
	add_action( 'wp_ajax_pls_update_cart_widget_quantity', 'pls_update_cart_widget_quantity' );
	add_action( 'wp_ajax_nopriv_pls_update_cart_widget_quantity', 'pls_update_cart_widget_quantity' );
}

if ( ! function_exists( 'pls_update_checkout_quantity' ) ){
	/** 
	 * Checkout Quantity Update
	 */
	function pls_update_checkout_quantity(){
		check_ajax_referer( 'pls_nonce', 'nonce' );
		$data = array();
        parse_str($_POST['form_data'], $data);
        $cart 			= $data['cart'];
        $cart_updated 	= false;
		$cart_totals  	= isset( $data['cart'] ) ? wp_unslash( $data['cart'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		
		if ( ! WC()->cart->is_empty() && is_array( $cart_totals ) ) {
			foreach ( WC()->cart->get_cart() as $cart_item_key => $values ) {

				$_product = $values['data'];

				// Skip product if no updated quantity was posted.
				if ( ! isset( $cart_totals[ $cart_item_key ] ) || ! isset( $cart_totals[ $cart_item_key ]['qty'] ) ) {
					continue;
				}

				// Sanitize.
				$quantity = apply_filters( 'woocommerce_stock_amount_cart_item', wc_stock_amount( preg_replace( '/[^0-9\.]/', '', $cart_totals[ $cart_item_key ]['qty'] ) ), $cart_item_key );

				if ( '' === $quantity || $quantity === $values['quantity'] ) {
					continue;
				}

				// Update cart validation.
				$passed_validation = apply_filters( 'woocommerce_update_cart_validation', true, $cart_item_key, $values, $quantity );

				// is_sold_individually.
				if ( $_product->is_sold_individually() && $quantity > 1 ) {
					/* Translators: %s Product title. */
					wc_add_notice( sprintf( __( 'You can only have 1 %s in your cart.', 'modave' ), $_product->get_name() ), 'error' );
					$passed_validation = false;
				}

				if ( $passed_validation ) {
					WC()->cart->set_quantity( $cart_item_key, $quantity, false );
					$cart_updated = true;
				}
			}
		}
		
		if ( $cart_updated ) {
            WC()->cart->calculate_totals();            
			WC_AJAX::get_refreshed_fragments();
        }
		die();
	}	
	add_action( 'wp_ajax_pls_update_checkout_quantity','pls_update_checkout_quantity' );
	add_action( 'wp_ajax_nopriv_pls_update_checkout_quantity','pls_update_checkout_quantity' );
}

if( ! function_exists( 'pls_product_quick_view' ) ){
	/**
	 * Product Quick View
	 */
	function pls_product_quick_view(){
		
		check_ajax_referer( 'pls-quick-view', 'nonce' );
		
		if( isset( $_REQUEST['pid'] ) ) {
			$pid = absint( $_REQUEST['pid'] );
		}
		
		global $post, $product;
		$post = get_post( $pid );
		setup_postdata( $post );
		$product = wc_get_product( $post->ID );
		ob_start();
			remove_action( 'woocommerce_product_thumbnails', 'woocommerce_show_product_thumbnails', 20 );
			remove_action( 'pls_wc_product_gallery_bottom', 'pls_wc_single_product_video_btn', 10 );
			remove_action( 'pls_wc_product_gallery_bottom', 'pls_wc_single_product_degree360_btn', 15 );
			if ( ! is_user_logged_in() && pls_get_option( 'login-to-see-price', 0 ) ) {
				add_filter( 'woocommerce_get_price_html', 'pls_login_to_see_prices' );  
				add_filter( 'woocommerce_loop_add_to_cart_link', '__return_false' );  

				//Add to cart btns
				remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_add_to_cart', 10 );
				remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart', 30 );
			}
			if( pls_get_option( 'catalog-mode', 0 ) ) {			
				remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart', 30 );
			}
			
			$summary_hooks = [
				'pls_wc_single_product_visitor_count'      => 21,
				'pls_wc_product_sale_countdown'            => 25,
				'pls_wc_single_product_stock_progress_bar' => 25,
				'pls_wc_single_product_delivery_return'    => 36,
				'pls_wc_single_product_ask_question'       => 36,
				'pls_wc_single_product_share'              => 36,
				'pls_wc_single_product_estimated_delivery' => 37,
				'pls_wc_single_product_return_refund'      => 37,
				'pls_wc_single_product_store_location'        => 37,
				'pls_wc_single_product_trust_badge'        => 42,
			];

			foreach ( $summary_hooks as $function => $priority ) {
				remove_action( 'woocommerce_single_product_summary', $function, $priority );
			}
		
			get_template_part( 'woocommerce/content-quick-view' );
			wp_reset_postdata();
		echo ob_get_clean();
		die();
	}
	add_action( 'wp_ajax_pls_product_quick_view','pls_product_quick_view' );
	add_action( 'wp_ajax_nopriv_pls_product_quick_view','pls_product_quick_view' );
}

if ( !function_exists( 'pls_ajax_get_size_chart' ) ) {
	function pls_ajax_get_size_chart(){
		check_ajax_referer( 'pls-ajax-nonce', 'nonce' );	
		$post_id = isset($_POST['id']) ? (int) sanitize_text_field( $_POST['id'] ) : 0;
		if( $post_id ){
			$content_post = get_post($post_id);
			if( $content_post ){
				$prefix				= PLS_PREFIX;
				$title				= $content_post->post_title;
				$content			= $content_post->post_content;
				$chart_data 		= get_post_meta( $post_id, $prefix.'size_chart_data', true);
				$disable_chart_data	= get_post_meta($post_id, $prefix.'disable_chart_data', true);
				$content 			= apply_filters('the_content', $content);
				$content 			= str_replace(']]>', ']]&gt;', $content);
				$table_html 		= '';
				if( ! empty( $chart_data ) ){
					$chart_table = json_decode($chart_data);
					if ( ! empty( $chart_table ) ) {
						$table_html .= "<table id='size-chart' class='table'>";
						$i = 0;
						foreach ( $chart_table as $chart ) {

							$table_html .= "<tr>";
							for ($j = 0; $j < count($chart); $j++) {
								//If data avaible
								if (!empty($chart[$j])) {
									$table_html .= ($i == 0) ? "<th>" . $chart[$j]. "</th>" : "<td>" .$chart[$j] . "</td>";
								}  else {
									$table_html .= ($i == 0) ? "<th>" . $chart[$j] . "</th>" : "<td></td>";
								}
							}
							$table_html .= "</tr>";
							$i++;
						}
						$table_html .= "</table>";
					}
				}
				$args = array( 'chart_id'=> $post_id, 'title' => $title, 'content' => $content, 'table_html' => $table_html, 'disable_chart_data' => $disable_chart_data );
				wc_get_template( 'content-size-chart.php', $args );
				
			}else{
				echo esc_html__( 'Something wrong..', 'modave' );
			}	
			
		}else{
			echo esc_html__( 'Something wrong..', 'modave' );
		}
		die();
	}
	//Size Chart
	add_action( 'wp_ajax_pls_ajax_get_size_chart', 'pls_ajax_get_size_chart' );
	add_action( 'wp_ajax_nopriv_pls_ajax_get_size_chart', 'pls_ajax_get_size_chart' );
}

if ( ! function_exists( 'pls_ajax_get_block' ) ) {
	function pls_ajax_get_block(){
		check_ajax_referer( 'pls_nonce', 'nonce' );
		$post_id = isset($_POST['id']) ? (int)sanitize_text_field( $_POST['id'] ) : 0;
		$title = isset($_POST['title']) ? sanitize_text_field( $_POST['title'] ) : '';
		if( $post_id ){
			echo '<div class="pls-ajax-blok-content">';
			if( !empty( $title ) ){
				echo '<div class="pls-blok-title">';
				echo esc_html($title);
				echo '</div>';
			}			
			echo do_shortcode('[pls_block_html id="'.$post_id.'"]');
			echo '</div>';
		}else{
			echo esc_html__( 'Something wrong..', 'modave' );
		}
		die();
	}
	//Ajax Block
	add_action( 'wp_ajax_pls_ajax_get_block', 'pls_ajax_get_block' );
	add_action( 'wp_ajax_nopriv_pls_ajax_get_block', 'pls_ajax_get_block' );
}

if ( ! function_exists( 'pls_wc_get_gallery_image_html' ) ) {
	/**
	 * Get Product Gallery Image HTML
	 * @since 1.0
	 */
	function pls_wc_get_gallery_image_html( $attachment_id, $main_image = false ){
		$flexslider        = (bool) apply_filters( 'woocommerce_single_product_flexslider_enabled', get_theme_support( 'wc-product-gallery-slider' ) );
		$gallery_thumbnail = wc_get_image_size( 'gallery_thumbnail' );
		$thumbnail_size    = apply_filters( 'woocommerce_gallery_thumbnail_size', array( $gallery_thumbnail['width'], $gallery_thumbnail['height'] ) );
		$image_size        = apply_filters( 'woocommerce_gallery_image_size', $flexslider || $main_image ? 'woocommerce_single' : $thumbnail_size );
		$full_size         = apply_filters( 'woocommerce_gallery_full_size', apply_filters( 'woocommerce_product_thumbnails_large_size', 'full' ) );
		$thumbnail_src     = wp_get_attachment_image_src( $attachment_id, $thumbnail_size );
		if( empty( $thumbnail_src ) ){
			return;			
		}
		$full_src          = wp_get_attachment_image_src( $attachment_id, $full_size );
		$alt_text          = trim( wp_strip_all_tags( get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ) ) );
		if( $main_image ){
			$image  = wp_get_attachment_image(
				$attachment_id,
				$image_size,
				false,
				apply_filters(
					'woocommerce_gallery_image_html_attachment_image_params',
					array(
						'title'                   => _wp_specialchars( get_post_field( 'post_title', $attachment_id ), ENT_QUOTES, 'UTF-8', true ),
						'data-caption'            => _wp_specialchars( get_post_field( 'post_excerpt', $attachment_id ), ENT_QUOTES, 'UTF-8', true ),
						'data-src'                => esc_url( $full_src[0] ),
						'data-large_image'        => esc_url( $full_src[0] ),
						'data-large_image_width'  => esc_attr( $full_src[1] ),
						'data-large_image_height' => esc_attr( $full_src[2] ),
						'class'                   => esc_attr( $main_image ? 'wp-post-image' : '' ),
					),
					$attachment_id,
					$image_size,
					$main_image
				)
			);
			return '<div data-thumb="' . esc_url( $thumbnail_src[0] ) . '" data-thumb-alt="' . esc_attr( $alt_text ) . '" class="woocommerce-product-gallery__image"><a href="' . esc_url( $full_src[0] ) . '">' . $image . '</a></div>';
		}else{
			return '<div class="woocommerce-product-gallery__thumbnail">'.wp_get_attachment_image( $attachment_id, $image_size ).'</div>';
		}
	}
}

if( ! function_exists( 'pls_wc_gallery_zoom_data' ) ) {
	function pls_wc_gallery_zoom_data( $gallery_style = '' ) {
		global $product;
		if ( !$product ){
			return;
		}
		if( ! pls_get_option( 'product-gallery-zoom', 0 ) ) {
			return;
		}
		$pls_product_image_zoom  = pls_get_option( 'product-gallery-zoom-style', 'inner' ); //magnifier, inner, bounding
		if( $gallery_style == 'product-gallery-horizontal-2' && $pls_product_image_zoom == 'bounding' ){
			$pls_product_image_zoom = 'inner';
		}
		$options = array();
		$options['pls_product_image_zoom'] = $pls_product_image_zoom ;
		$options['containInline'] = true;
		$options['sourceAttribute'] = 'data-large_image';
		
		
		
		if( $pls_product_image_zoom == 'bounding' ){
			$options['hoverBoundingBox'] = true;
			$options['zoomFactor'] = 2;
		}
		if( $pls_product_image_zoom == 'magnifier' ){
			$options['inlinePane'] = true;
			$options['zoomFactor'] = 2;
		}
		
		if( $pls_product_image_zoom == 'inner' ){
			$options['zoomFactor'] = 2;
		}
		$zoom_options = apply_filters( 'pls_wc_gallery_zoom_option', $options );
		
		$data = "data-zoom_options='".esc_attr( wp_json_encode( $zoom_options ) )."'";
		echo apply_filters( 'pls_wc_gallery_zoom_option_data', $data );
	}
}

if( ! function_exists( 'pls_wc_gallery_slider_data' ) ) {
	function pls_wc_gallery_slider_data( $attachment_ids = array(), $gallery_style = '' ) {
		
		$data = '';
		$slider_options = array();
		$slider_style = array('product-thumbnail-left', 'product-thumbnail-right', 'product-thumbnail-bottom', 'product-thumbnail-none', 'product-gallery-horizontal', 'product-gallery-horizontal-2' );
		if( empty( $attachment_ids ) || ! in_array( $gallery_style, $slider_style ) ){
			return;
		}
		
		$slider_options['slidesPerView']	= 1;
		$slider_options['spaceBetween'] 	= 16;
		$slider_options['loop']				= false;
		$slider_options['navigation']		= ( pls_get_option( 'product-gallery-slider-nav', 1 ) ? true : false ) ;
		$slider_options['thumbs']			= true;
		$slider_options['tablet_slidesPerView']	= 1;
		$slider_options['mobile_slidesPerView']	= 1;
		
		if( 'product-gallery-horizontal' == $gallery_style ){
			$slider_options['slidesPerView']	= 3;
			$slider_options['spaceBetween']		= 0;
			$slider_options['tablet_slidesPerView']	= 3;
			$slider_options['mobile_slidesPerView']	= 2;
		}
		
		if( 'product-gallery-horizontal-2' == $gallery_style ){
			$slider_options['slidesPerView']	= 3;
			$slider_options['tablet_slidesPerView']	= 3;
			$slider_options['mobile_slidesPerView']	= 2;
		}
		
		$slider_options = apply_filters( 'pls_wc_galley_get_slider_option', $slider_options );
		
		$data = "data-slider_options='".esc_attr( wp_json_encode( $slider_options ) )."'";
		echo apply_filters( 'pls_wc_galley_slider_option', $data );
	}
}

if( ! function_exists( 'pls_wc_thumbnail_slider_data' ) ) {
	function pls_wc_thumbnail_slider_data( $attachment_ids = array(), $gallery_style = '' ) {
		
		$data = '';
		$slider_options = array();
		
		$slider_options['slidesPerView']		= 6;
		$slider_options['spaceBetween']			= 10;
		$slider_options['loop']					= false;
		$slider_options['navigation']			= ( pls_get_option( 'product-thumbnail-slider-nav', 0 ) ? true : false ) ;
		$slider_options['direction']			= 'horizontal';
		$slider_options['mobile_direction']		= 'horizontal';
		$slider_options['tablet_slidesPerView']	= 6;
		$slider_options['mobile_slidesPerView']	= 5;
		
		if( 'product-thumbnail-left' == $gallery_style || 'product-thumbnail-right' == $gallery_style ){
			$slider_options['direction'] = 'vertical';			
		}
		$slider_options = apply_filters( 'pls_wc_thumbnail_get_slider_option', $slider_options );
		
		$data = "data-slider_options='".esc_attr( wp_json_encode( $slider_options ) )."'";
		echo apply_filters( 'pls_wc_thumbnail_slider_option', $data );	
	}
}

if ( ! function_exists( 'pls_get_image_size_gallery_thumbnail' ) ) {
	function pls_get_image_size_gallery_thumbnail( $size ) {		
		$size['width'] = 90;
		$size['height'] = 0;
		$size['crop']   = 1;		
		return $size;
	}
	add_filter( 'woocommerce_get_image_size_gallery_thumbnail', 'pls_get_image_size_gallery_thumbnail' );
}

if ( ! function_exists( 'pls_default_selectd_attribute_label' ) ) {
	/**
	 * Get Default Selected Attribute Label
	 * @since 1.0
	*/
	function pls_default_selectd_attribute_label( $attribute_name, $product ){
		if( !empty( $attribute_name ) && $product instanceof WC_Product ){
			$selected_key = 'attribute_' . sanitize_title( $attribute_name );
			$selected_slug = isset( $_REQUEST[ $selected_key ] ) ? wc_clean( wp_unslash( $_REQUEST[ $selected_key ] ) ) : $product->get_variation_default_attribute( $attribute_name );
			$term = get_term_by('slug', $selected_slug, $attribute_name ); 
			if( !empty( $term ) ){
				$attribute_label = $term->name; 
				echo '<span class="pls-swatches-selected-label">'.esc_html( $attribute_label ).'</span>';
			}		
		}
	}
}

/**
 * Check enable swatch
 * @since 1.0
*/
function pls_has_enable_switch($attribute_name){
	$prefix 		= PLS_PREFIX;
	$enable_swatch 	= get_option($prefix.$attribute_name.'_enable_swatch',false);
	if( !empty( $enable_swatch ) && $enable_swatch ){
		return true;
	}
	return false;
}

/**
 * Swatch HTML
 * @since 1.0
*/
function pls_swatch_html( $html, $terms, $options, $attribute_name, $selected_attributes, $product ){
	
	if ( isset( $_REQUEST[ 'attribute_' . $attribute_name ] ) ) {
		$selected_value = $_REQUEST[ 'attribute_' . $attribute_name ];
	} elseif ( isset( $selected_attributes[ $attribute_name ] ) ) {
		$selected_value = $selected_attributes[ $attribute_name ];
	} else {
		$selected_value = '';
	}	
	foreach ( $terms as $term ) {
		$html .= pls_get_swatch_html( $term, $selected_value, $attribute_name, $product );
	}
	return $html;
}

/**
 * Get Swatch HTML
 * @since 1.0
*/
function pls_get_swatch_html( $term, $selected_value ='', $attribute_name = '', $product = null ){
	$html 					= '';
	$prefix 				= PLS_PREFIX;
	$swatch_display_style 	= get_option($prefix.$attribute_name.'_swatch_display_style',true);
	$swatch_display_type 	= get_option($prefix.$attribute_name.'_swatch_display_type',true);
	$name     				= esc_html( apply_filters( 'woocommerce_variation_option_name', $term->name ) );
		$selected = sanitize_title( $selected_value ) == $term->slug ? 'swatch-selected' : '';
		if($swatch_display_type == 'color'){			
			$color = get_term_meta( $term->term_id,  $prefix.'color', true );
			list( $r, $g, $b ) = sscanf( $color, "#%02x%02x%02x" );
			$html .= sprintf(
			'<span class="swatch-term swatch swatch-color term-%s swatch-%s %s"  title="%s" data-term="%s"><span class="pls-tooltip" style="background-color:%s;color:%s;">%s</span></span>',
			esc_attr( $term->slug ),
			$swatch_display_style,
			$selected,					
			esc_attr( $name ),
			esc_attr( $term->slug ),
			esc_attr( $color ),
			"rgba($r,$g,$b,0.5)",
			$name
			);
		}else if($swatch_display_type == 'color-label' ){
			$swatch_display_style = 'rounded';
			$color = get_term_meta( $term->term_id,  $prefix.'color', true );
			list( $r, $g, $b ) = sscanf( $color, "#%02x%02x%02x" );
			$html .= sprintf(
			'<span class="swatch-term swatch swatch-color-label term-%s swatch-%s %s"  title="%s" data-term="%s"><span class="pls-rounded-color" style="background-color:%s">%s</span><span class="pls-swatch-text">%s</span></span>',
			esc_attr( $term->slug ),
			$swatch_display_style,
			$selected,					
			esc_attr( $name ),
			esc_attr( $term->slug ),
			esc_attr( $color ),
			$name,
			$name,
			);
		}else if($swatch_display_type == 'image'){
			$image = get_term_meta( $term->term_id, $prefix.'attachment_id', true );
			
			$show_variation_image = apply_filters( 'pls_show_variation_image', true, $term );
			if( $show_variation_image ) {
				$pid 	= $product->get_id();
				$cache_enabled = apply_filters( 'pls_has_swatches_cache', true );
				$transient_name     = 'pls_swatches_cache_' . $pid;
				
				if ( $cache_enabled ) {
					$available_variations = get_transient( $transient_name );
				} else {
					$available_variations = array();
				}
				
				if ( ! $available_variations ) {					
					$available_variations = $product->get_available_variations();
					if ( $cache_enabled ) {
						set_transient( $transient_name, $available_variations, apply_filters( 'pls_swatches_cache_time', WEEK_IN_SECONDS ) );
					}
				}
				if ( empty( $available_variations ) ) {
					return;
				}
				foreach ( $available_variations as $variation ) {
					if ( $variation['attributes'][ 'attribute_' . $attribute_name ] == $term->slug ) {
						$data_var_id = $variation['variation_id'];
					}
				}
				$variation = new WC_Product_Variation( $data_var_id );
				$image_id = $variation->get_image_id(); 
				
				if($image_id){
					$image = $image_id;
				}
			}
			
			$image = $image ? wp_get_attachment_image_src( $image ) : '';
			$image = $image ? $image[0] : WC()->plugin_url() . '/assets/images/placeholder.png';
			$html  .= sprintf(
				'<span class="swatch-term swatch swatch-image term-%s swatch-%s %s" title="%s" data-term="%s"><img src="%s" alt="%s"></span>',
				esc_attr( $term->slug ),
				$swatch_display_style,
				$selected,
				esc_attr( $name ),
				esc_attr( $term->slug ),
				esc_url( $image ),
				esc_attr( $name )
			);
		}else{
			$label = get_term_meta( $term->term_id, 'label', true );
			$label = $label ? $label : $name;
			$html  .= sprintf(
				'<span class="swatch-term swatch swatch-label term-%s swatch-%s %s" title="%s" data-term="%s"><span>%s</span></span>',
				esc_attr( $term->slug ),
				$swatch_display_style,
				$selected,
				esc_attr( $name ),
				esc_attr( $term->slug ),
				esc_html( $label )
			);
		}
	return apply_filters('pls_single_swatch_html',$html,$term,$selected_value);
}

/**
 * Vendor options
*/
function pls_vendor_theme_options(){
	
	$options = array();
	
	if ( class_exists( 'WC_Vendors' ) || class_exists( 'WCMp' ) ) {
		$options[] = array(
					'id'       => 'vendor-page-layout',
					'type'     => 'image_select',
					'title'    => esc_html__( 'Page Layout', 'modave' ),
					'subtitle' => esc_html__( 'Select vendor page layout with sidebar postion.', 'modave' ),
					'options'  => array(
						'full-width' => array(
							'alt' => esc_html__( 'Full Width', 'modave' ),
							'img' => PLS_ADMIN_IMAGES . 'layout/sidebar-none.png',
						),                   
						'left-sidebar' => array(
							'alt' => esc_html__( 'Left Sidebar', 'modave' ),
							'img' => PLS_ADMIN_IMAGES . 'layout/sidebar-left.png',
						), 
						'right-sidebar' => array(
							'alt' => esc_html__( 'Right Sidebar', 'modave' ),
							'img' => PLS_ADMIN_IMAGES . 'layout/sidebar-right.png',
						), 
					),
					'default'  => 'left-sidebar'
				);
		$options[] = array(
					'id'       => 'vendor-page-sidebar',
					'type'     => 'select',
					'title'    => esc_html__('Sidebar','modave'),
					'data'     => 'sidebars',
					'default'  => 'shop-page-sidebar',
					'required' => array( 'vendor-page-layout', '=', array( 'left-sidebar', 'right-sidebar' ) )
				);
	}
	$options[] = array(
                'id'       		=> 'enable-sold-by-in-loop',
                'type'     		=> 'switch',
                'title'    		=> esc_html__( 'Loop Sold By', 'modave' ),
				'subtitle'		=> esc_html__('Display sold by vendor name in loop.', 'modave' ),
                'default'  		=> 1,
                'on'       		=> esc_html__('Yes','modave'),
                'off'      		=> esc_html__('No','modave'),
            );
	$options[] = array(
                'id'       		=> 'enable-sold-by-in-single',
                'type'     		=> 'switch',
                'title'    		=> esc_html__( 'Single Sold By', 'modave' ),
				'subtitle'		=> esc_html__('Display sold by vendor name in single page.', 'modave' ),
                'default'  		=> 1,
                'on'       		=> esc_html__('Yes','modave'),
                'off'      		=> esc_html__('No','modave'),
            );

	if ( class_exists( 'WC_Vendors' ) || class_exists( 'WCFMmp' ) ) {

		$options[] = array(
                'id'       => 'vendor-sold-by-template',
                'type'     => 'select',
                'title'    => esc_html__( 'Sold By Template', 'modave' ),
                'options'  => array(
					'theme'=>esc_html__('By Theme','modave'),
					'plugin'=>esc_html__('By Plugin','modave'),
				),
                'default'  => 'theme',
            );	
	}
	
	return apply_filters('pls_vendor_options', $options);
}

if ( ! function_exists( 'pls_minicart_shipping_html' ) ) {
	
	function pls_minicart_shipping_html(){
		WC()->cart->calculate_shipping();
		WC()->cart->calculate_totals();
		
		$packages = WC()->shipping()->get_packages();
		
		foreach ( $packages as $i => $package ) {
			$available_methods = $package['rates'];
			$package_name =  apply_filters( 'woocommerce_shipping_package_name', ( ( $i + 1 ) > 1 ) ? sprintf( _x( 'Shipping %d', 'shipping packages', 'modave' ), ( $i + 1 ) ) : _x( 'Shipping', 'shipping packages', 'modave' ), $i, $package );
			if( empty( $available_methods ) ){
				return;
			}
			$index			= $i;
			$chosen_method 	= isset( WC()->session->chosen_shipping_methods[ $i ] ) ? WC()->session->chosen_shipping_methods[ $i ] : '';		
			?>
			<tr>
				<th><?php echo wp_kses_post( $package_name ); ?></th>
				<td data-title="<?php echo esc_attr( $package_name ); ?>">
					<ul id="shipping_method" class="woocommerce-shipping-methods">
						<?php foreach ( $available_methods as $method ) : ?>
							<li>
								<?php
								if ( 1 < count( $available_methods ) ) {
									printf( '<input type="radio" name="shipping_method[%1$d]" data-index="%1$d" id="shipping_method_%1$d_%2$s" value="%3$s" class="shipping_method" %4$s />', $index, esc_attr( sanitize_title( $method->id ) ), esc_attr( $method->id ), checked( $method->id, $chosen_method, false ) ); // WPCS: XSS ok.
								} else {
									printf( '<input type="hidden" name="shipping_method[%1$d]" data-index="%1$d" id="shipping_method_%1$d_%2$s" value="%3$s" class="shipping_method" />', $index, esc_attr( sanitize_title( $method->id ) ), esc_attr( $method->id ) ); // WPCS: XSS ok.
								}
								printf( '<label for="shipping_method_%1$s_%2$s">%3$s</label>', $index, esc_attr( sanitize_title( $method->id ) ), wc_cart_totals_shipping_method_label( $method ) ); // WPCS: XSS ok.
								do_action( 'woocommerce_after_shipping_rate', $method, $index );
								?>
							</li>
						<?php endforeach; ?>
					</ul>
				</td>
			</tr>
			<?php
		}
	}
}

/**
 * Mini Cart Features
 */

if ( ! function_exists( 'pls_get_checkout_order_notes' ) ){
	/**
	 * Get Checkout order note
	 */
	function pls_get_checkout_order_notes( $value, $input ) {
		if ( 'order_comments' === $input ) {
			$notes = WC()->session->get( 'pls_order_notes','' );
			if ( ! empty( $notes ) ) {
				return $notes;
			}
		}
		return $value;
	}
}
add_filter( 'woocommerce_checkout_get_value', 'pls_get_checkout_order_notes', 10, 2 );

if ( ! function_exists( 'pls_add_order_note' ) ){
	/**
	 * AJAX Add order note
	 */
	function pls_add_order_note(){
		check_ajax_referer( 'pls-minicart-action', 'nonce' );
		$note = isset($_POST['note']) ? sanitize_textarea_field( $_POST['note'] ) : null;
		WC()->session->set('pls_order_notes' , $note);
		wc_add_notice( esc_html__( 'Your order notes saved.', 'modave' ) );
		
		ob_start();
        wc_print_notices();
        $notices_html = ob_get_clean();
		
		$fragments = array();
		
		ob_start();
        echo '<div class="pls-mini-cart-notices">' . $notices_html . '</div>';
        $fragments['.pls-mini-cart-notices'] = ob_get_clean();
		
		$data = array(
			'fragments' => $fragments,
			'notices_html' => $notices_html,
		);	
		
		wp_send_json($data);
		
	}
}

add_action( 'wp_ajax_nopriv_pls_add_order_note',  'pls_add_order_note' );
add_action( 'wp_ajax_pls_add_order_note',  'pls_add_order_note' );

/*
	PLS Coupon Code
*/
if ( ! function_exists( 'pls_apply_coupon' ) ){
	function pls_apply_coupon(){
		
		check_ajax_referer( 'pls-minicart-action', 'nonce' );
		
		$coupon_code = isset($_POST['coupon_code']) ? wp_unslash( $_POST['coupon_code'] ) : null;

	 
		$is_applied = false;
		if( ! empty( $coupon_code ) ) {
			WC()->cart->add_discount( wc_format_coupon_code( wp_unslash( $coupon_code ) ) );			
		} else {
			wc_add_notice( WC_Coupon::get_generic_coupon_error(WC_Coupon::E_WC_COUPON_PLEASE_ENTER), 'error');
		}
		
		WC()->cart->calculate_totals();
		
		$fragments = array();
		
		ob_start();
        wc_print_notices();
        $notices_html = ob_get_clean();
		
		ob_start();
        echo '<div class="pls-mini-cart-notices">' . $notices_html . '</div>';
        $fragments['.pls-mini-cart-notices'] = ob_get_clean();
		
		ob_start();
		wc_get_template( 'cart/mini-cart-totals.php' );
		$mini_cart_total = ob_get_clean();
		$fragments['.pls-minicart-cart-totals'] = $mini_cart_total;
		
		$data = array(
			'fragments' => $fragments,
			'notices_html' => $notices_html,
		);
		
		wp_send_json($data);
	}
}
add_action( 'wp_ajax_nopriv_pls_apply_coupon',  'pls_apply_coupon' );
add_action( 'wp_ajax_pls_apply_coupon',  'pls_apply_coupon' );

if ( ! function_exists( 'pls_remove_coupon' ) ){
	function pls_remove_coupon(){
		
		check_ajax_referer( 'pls-minicart-action', 'nonce' );
		
		$coupon_code = isset($_POST['coupon_code']) ? wp_unslash( $_POST['coupon_code'] ) : null;

	 
		$is_applied = false;
		if ( empty( $coupon_code ) ) {
			wc_add_notice( __( 'Sorry, Something went wrong!.', 'modave' ), 'error' );
		} else {
			$is_applied = WC()->cart->remove_coupon( $coupon_code );
			wc_add_notice( __( 'Coupon has been removed.', 'modave' ) );
			WC()->cart->calculate_totals();
		}
		
		$fragments = array();
		
		ob_start();
        wc_print_notices();
        $notices_html = ob_get_clean();
		
		ob_start();
        echo '<div class="pls-mini-cart-notices">' . $notices_html . '</div>';
        $fragments['.pls-mini-cart-notices'] = ob_get_clean();
		
		ob_start();
		wc_get_template( 'cart/mini-cart-totals.php' );
		$mini_cart_total = ob_get_clean();
		$fragments['.pls-minicart-cart-totals'] = $mini_cart_total;
		
		$data = array(
			'fragments' => $fragments,
			'notices_html' => $notices_html,
		);
		
		wp_send_json( $data );
	}
}
add_action( 'wp_ajax_nopriv_pls_remove_coupon',  'pls_remove_coupon' );
add_action( 'wp_ajax_pls_remove_coupon',  'pls_remove_coupon' );

/**
 * Shipping method
 */

if ( ! function_exists( 'pls_update_shipping_method' ) ){
	function pls_update_shipping_method() {	
		
		check_ajax_referer( 'pls-minicart-action', 'nonce' );
		
		wc_maybe_define_constant( 'WOOCOMMERCE_CART', true );
		$chosen_shipping_methods = WC()->session->get( 'chosen_shipping_methods' );
		$posted_shipping_methods = isset( $_POST['shipping_method'] ) ? wc_clean( wp_unslash( $_POST['shipping_method'] ) ) : array();

		if ( is_array( $posted_shipping_methods ) ) {
			foreach ( $posted_shipping_methods as $i => $value ) {
				$chosen_shipping_methods[ $i ] = $value;
			}
		}

		WC()->session->set( 'chosen_shipping_methods', $chosen_shipping_methods );
		
		$fragments = array();
		
		wc_add_notice( __( 'Shipping cost updated.', 'modave' ),'notice' );
		
		ob_start();
		wc_print_notices();
		$notices_html = ob_get_clean();	
		
		ob_start();
		echo '<div class="pls-mini-cart-notices">' . $notices_html . '</div>';
		$fragments['.pls-mini-cart-notices'] = ob_get_clean();
		
		ob_start();
		wc_get_template( 'cart/mini-cart-totals.php' );
		$mini_cart_total = ob_get_clean();
		$fragments['.pls-minicart-cart-totals'] = $mini_cart_total;
			
		
		$data = array(
			'fragments' 	=> $fragments,
			'notices_html' 	=> $notices_html,
		);
		wp_send_json($data);		
	}
}
add_action( 'wp_ajax_nopriv_pls_update_shipping_method',  'pls_update_shipping_method' );
add_action( 'wp_ajax_pls_update_shipping_method',  'pls_update_shipping_method' );

if ( ! function_exists( 'pls_wc_calculate_shipping' ) ){
	
	function pls_wc_calculate_shipping() {
		
		wc_maybe_define_constant( 'WOOCOMMERCE_CART', true );

		\WC_Shortcode_Cart::calculate_shipping();

		WC()->cart->calculate_totals();
			
		ob_start();
		wc_print_notices();
		$notices_html = ob_get_clean();		
			
		$fragments = array();
		
		ob_start();
		echo '<div class="pls-mini-cart-notices">' . $notices_html . '</div>';
		$fragments['.pls-mini-cart-notices'] = ob_get_clean();
		
		ob_start();
		wc_get_template( 'cart/mini-cart-totals.php' );
		$mini_cart_total = ob_get_clean();
		$fragments['.pls-minicart-cart-totals'] = $mini_cart_total;
		
		$data = array(
			'fragments' => $fragments,
			'notices_html' => $notices_html,
		);

		wp_send_json($data);
	}
}

add_action( 'wp_ajax_nopriv_pls_wc_calculate_shipping',  'pls_wc_calculate_shipping' );
add_action( 'wp_ajax_pls_wc_calculate_shipping',  'pls_wc_calculate_shipping' );

if ( ! function_exists( 'pls_wpml_multi_currency_compatibility' ) ) {
	function pls_wpml_multi_currency_compatibility( $ajax_actions ) {
		$ajax_actions[] = 'pls_ajax_add_to_cart';		
		$ajax_actions[] = 'pls_product_quick_view';
		$ajax_actions[] = 'pls_quick_shop_add_to_cart';
		$ajax_actions[] = 'pls_update_cart_widget_quantity';
		$ajax_actions[] = 'pls_update_checkout_quantity';
		$ajax_actions[] = 'pls_update_shipping_method';
		$ajax_actions[] = 'pls_wc_calculate_shipping';
		$ajax_actions[] = 'pls_apply_coupon';
		$ajax_actions[] = 'pls_remove_coupon';
		$ajax_actions[] = 'pls_ajax_search';
		$ajax_actions[] = 'pls_all_add_to_cart';
		return $ajax_actions;
	}
	add_filter( 'wcml_multi_currency_ajax_actions', 'pls_wpml_multi_currency_compatibility', 10, 1 );
}

/*WPC Options*/
add_filter('wpcpo_wrap_class','pls_wpcpo_enable_price', 10, 2 );
if ( ! function_exists( 'pls_wpcpo_enable_price' ) ){
	
	function pls_wpcpo_enable_price( $class, $product_id ){	
		$fields = Wpcpo_Frontend::get_fields( (int)$product_id  );
		$has_enable_price = false;	
		$has_prouct_enable = ' pls-hide-price-table';
		foreach( $fields as $data ){
			
			if( isset( $data['enable_price'] ) && $data['enable_price'] == '1' ){
				$has_enable_price = true;
			}
			if($has_enable_price){
				break;
			}		
		}
		
		if( $has_enable_price ){
			$has_prouct_enable = '';
		}
		$class .= $has_prouct_enable;
		return  $class;
	}
}