function loadWebFont(fonts, fontElements = new Map()) {
    if (fonts.length === 0) return;

    WebFont.load({
        google: { families: fonts },
        classes: false,
        fontactive: function (familyName) {
            if (fontElements.has(familyName)) {
                fontElements.get(familyName).removeClass('carregando');
            }
        },
        fontinactive: function (familyName) {
            if (fontElements.has(familyName)) {
                fontElements.get(familyName).removeClass('carregando').addClass('error');
            }
        },
    });
}

async function loadGoogleFontsJson() {
    if (!window.googleFontsList) {
        if (window.googleFontsLoading) {
            await window.googleFontsLoading;
        } else {
            window.googleFontsLoading = new Promise(async (resolve, reject) => {
                try {
                    const response = await fetch(masterstudy_certificate_data.googleFonts);
                    window.googleFontsList = await response.json();

                    if (!window.defaultFontsLoaded) {
                        loadWebFont([
                            "Open Sans:italic,700,700italic,400;700",
                            "Montserrat:italic,700,700italic,400;700",
                            "Katibeh:italic,700,700italic,400;700",
                            "Oswald:italic,700,700italic,400;700"
                        ]);
                        window.defaultFontsLoaded = true;
                    }

                    resolve();
                } catch (error) {
                    reject();
                }
            });

            await window.googleFontsLoading;
        }
    }
}

selectGfont = async function (config) {
    let $selectFont = jQuery(config.containerFonte);
    $selectFont.after('<div class="selectGFontContainer"></div>');
    let $selectGFontContainer = $selectFont.nextAll('.selectGFontContainer');
    let data = [];
    let $selectFontResult;
    let scrollTimer;
    let currentQuery;

    if (!window.loadedFonts) {
        window.loadedFonts = new Set(["Open Sans", "Montserrat", "Katibeh", "Oswald"]);
    }

    await loadGoogleFontsJson();
    let fonts = window.googleFontsList;

    let defaultFont = $selectFont.data('default') || '';

    jQuery.each(fonts, function (index, element) {
        let fontWeightRange = element.minWeight + ',' + element.maxWeight;
        let item = {
            id: index,
            text: element.family + ' (custom)',
            view: element.family,
            category: 'custom',
            weight: fontWeightRange,
        };
        if (element.family === defaultFont) item.selected = 'true';
        data.push(item);
    });

    $selectFont.select2({
        dropdownParent: $selectGFontContainer,
        allowClear: false,
        dropdownAutoWidth: true,
        data: data,
        theme: 'bootstrap selectGFont',
        templateResult: formatSelectFont,
        templateSelection: formatSelectFont,
    });

    let fontId = data.find(item => item.view === defaultFont)?.id;

    if (fontId !== undefined) {
        $selectFont.val(fontId).trigger('change');
        loadFonts({ id: fontId, quantity: 1 });
    }

    $selectFont.on('select2:open', function () {
        $selectFontResult = $selectGFontContainer.find('.select2-results__options');

        setTimeout(() => {
            if (currentQuery?.length) {
                $selectGFontContainer
                    .find('.select2-search .select2-search__field')
                    .val(currentQuery)
                    .trigger('input');
            }

            $selectFontResult.scrollTop(0);
            loadFonts({ position: 0, quantity: 20 });

            if ($selectFontResult.length) {
                $selectFontResult.off('scroll').on('scroll', function () {
                    clearTimeout(scrollTimer);
                    scrollTimer = setTimeout(loadFontsOnScroll, 150);
                });
            }
        }, 0);
    });

    function formatSelectFont(state) {
        if (!state.id) return state.view;
        return jQuery(
            `<span data-id=${state.id} style="font-family:${state.view};">${state.view}</span>
            <span class='select2FontName'>${state.view} | ${state.category} | ${state.weight}</span>`
        );
    }

    function loadFontsOnScroll() {
        if (!$selectFontResult.length) return;
        let firstItem = $selectFontResult.find('li:first-child');
        let position = Math.max(
            0,
            Math.floor(Math.abs(parseInt(firstItem.position().top - 42) / firstItem.outerHeight())) - 3
        );

        setTimeout(() => {
            loadFonts({ position });
        }, 10);
    }

    function loadFonts(config) {
        let position = parseInt(config.position) || 0;
        let quantity = parseInt((config.quantity || 20) + position);
        let id = config.id || false;
        let fontsToLoad = new Set();
        let fontElements = new Map();

        if (id) {
            let font = fonts[id];
            let fontName = `${font.family}:italic,700,700italic,400;700`;

            if (!window.loadedFonts.has(font.family)) {
                fontsToLoad.add(fontName);
                window.loadedFonts.add(font.family);
            }
        } else {
            jQuery('.select2-results li:nth-child(n+' + position + '):nth-child(-n+' + quantity + ') > span:first-child')
                .each(function () {
                    let element = jQuery(this);
                    let fontId = element.data('id');

                    if (fonts[fontId] && fonts[fontId].loaded !== 1) {
                        let font = fonts[fontId];
                        let fontName = `${font.family}:italic,700,700italic,400;700`;

                        if (!window.loadedFonts.has(font.family)) {
                            fontsToLoad.add(fontName);
                            window.loadedFonts.add(font.family);
                            fontElements.set(font.family, element);
                            element.addClass('carregando');
                        }
                    }
                });
        }

        if (fontsToLoad.size > 0) {
            loadWebFont(Array.from(fontsToLoad), fontElements);
        }
    }
};
