<?php
if (!defined('ABSPATH')) {exit;}

/**
 * Helper Class copy data from WP Offload Media to Leopard.
 *
 * @since      3.0.1
 * @package    Leopard_Wordpress_Offload_Media
 * @subpackage Leopard_Wordpress_Offload_Media/includes
 * @author     Nouthemes <nguyenvanqui89@gmail.com>
 */
class Leopard_Wordpress_Offload_Media_AS3CF {

	public function __construct() {
		self::init();
	}

	public static function init() {
		add_action( 'wp_ajax_nou_leopard_offload_media_scan_as3cf_items', array(__CLASS__, 'scan_as3cf_items') );
		add_action( 'wp_ajax_nou_leopard_offload_media_copy_as3cf_items', array(__CLASS__, 'copy_as3cf_items') );
	}

	public static function check_tables()
	{
		global $wpdb;
		$table_name = $wpdb->get_blog_prefix() . 'as3cf_items';;

		$table_exists = $wpdb->get_var( "SHOW TABLES LIKE '{$table_name}'" );
		if ( is_wp_error( $table_exists ) || is_null( $table_exists ) ) {
			return false;
		}

		return true;
	}

	public static function get_as3cf_item($source_id) {
		global $wpdb;
		$table_name = $wpdb->get_blog_prefix() . 'as3cf_items';;
		$query = $wpdb->prepare("SELECT * FROM {$table_name} WHERE source_id = %d", $source_id);
		return $wpdb->get_row($query, ARRAY_A);
	}

	public static function get_table_ids($total_records, $per_page = 10) {
		global $wpdb;

		$table_name = $wpdb->get_blog_prefix() . 'as3cf_items';;
		$total_pages = ceil($total_records / $per_page);
		$ids = [];

		// Query to get all IDs
		for ($page = 1; $page <= $total_pages; $page++) {
			$offset = ($page - 1) * $per_page;
			$sql = "SELECT source_id FROM $table_name LIMIT ". $per_page ." OFFSET " . $offset;
			$new_result = $wpdb->get_col($sql);
        	if (!empty($new_result)) {
				$ids = array_merge($ids, $new_result);
			}
		}
	
		return $ids;
	}

	public static function scan_as3cf_items() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( ['status' => 'fail', 'message' => __( 'You do not have sufficient permissions to access this page.', 'leopard-wordpress-offload-media' )] );
		}

		$nonce = $_POST['_wpnonce'];
		if ( ! wp_verify_nonce( $nonce, 'leopard_wordpress_offload_media_nonce' ) ) {
			wp_send_json_error( ['status' => 'fail', 'message' => __( 'Your nonce did not verify.', 'leopard-wordpress-offload-media' )] );
		}

		if ( ! self::check_tables() ) {
			wp_send_json_error( ['status' => 'fail', 'message' => __( 'The WP Offload Media plugin is not installed or activated.', 'leopard-wordpress-offload-media' )] );
		}

		// Count items in this table.
		global $wpdb;
		$table_name = $wpdb->get_blog_prefix() . 'as3cf_items';
		$count      = $wpdb->get_var( "SELECT COUNT(source_id) FROM {$table_name}" );
		$total     = 0;
		if ( ! is_null( $count ) && intval( $count ) > 0 ) {
			$total = intval( $count );
		}

		$ids = get_option('leopard_as3cf_items_ids');
		if(!is_array($ids)){
			$ids = [];
		}
		
		if(count($ids) == 0){
			$ids = self::get_table_ids($total, 100);
			update_option('leopard_as3cf_items_ids', $ids);
		}

		wp_send_json_success( [
			'status' => 'success', 
			'message' => sprintf( __( '%d items found in the WP Offload Media.', 'leopard-wordpress-offload-media' ), $total )
			] 
		);
	}

	public static function get_leopard_item($source_type, $source_id) {
		$leopard_item = null;

		switch($source_type){
			case "media-library":
				$class 			= leopard_offload_media_get_source_type_name();
				$leopard_item 	= $class::get_by_source_id( $source_id );
				if($leopard_item === false){
					$leopard_item 	= $class::create_from_source_id( $source_id );
				}
				break;
			case "bboss-user-avatar":
			case "bboss-user-cover":
			case "bboss-group-avatar":
			case "bboss-group-cover":
				if ( nou_leopard_offload_media_is_bb_activate() ) {
					$source_types 	= Leopard_Wordpress_Offload_Media_Buddyboss::get_resource_type();
					$class 			= $source_types[$source_type]['class'];
					$leopard_item 	= $class::get_by_source_id( $source_id );
					if($leopard_item === false){
						$leopard_item 	= $class::create_from_source_id( $source_id );
					}
				}
				break;
			default:
				break;
		}

		return $leopard_item;
	}

	public static function copy_as3cf_items() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( ['status' => 'fail', 'message' => __( 'You do not have sufficient permissions to access this page.', 'leopard-wordpress-offload-media' )] );
		}

		$nonce = $_POST['_wpnonce'];
		if ( ! wp_verify_nonce( $nonce, 'leopard_wordpress_offload_media_nonce' ) ) {
			wp_send_json_error( ['status' => 'fail', 'message' => __( 'Your nonce did not verify.', 'leopard-wordpress-offload-media' )] );
		}

		if ( ! self::check_tables() ) {
			wp_send_json_error( ['status' => 'fail', 'message' => __( 'The WP Offload Media plugin is not installed or activated.', 'leopard-wordpress-offload-media' )] );
		}

		$ids = get_option('leopard_as3cf_items_ids');
		if(!is_array($ids)){
			$ids = [];
		}

		if(count($ids) == 0){
			wp_send_json_error( ['status' => 'fail', 'message' => __( 'No items to copy.', 'leopard-wordpress-offload-media' )] );
		}

		// Count items in this table.
		$chunks = array_chunk( $ids, 10 );
		$recordsToProcess = count($chunks);
		$processed  = ( isset( $_POST['processed'] ) ? $_POST['processed'] : 0 );
		$percent = 0;
		$last = 0;

		list( $aws_s3_client, $Bucket, $Region, $basedir_absolute ) = leopard_offload_media_provider_info();

		while (1) {
			// Copy here
			if(isset($chunks[$processed])){
				foreach($chunks[$processed] as $source_id){
					// get by id
					$row = self::get_as3cf_item($source_id);
					if($row){
						$leopard_item = self::get_leopard_item($row['source_type'], $source_id);
						if($leopard_item){
							$extra_info = unserialize($row['extra_info']);
							if(!empty($extra_info['objects'])){
								$object_key = Leopard_Wordpress_Offload_Media_Item::primary_object_key();
								$objects = $extra_info['objects'];
								$objects[$object_key] = $objects['__as3cf_primary'];
								unset($objects['__as3cf_primary']);
								$extra_info['objects'] = $objects;
								$leopard_item->set_provider( $row['provider'] );
								$leopard_item->set_region( $row['region'] );
								$leopard_item->set_bucket( $row['bucket'] );
								$leopard_item->set_path( $row['path'] );
								$leopard_item->set_original_path($row['original_path']);
								$leopard_item->set_extra_info( $extra_info );
							}
							$leopard_item->save();

							try {
								$provider = isset($row['provider']) ? $row['provider'] : 'aws';
								$data = wp_get_attachment_metadata( $source_id );
					
								$provider_object = array(
									'provider' => $provider,
									'region'   => $Region,
									'bucket'   => $Bucket,
									'key' 	   => $leopard_item->source_path(),
									'data'     => $data
								);
								$urlOffloaded = $aws_s3_client->getObjectUrl( $Bucket, $Region, $leopard_item->source_path());
								update_post_meta( $source_id, '_nou_leopard_wom_amazonS3_info', $provider_object );
								update_post_meta( $source_id, '_wp_nou_leopard_wom_s3_wordpress_path', '1' );
								update_post_meta( $source_id, '_wp_nou_leopard_wom_s3_path', $urlOffloaded );
				
							} catch (\Throwable $th) {}
						}
					}
				}
			}

			$processed++;
			$last = $processed;

			if ($processed >= $recordsToProcess ) {
				update_option('leopard_as3cf_items_completed', count($ids));
				update_option('leopard_as3cf_items_ids', []);
				wp_send_json_success(
					array(
						'status' => 'success',
						'count' => $recordsToProcess,
						'message' => esc_html__('Done', 'leopard-wordpress-offload-media'),
						'percent' => round(($last > 0) ? ($last / $recordsToProcess * 100) : 0),
						'processed' => $recordsToProcess
					)
				);
				break;
			}else{
				wp_send_json_success(
					array(
						'status' => 'continue',
						'count' => $recordsToProcess,
						'message' => esc_html__('Continue', 'leopard-wordpress-offload-media'),
						'percent' => round(($last > 0) ? ($last / $recordsToProcess * 100) : 0),
						'processed' => $processed
					)
				);
			}
		}
		
		wp_send_json_success( [
				'status' => 'success', 
				'message' => ''
			] 
		);
	}
}