<?php
/**
 * Plugin Name: EXMAGE - WordPress Image Links Premium
 * Plugin URI:
 * Description: Add images using external links - Save your storage with EXMAGE effortlessly
 * Version: 1.0.1
 * Author: VillaTheme
 * Author URI: https://villatheme.com
 * Text Domain: exmage-wordpress-image-links
 * Domain Path: /languages
 * Copyright 2025 VillaTheme.com. All rights reserved.
 * Requires PHP: 7.0
 * Requires at least: 6.0
 * Tested up to: 6.8.2
 **/

namespace EXMAGE;

use EXMAGE\Admin\EXMAGEAdmin;
use EXMAGE\Frontend\Frontend;
use EXMAGE\Frontend\Shortcode;
use EXMAGE\Includes\EXMAGEScript;

defined( 'ABSPATH' ) || exit;

require_once plugin_dir_path( __FILE__ ) . 'autoload.php';
include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
define( 'EXMAGE_WP_IMAGE_PREMIUM_LINKS_DIR', plugin_dir_path( __FILE__ ) );
define( 'EXMAGE_WP_IMAGE_PREMIUM_LINKS_INCLUDES', EXMAGE_WP_IMAGE_PREMIUM_LINKS_DIR . "includes" . DIRECTORY_SEPARATOR );
require_once EXMAGE_WP_IMAGE_PREMIUM_LINKS_INCLUDES . "define.php";

class EXMAGEEnv {
	private static $args        = [];
	private static $admin_pages = [];

	public static function get( $key ) {
		if ( ! self::$args ) {
			$assets_url = plugins_url( 'assets/', __FILE__ );
			if ( isset( $_SERVER['HTTPS'] ) && ( $_SERVER['HTTPS'] == 'on' || $_SERVER['HTTPS'] == 1 ) || isset( $_SERVER['HTTP_X_FORWARDED_PROTO'] ) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https' ) {
				$protocol = 'https://';
			} else {
				$protocol = 'http://';
			}
			self::$args = [
				'version'       => '1.0.1',
				'plugin_name'   => 'EXMAGE - WordPress Image Links Premium',
				'slug'          => 'exmage-wordpress-image-links',
				'file'          => __FILE__,
				'basename'      => plugin_basename( __FILE__ ),
				'plugin_dir'    => plugin_dir_path( __FILE__ ),
				'templates_dir' => plugin_dir_path( __FILE__ ) . 'templates/',
				'js_url'        => $assets_url . 'js/',
				'css_url'       => $assets_url . 'css/',
				'img_url'       => $assets_url . 'images/',
				'libs_url'      => $assets_url . 'libs/',
				'protocol'      => $protocol,
			];
		}

		return self::$args[ $key ] ? self::$args[ $key ] : '';
	}

	public static function set_page( $key, $value = '' ) {
		if ( ! $value ) {
			$value = $key;
		}
		self::$admin_pages[ $key ] = $value;
	}

	public static function get_pages() {
		return self::$admin_pages;
	}
}

class EXMAGE {

	protected $checker;

	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );

		register_activation_hook( __FILE__, [ $this, 'active' ] );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
	}

	public function init() {
		$include_dir = plugin_dir_path( __FILE__ ) . 'includes/';

		if ( ! class_exists( 'VillaTheme_Require_Environment' ) ) {
			include_once $include_dir . 'support.php';
		}
		$environment = new \VillaTheme_Require_Environment( [
				'plugin_name' => EXMAGEEnv::get( 'plugin_name' ),
				'php_version' => '7.0',
				'wp_version'  => '6.0',
			]
		);

		if ( $environment->has_error() ) {
			return;
		}

		add_filter( 'plugin_action_links_' . EXMAGEEnv::get( 'basename' ), [ $this, 'setting_link' ] );

		$this->load_text_domain();
		$this->load_classes();
	}


	protected function load_classes() {

		require_once plugin_dir_path( __FILE__ ) . 'admin/widgets/widget-selector.php';
		require_once plugin_dir_path( __FILE__ ) . 'includes/functions.php';

		EXMAGEScript::instance();
		Frontend::instance();
		Shortcode::instance();
		if ( is_admin() ) {
			EXMAGEAdmin::instance();
			$this->support();
		}

	}

	public function setting_link( $links ) {
		$settings_link = [
			sprintf( "<a href='%1s' >%2s</a>", esc_url( admin_url( 'admin.php?page=exmage-wordpress-image-links' ) ), esc_html__( 'Settings', 'exmage-wordpress-image-links' ) )
		];

		return array_merge( $settings_link, $links );
	}

	public function load_text_domain() {
		$locale   = determine_locale();
		$locale   = apply_filters( 'plugin_locale', $locale, 'exmage-wordpress-image-links' );
		$basename = plugin_basename( dirname( __FILE__ ) );

		unload_textdomain( 'exmage-wordpress-image-links' );

		load_textdomain( 'exmage-wordpress-image-links', WP_LANG_DIR . "/{$basename}/{$basename}-{$locale}.mo" );
		load_plugin_textdomain( 'exmage-wordpress-image-links', false, $basename . '/languages' );
	}

	public function active() {

		add_action( 'activated_plugin', [ $this, 'activated_plugin' ] );
	}

	/**
	 * When plugin is deactivated, replace and backup metadata attachment
	 */
	public function deactivate() {
		$option = get_option( 'EXMAGE_data', [] );

		if ( isset( $option['enable_replace_media_after_deactive_plugin'] ) && $option['enable_replace_media_after_deactive_plugin'] ) {
			if ( ! empty( $option['placeholder_replace_image'] ) ) {
				$replace_attachment_url = wp_get_attachment_url( $option['placeholder_replace_image'] );

				if ( $replace_attachment_url ) {
					$this->exmage_replace_and_backup_attachments( $replace_attachment_url );

				}
			}

		}
	}


	public function activated_plugin() {

		$option = get_option( 'EXMAGE_data', [] );
		if ( empty( $option['placeholder_replace_image'] ) ) {
			$image_name = 'exmage-placeholder.png';
			$image_path = plugin_dir_path( __FILE__ ) . 'assets/images/' . $image_name;

			$upload_dir = wp_upload_dir();
			// Copy image to Upload folder
			$target_path = $upload_dir['path'] . '/' . $image_name;
			if ( ! copy( $image_path, $target_path ) ) {
				return;
			}
			$attachment = array(
				'post_mime_type' => 'image/png',
				'post_title'     => sanitize_file_name( $image_name ),
				'post_content'   => '',
				'post_status'    => 'inherit'
			);

			$attach_id = wp_insert_attachment( $attachment, $target_path );

			// Generate metadata for image
			require_once( ABSPATH . 'wp-admin/includes/image.php' );
			$attach_data = wp_generate_attachment_metadata( $attach_id, $target_path );
			wp_update_attachment_metadata( $attach_id, $attach_data );
			$option['placeholder_replace_image'] = $attach_id;

			update_option( "EXMAGE_data", $option );
		} else {
			$this->exmage_restore_attachments();
		}

	}

	public function exmage_replace_and_backup_attachments( $replace_attachment_url ) {
		global $wpdb;

		$new_metadata   = array(
			'width'  => 1000,
			'height' => 1000,
			'file'   => esc_url_raw( $replace_attachment_url ),
			'sizes'  => array(),
		);
		$metadata_value = maybe_serialize( $new_metadata );

		// Đổi tên key _wp_attachment_metadata
		$wpdb->query( "
		    UPDATE {$wpdb->postmeta}
		    SET meta_key = '_exmage_original_wp_attachment_metadata'
		    WHERE meta_key = '_wp_attachment_metadata'
		    AND post_id IN (
		        SELECT ID FROM {$wpdb->posts}
		        WHERE post_type = 'attachment'
		        AND post_mime_type LIKE 'image/%'
		        AND ID IN (
		            SELECT post_id FROM {$wpdb->postmeta}
		            WHERE meta_key = '_exmage_external_url'
		        )
		    )
		" );

		// Đổi tên key _wp_attached_file
		$wpdb->query( "
		    UPDATE {$wpdb->postmeta}
		    SET meta_key = '_exmage_original_wp_attached_file'
		    WHERE meta_key = '_wp_attached_file'
		    AND post_id IN (
		        SELECT ID FROM {$wpdb->posts}
		        WHERE post_type = 'attachment'
		        AND post_mime_type LIKE 'image/%'
		        AND ID IN (
		            SELECT post_id FROM {$wpdb->postmeta}
		            WHERE meta_key = '_exmage_external_url'
		        )
		    )
		" );

		// Insert _wp_attachment_metadata mới
		$wpdb->query( $wpdb->prepare( "
		    INSERT INTO {$wpdb->postmeta} (post_id, meta_key, meta_value)
		    SELECT ID, '_wp_attachment_metadata', %s
		    FROM {$wpdb->posts}
		    WHERE post_type = 'attachment'
		    AND post_mime_type LIKE 'image/%'
		    AND ID IN (
		        SELECT post_id FROM {$wpdb->postmeta}
		        WHERE meta_key = '_exmage_external_url'
		    )
		", $metadata_value ) );

		// Insert _wp_attached_file mới
		$wpdb->query( $wpdb->prepare( "
		    INSERT INTO {$wpdb->postmeta} (post_id, meta_key, meta_value)
		    SELECT ID, '_wp_attached_file', %s
		    FROM {$wpdb->posts}
		    WHERE post_type = 'attachment'
		    AND post_mime_type LIKE 'image/%'
		    AND ID IN (
		        SELECT post_id FROM {$wpdb->postmeta}
		        WHERE meta_key = '_exmage_external_url'
		    )
		", esc_url_raw( $replace_attachment_url ) ) );
	}

	function exmage_restore_attachments() {
		global $wpdb;

		$wpdb->query("
	        UPDATE {$wpdb->postmeta} pm
	        INNER JOIN {$wpdb->postmeta} backup
	            ON pm.post_id = backup.post_id
	            AND backup.meta_key = '_exmage_original_wp_attachment_metadata'
	        SET pm.meta_value = backup.meta_value
	        WHERE pm.meta_key = '_wp_attachment_metadata'
	    ");

		// Restore _wp_attached_file từ _exmage_original_wp_attached_file
		$wpdb->query("
	        UPDATE {$wpdb->postmeta} pm
	        INNER JOIN {$wpdb->postmeta} backup
	            ON pm.post_id = backup.post_id
	            AND backup.meta_key = '_exmage_original_wp_attached_file'
	        SET pm.meta_value = backup.meta_value
	        WHERE pm.meta_key = '_wp_attached_file'
	    ");

		// Delete backup _exmage_original_wp_attachment_metadata
		$wpdb->query("
	        DELETE FROM {$wpdb->postmeta}
	        WHERE meta_key IN ('_exmage_original_wp_attachment_metadata', '_exmage_original_wp_attached_file')
	    ");

		// Delete mask replaced
		$wpdb->query("
	        DELETE FROM {$wpdb->postmeta}
	        WHERE meta_key = '_exmage_replaced'
	    ");
	}

	public function support() {
		if ( ! class_exists( 'VillaTheme_Support_Pro' ) ) {
			include_once EXMAGEEnv::get( 'plugin_dir' ) . 'includes/support.php';
		}

		new \VillaTheme_Support_Pro(
			array(
				'support'    => 'https://villatheme.com/supports/forum/plugins/exmage-wordpress-image-links/',
				'docs'       => 'https://docs.villatheme.com/?item=exmage',
				'review'     => 'https://codecanyon.net/downloads',
				'slug'       => EXMAGEEnv::get( 'slug' ),
				'menu_slug'  => EXMAGEEnv::get( 'slug' ),
				'css'        => EXMAGEEnv::get( 'css_url' ),
				'image'      => EXMAGEEnv::get( 'img_url' ),
				'version'    => EXMAGEEnv::get( 'version' ),
				'survey_url' => ''
			)
		);

	}


}

new EXMAGE();

// Plugin Setup Manager 
if ( file_exists( __DIR__ . '/inc/plugin-setup.php' ) ) {
    require_once __DIR__ . '/inc/plugin-setup.php';
}