<?php
require get_template_directory() . '/inc/theme-setup.php';

/**
 * eduvalt functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package eduvalt
 */

if ( !function_exists( 'eduvalt_setup' ) ):
    /**
     * Sets up theme defaults and registers support for various WordPress features.
     *
     * Note that this function is hooked into the after_setup_theme hook, which
     * runs before the init hook. The init hook is too late for some features, such
     * as indicating support for post thumbnails.
     */
    function eduvalt_setup() {
        /*
         * Make theme available for translation.
         * Translations can be filed in the /languages/ directory.
         * If you're building a theme based on eduvalt, use a find and replace
         * to change 'eduvalt' to the name of your theme in all the template files.
         */
        load_theme_textdomain( 'eduvalt', get_template_directory() . '/languages' );

        // Add default posts and comments RSS feed links to head.
        add_theme_support( 'automatic-feed-links' );

        /*
         * Let WordPress manage the document title.
         * By adding theme support, we declare that this theme does not use a
         * hard-coded <title> tag in the document head, and expect WordPress to
         * provide it for us.
         */
        add_theme_support( 'title-tag' );

        /*
         * Enable support for Post Thumbnails on posts and pages.
         *
         * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
         */
        add_theme_support( 'post-thumbnails' );

        // This theme uses wp_nav_menu() in one location.
        register_nav_menus( [
            'main-menu'       => esc_html__( 'Main Menu', 'eduvalt' ),
            'categories-menu' => esc_html__( 'Categories Menu', 'eduvalt' ),
            'footer-menu'     => esc_html__( 'Footer Menu', 'eduvalt' ),
        ] );

        /*
         * Switch default core markup for search form, comment form, and comments
         * to output valid HTML5.
         */
        add_theme_support( 'html5', [
            'search-form',
            'comment-form',
            'comment-list',
            'gallery',
            'caption',
        ] );

        // Set up the WordPress core custom background feature.
        add_theme_support( 'custom-background', apply_filters( 'eduvalt_custom_background_args', [
            'default-color' => 'ffffff',
            'default-image' => '',
        ] ) );

        // Add theme support for selective refresh for widgets.
        add_theme_support( 'customize-selective-refresh-widgets' );

        //Enable custom header
        add_theme_support( 'custom-header' );

        /**
         * Add support for core custom logo.
         *
         * @link https://codex.wordpress.org/Theme_Logo
         */
        add_theme_support( 'custom-logo', [
            'height'      => 250,
            'width'       => 250,
            'flex-width'  => true,
            'flex-height' => true,
        ] );

        /**
         * Enable suporrt for Post Formats
         *
         * see: https://codex.wordpress.org/Post_Formats
         */
        add_theme_support( 'post-formats', [
            'image',
            'audio',
            'video',
            'gallery',
            'quote',
        ] );

        // Add support for Block Styles.
        add_theme_support( 'wp-block-styles' );

        // Add support for full and wide align images.
        add_theme_support( 'align-wide' );

        // Add support for editor styles.
        add_theme_support( 'editor-styles' );

        // Add support for responsive embedded content.
        add_theme_support( 'responsive-embeds' );

        remove_theme_support( 'widgets-block-editor' );

        // Add support for WooCommerce.
        add_theme_support('woocommerce', array(
            'thumbnail_image_width' => 350,
            'gallery_thumbnail_image_width' => 350,
            'single_image_width' => 650,
        ));

        // WooCommerce LightBox
        add_theme_support('wc-product-gallery-lightbox');
        add_theme_support('wc-product-gallery-slider');

        add_image_size( 'eduvalt-image-full', 1170, 600, [ 'center', 'center' ] );
    }
endif;
add_action( 'after_setup_theme', 'eduvalt_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function eduvalt_content_width() {
    // This variable is intended to be overruled from themes.
    // Open WPCS issue: {@link https://github.com/WordPress-Coding-Standards/WordPress-Coding-Standards/issues/1043}.
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
    $GLOBALS['content_width'] = apply_filters( 'eduvalt_content_width', 640 );
}
add_action( 'after_setup_theme', 'eduvalt_content_width', 0 );



/**
 * Enqueue scripts and styles.
 */

define( 'EDUVALT_THEME_DIR', get_template_directory() );
define( 'EDUVALT_THEME_URI', get_template_directory_uri() );
define( 'EDUVALT_THEME_CSS_DIR', EDUVALT_THEME_URI . '/assets/css/' );
define( 'EDUVALT_THEME_JS_DIR', EDUVALT_THEME_URI . '/assets/js/' );
define( 'EDUVALT_THEME_INC', EDUVALT_THEME_DIR . '/inc/' );



// wp_body_open
if ( !function_exists( 'wp_body_open' ) ) {
    function wp_body_open() {
        do_action( 'wp_body_open' );
    }
}

/**
 * Implement the Custom Header feature.
 */
require EDUVALT_THEME_INC . 'custom-header.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require EDUVALT_THEME_INC . 'template-functions.php';

/**
 * Custom template helper function for this theme.
 */
require EDUVALT_THEME_INC . 'template-helper.php';

/**
 * initialize kirki customizer class.
 */
include_once EDUVALT_THEME_INC . 'kirki-customizer.php';
include_once EDUVALT_THEME_INC . 'class-eduvalt-kirki.php';

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
    require EDUVALT_THEME_INC . 'jetpack.php';
}

/**
 * Load Tutor compatibility file.
 */
if (function_exists('tutor')) {
    require EDUVALT_THEME_INC . 'eduvalt-tutor-function.php';
    require EDUVALT_THEME_INC . 'class-eduvalt-tutor-utils.php';
}


/**
 * include eduvalt functions file
 */
require_once EDUVALT_THEME_INC . 'class-navwalker.php';
require_once EDUVALT_THEME_INC . 'class-tgm-plugin-activation.php';
require_once EDUVALT_THEME_INC . 'add_plugin.php';
require_once EDUVALT_THEME_INC . '/common/eduvalt-breadcrumb.php';
require_once EDUVALT_THEME_INC . '/common/eduvalt-scripts.php';
require_once EDUVALT_THEME_INC . '/common/eduvalt-widgets.php';
if (class_exists('WooCommerce')) {
    require_once EDUVALT_THEME_INC . '/woocommerce/eduvalt-woocommerce.php';
}

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function eduvalt_pingback_header() {
    if ( is_singular() && pings_open() ) {
        printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
    }
}
add_action( 'wp_head', 'eduvalt_pingback_header' );

// change textarea position in comment form
// ----------------------------------------------------------------------------------------
function eduvalt_move_comment_textarea_to_bottom( $fields ) {
    $comment_field       = $fields[ 'comment' ];
    unset( $fields[ 'comment' ] );
    $fields[ 'comment' ] = $comment_field;
    return $fields;
}
add_filter( 'comment_form_fields', 'eduvalt_move_comment_textarea_to_bottom' );


// eduvalt_comment
if ( !function_exists( 'eduvalt_comment' ) ) {
    function eduvalt_comment( $comment, $args, $depth ) {
        $GLOBAL['comment'] = $comment;
        extract( $args, EXTR_SKIP );
        $args['reply_text'] = 'Reply';
        $replayClass = 'comment-depth-' . esc_attr( $depth );
        ?>
            <li id="comment-<?php comment_ID();?>">

                <div class="comments-box">

                    <?php if( get_comment_type($comment) === 'comment' ):?>
                    <div class="comments-avatar">
                        <?php print get_avatar( $comment, 110, null, null, [ 'class' => [] ] );?>
                    </div>
                    <?php endif; ?>

                    <div class="comment-text">
                        <div class="avatar-name">
                            <h6 class="name">
                                <?php print get_comment_author_link();?>
                            </h6>
                            <span class="date"><?php comment_time( get_option( 'date_format' ) );?></span>
                        </div>

                        <?php comment_text();?>

                        <div class="comment-reply">
                            <?php comment_reply_link( array_merge($args,
                                array(
                                    'reply_text' => __('Reply', 'eduvalt'),
                                    'depth'      => $depth,
                                    'max_depth'  => $args['max_depth']
                                )
                            )); ?>
                        </div>

                    </div>
                </div>
        <?php
    }
}


// eduvalt_search_filter_form
if ( !function_exists( 'eduvalt_search_filter_form' ) ) {
    function eduvalt_search_filter_form( $form ) {

        $form = sprintf(
            '<div class="sidebar-search-form"><form action="%s" method="get">
      	<input type="text" value="%s" required name="s" placeholder="%s">
      	<button type="submit"> <i class="flaticon-magnifying-glass"></i>  </button>
		</form></div>',
            esc_url( home_url( '/' ) ),
            esc_attr( get_search_query() ),
            esc_html__('Search here', 'eduvalt' )
        );

        return $form;
    }
    add_filter( 'get_search_form', 'eduvalt_search_filter_form' );
}

add_action( 'admin_enqueue_scripts', 'eduvalt_admin_custom_scripts' );

function eduvalt_admin_custom_scripts() {
    wp_enqueue_media();
    wp_enqueue_style( 'customizer-style', get_template_directory_uri() . '/inc/css/customizer-style.css',array());
    wp_register_script( 'eduvalt-admin-custom', get_template_directory_uri() . '/inc/js/admin_custom.js', [ 'jquery' ], '', true );
    wp_enqueue_script( 'eduvalt-admin-custom' );
}


// Archive count on rightside
function eduvalt_archive_count_on_rightside($links) {
    $links = str_replace('</a>&nbsp;(', '</a> <span class="float-right">', $links);
    $links = str_replace(')', '</span>', $links);
    return $links;
}
add_filter( 'get_archives_link', 'eduvalt_archive_count_on_rightside' );


// Categories count on rightside
function eduvalt_categories_count_on_rightside($links) {

	$links = str_replace('<span class="count">(', '</a> <span class="float-right">', $links);
	// For blog
	$links = str_replace('</a> (', '</a> <span class="float-right">', $links);
    $links = str_replace(')', '</span>', $links);
    return $links;
}
add_filter( 'wp_list_categories', 'eduvalt_categories_count_on_rightside',10,2 );


/**
 * //Course header background images
*/

function eduvalt_tutor_header() {
    global $post, $authordata;
    $course_id       = get_the_ID();
    $course_rating   = tutor_utils()->get_course_rating($course_id);
    $is_enrolled     = tutor_utils()->is_enrolled($course_id, get_current_user_id());

    // Prepare the nav items.
    $course_nav_item = apply_filters('tutor_course/single/nav_items', array(
        'info'          => array(
            'title'  => __('Course Info', 'eduvalt'),
            'method' => 'tutor_course_info_custom',
        ),
        'curriculam'          => array(
            'title'  => __('Curriculam', 'eduvalt'),
            'method' => 'tutor_course_topics',
        ),
        'reviews'       => array(
            'title'  => __('Reviews', 'eduvalt'),
            'method' => 'tutor_course_target_reviews_html',
        ),
        'announcements' => array(
            'title'             => __('Announcements', 'eduvalt'),
            'method'            => 'tutor_course_announcements'
        ),
    ), $course_id);
    $is_public       = \TUTOR\Course_List::is_public($course_id);
    $is_mobile       = wp_is_mobile();

    $enrollment_box_position = tutor_utils()->get_option('enrollment_box_position_in_mobile', 'bottom');
    if ('-1' === $enrollment_box_position) {
        $enrollment_box_position = 'bottom';
    }
    $student_must_login_to_view_course = tutor_utils()->get_option('student_must_login_to_view_course');
    $course_duration   = get_tutor_course_duration_context($course_id, true);
    $total_students_in_course = tutor_utils()->count_enrolled_users_by_course($course_id);
    $course_students   = tutor_utils()->count_enrolled_users_by_course($course_id);
    $course_categories     = get_tutor_course_categories($course_id);
    $profile_url       = tutor_utils()->profile_url($authordata->ID, true);
    tutor_utils()->tutor_custom_header();

    if (!is_user_logged_in() && !$is_public && $student_must_login_to_view_course) {
        tutor_load_template('login');
        tutor_utils()->tutor_custom_footer();
        return;
    }

    $topics_ids = get_posts(array(
        'post_parent'    => $course_id,
        'post_type'        => 'topics',
        'fields'         => 'ids'
    ));
    $lessons_count = 0;
    foreach ($topics_ids as $topic) {
        $lessons_ids = get_posts(array(
            'post_type'        => 'lesson',
            'post_parent'    => $topic,
            'fields'         => 'ids'
        ));

        $lessons_count += count($lessons_ids);
    }
    ?>
    <!-- breadcrumb-area -->
    <section class="courses__breadcrumb-area">
        <div class="container">
            <div class="row">
                <div class="col-lg-8">
                    <div class="courses__breadcrumb-content">
                        <?php
                            $course_categories = get_tutor_course_categories();
                            if(is_array($course_categories) && count($course_categories)){
                                foreach ($course_categories as $course_category){
                                    $category_name = $course_category->name;
                                    $category_link = get_term_link($course_category->term_id);
                                    echo "<a class='category' href='$category_link'>$category_name</a>";
                                }
                            }
                        ?>
                        <h3 class="title"><?php the_title(); ?></h3>
                        <p><?php echo wp_trim_words(get_the_excerpt(), 17); ?></p>
                        <ul class="courses__item-meta list-wrap">
                            <li>
                                <?php
                                   $disable_course_author = get_tutor_option('disable_course_author');
                                   if ( !$disable_course_author){
                                        global $post, $authordata;
                                         $profile_url = tutor_utils()->profile_url($authordata->ID, true);
                                        ?>
                                        <div class="author">
                                            <a href="<?php echo esc_url($profile_url); ?>">
                                               <?php echo get_avatar( get_the_author_meta( 'ID' ), 60 ); ?>
                                            </a>
                                            <a href="<?php echo esc_url($profile_url); ?>"><?php echo get_the_author(); ?></a>
                                        </div>
                                <?php } ?>
                            </li>                        
                            <li><i class="flaticon-file"></i> <?php echo esc_html($lessons_count); ?></li>
                            <li><i class="flaticon-timer"></i> <?php echo tutor_utils()->clean_html_content($course_duration); ?></li>
                            <li><i class="flaticon-user-1"></i> <?php echo esc_html($total_students_in_course); ?></li>
                            <?php
                              $disable = get_tutor_option('disable_course_review');
                              if ( ! $disable){
                                ?>
                                <li>
                                    <div class="rating">
                                            <?php
                                            $course_rating = tutor_utils()->get_course_rating();
                                            tutor_utils()->star_rating_generator($course_rating->rating_avg);
                                            ?>
                                            <span class="rating-count">
                                                <?php echo '('.esc_attr( $course_rating->rating_avg ).')'; ?>
                                            </span>
                                    </div>
                                </li>
                            <?php } ?>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </section>
    <!-- breadcrumb-area-end -->
    <?php
}