<?php

/**
 * Abstract class for managing share position settings in Easy Social Share Buttons.
 */
abstract class ESSB_Share_Position_Settings
{
    /**
     * Unique identifier for the share position settings.
     *
     * @var string
     */
    private $id  = '';

    /**
     * Name of the share position.
     *
     * @var string
     */
    private $name = '';

    /**
     * Type of the share position.
     *
     * @var string
     */
    private $type = '';

    /**
     * Supported features for the share position.
     *
     * @var array
     */
    private $supports = [
        'responsive' => true,
        'custom_css' => true,
        'custom_code' => true,
        'styles' => true,
        'networks' => true
    ];

    /**
     * Limited networks configuration.
     *
     * @var array
     */
    private $limited_networks = [
        'enabled' => false,
        'list' => []
    ];

    /**
     * Supported style properties for the share position.
     *
     * @var array
     */
    private $styles_supported = [
        'template' => true,
        'style' => true,
        'align' => true,
        'size_desktop' => true,
        'size_mobile' => true,
        'without_space' => false,
        'animation' => true,
        'counters' => true,
        'width' => true
    ];

    /**
     * Advanced options for the share position.
     *
     * @var array
     */
    private $advanced_options = [];

    /**
     * Values for advanced options.
     *
     * @var array
     */
    private $advanced_values = [];

    /**
     * Style values for the share position.
     *
     * @var array
     */
    private $style_values = [];

    /**
     * Responsive device disable settings.
     *
     * @var array
     */
    private $responsive_disable_devices = [];

    /**
     * Post types where the share position is enabled.
     *
     * @var array
     */
    private $post_types = [];

    /**
     * Whether the share position is enabled.
     *
     * @var bool
     */
    private $enabled = false;

    /**
     * Sets the unique identifier for the share position settings.
     *
     * @param string $id Optional. The ID to set. Default is an empty string.
     */
    public function set_id($id = '')
    {
        $this->id = $id;
    }

    /**
     * Retrieves the unique identifier for the share position settings.
     *
     * @return string The ID associated with the share position settings.
     */
    public function get_id()
    {
        return $this->id;
    }

    /**
     * Sets the name property for the share position settings.
     *
     * @param string $name Optional. The name to set. Default is an empty string.
     */
    public function set_name($name = '')
    {
        $this->name = $name;
    }

    /**
     * Retrieves the name associated with the share position settings.
     *
     * @return string The name of the share position.
     */
    public function get_name()
    {
        return $this->name;
    }

    /**
     * Sets the type for the share position settings.
     *
     * @param string $type The type to set. Defaults to an empty string.
     */
    public function set_type($type = '')
    {
        $this->type = $type;
    }

    /**
     * Retrieves the type of the share position settings.
     *
     * @return string The type of the share position.
     */
    public function get_type()
    {
        return $this->type;
    }

    /**
     * Sets the enabled state for the share position.
     *
     * @param bool $state True to enable, false to disable.
     */
    public function set_enabled(bool $state)
    {
        $this->enabled = $state;
    }

    /**
     * Checks if the share position is enabled.
     *
     * @return bool True if enabled, false otherwise.
     */
    public function is_enabled()
    {
        return $this->enabled;
    }

    /**
     * Checks if a specific feature is supported.
     *
     * @param string $key The feature key.
     * @return bool True if supported, false otherwise.
     */
    public function is_supported($key = '') {
        return isset($this->supports[$key]) ? $this->supports[$key] : false;
    }

    /**
     * Sets the supported features for the share position.
     *
     * @param array $options Associative array of supported features.
     */
    public function set_supports($options = [])
    {
        foreach ($this->supports as $key => $state) {
            if (isset($options[$key])) {
                $this->supports[$key] = $state;
            } else {
                $this->supports[$key] = false;
            }
        }
    }

    /**
     * Retrieves the supported features.
     *
     * @return array Supported features.
     */
    public function get_supports() {
        return $this->supports;
    }

    /**
     * Sets the limited network list.
     *
     * @param array $list List of limited networks.
     */
    public function set_limited_network_list($list = [])
    {
        $this->limited_networks['enabled'] = true;
        $this->limited_networks['list'] = $list;
    }

    /**
     * Checks if a style property is supported.
     *
     * @param string $key The style property key.
     * @return bool True if supported, false otherwise.
     */
    public function is_style_property_supported($key = '') {
        return isset($this->styles_supported[$key]) ? $this->styles_supported[$key] : false;
    }

    /**
     * Checks if at least one of the given style properties is supported.
     *
     * @param array $keys Array of style property keys.
     * @return bool True if at least one is supported, false otherwise.
     */
    public function is_one_of_style_properties_supported($keys = []) {
        $r = false;

        foreach ($keys as $key) {
            if (!$this->is_style_property_supported($key)) {
                $r = true;
                break;
            }
        }

        return $r;
    }

    /**
     * Retrieves the supported style properties.
     *
     * @return array Supported style properties.
     */
    public function get_styles_supported(){
        return $this->styles_supported;
    }

    /**
     * Sets the supported style properties.
     *
     * @param array $options Associative array of style properties.
     */
    public function set_styles_supported($options = [])
    {
        foreach ($this->styles_supported as $key => $state) {
            if (isset($options[$key])) {
                $this->styles_supported[$key] = $state;
            } else {
                $this->styles_supported[$key] = false;
            }
        }
    }

    /**
     * Reads the base options for the share position settings.
     *
     * This method retrieves the default configuration options used for share position settings
     * within the Easy Social Share Buttons plugin.
     */
    public function read_base_options()
    {
        // is position enabled for display
        $this->enabled = essb_options_bool_value($this->id . '_activated');

        // -- reading post types
        $this->post_types = essb_option_value($this->id . '_post_types');
        if (!is_array($this->post_types)) {
            $this->post_types = [];
        }

        // -- reading responsive
        $this->responsive_disable_devices = [];
        $this->responsive_disable_devices['desktop'] = essb_options_bool_value($this->id . '_desktop_deactivate');
        $this->responsive_disable_devices['tablet'] = essb_options_bool_value($this->id . '_tablet_deactivate');
        $this->responsive_disable_devices['mobile'] = essb_options_bool_value($this->id . '_mobile_deactivate');

        // -- reading styles
        $this->style_values = [];

        $this->style_values['template'] = essb_option_value($this->id . '_template');

        $this->style_values['button_align'] = essb_option_value($this->id . '_button_pos');
        $this->style_values['button_width'] = essb_option_value($this->id . '_button_width');
        $this->style_values['button_size'] = essb_option_value($this->id . '_button_size');
        $this->style_values['button_size_mobile'] = essb_option_value($this->id . 'button_size_mobile');
        $this->style_values['button_width_fixed_value'] = essb_option_value($this->id . '_fixed_width_value');
        $this->style_values['button_width_fixed_align'] = essb_option_value($this->id . '_fixed_width_align');
        $this->style_values['button_width_full_container'] = essb_option_value($this->id . '_fullwidth_share_buttons_container');
        $this->style_values['button_width_full_button'] = essb_option_value($this->id . '_fullwidth_share_buttons_correction');
        $this->style_values['button_width_full_button_mobile'] = essb_option_value($this->id . '_fullwidth_share_buttons_correction_mobile');
        $this->style_values['button_width_columns'] = essb_option_value($this->id . '_fullwidth_share_buttons_columns');

        $this->style_values['fullwidth_align'] = essb_option_value($this->id . '_fullwidth_align');
        $this->style_values['fullwidth_share_buttons_columns_align'] = essb_option_value($this->id . '_fullwidth_share_buttons_columns_align');

        // flex width applying
        $this->style_values['flex_width_value'] = essb_option_value($this->id . '_flex_width_value');
        $this->style_values['flex_width_align'] = essb_option_value($this->id . '_flex_width_align');
        $this->style_values['flex_button_value'] = essb_option_value($this->id . '_flex_button_value');

        // @since 3.0.3
        $more_button_icon = essb_option_value($this->id . '_more_button_icon');
        if ($more_button_icon != '') {
            $this->style_values['more_button_icon'] = $more_button_icon;
        }

        // @since 3.3
        $more_button_func = essb_option_value($this->id . '_more_button_func');
        if ($more_button_func != '') {
            $this->style_values['location_more_button_func'] = $more_button_func;
        }

        if (intval($this->style_values['button_width_full_container']) == 0) {
            $this->style_values['button_width_full_container'] = "100";
        }

        // @since 3.5 we add animations
        $this->style_values['button_animation'] = essb_option_value($this->id . '_css_animations');


        $this->style_values['button_style'] = essb_option_value($this->id . '_button_style');
        $this->style_values['nospace'] = essb_option_value($this->id . '_nospace');

        $this->style_values['show_counter'] = essb_option_value($this->id . '_show_counter');
        $this->style_values['counter_pos'] = essb_option_value($this->id . '_counter_pos');
        $this->style_values['total_counter_pos'] = essb_option_value($this->id . '_total_counter_pos');

        $this->style_values['share_button_func'] = essb_option_value($this->id . '_share_button_func');
        $this->style_values['share_button_icon'] = essb_option_value($this->id . '_share_button_icon');
        $this->style_values['share_button_style'] = essb_option_value($this->id . '_share_button_style');
        $this->style_values['share_button_counter'] = essb_option_value($this->id . '_share_button_counter');
    }

    /**
     * Retrieves the style values for the share position.
     *
     * @return array Style values.
     */
    public function get_style_values()
    {
        return $this->style_values;
    }

    /**
     * Retrieves the responsive device disable settings.
     *
     * @return array Responsive device settings.
     */
    public function get_responsive_values()
    {
        return $this->responsive_disable_devices;
    }

    /**
     * Retrieves the post types where the share position is enabled.
     *
     * @return array Post types.
     */
    public function get_post_types()
    {
        return $this->post_types;
    }

    /**
     * Retrieves custom code to be executed before and after the share buttons.
     *
     * @return array Associative array with 'before' and 'after' keys.
     */
    public function get_custom_code()
    {
        $r = ['before' => '', 'after' => ''];

        if (essb_options_bool_value($this->id . '_code')) {
            $r['before'] = essb_options_value($this->id . '_code_before');
            $r['after'] = essb_options_value($this->id . '_code_after');

            if (!empty($r['before'])) {
                $r['before'] = stripslashes($r['before']);
                $r['before'] = do_shortcode($r['before']);
            }

            if (!empty($r['after'])) {
                $r['after'] = stripslashes($r['after']);
                $r['after'] = do_shortcode($r['after']);
            }
        }

        return $r;
    }

    /**
     * Retrieves custom CSS code for the share position.
     *
     * @return string Custom CSS code.
     */
    public function get_custom_css()
    {
        $r = '';

        if (essb_option_bool_value($this->id . '_css')) {
            $r = essb_option_value($this->id . '_css_code');
            $r = stripslashes($r);
        }

        return $r;
    }
}
