<?php
/**
 * DeBlocker
 * Most effective way to detect ad blockers. Ask the visitors to disable their ad blockers.
 * Exclusively on https://1.envato.market/deblocker
 *
 * @encoding        UTF-8
 * @version         3.4.12
 * @copyright       (C) 2018-2025 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Deblocker\Unity;

use WP_Error;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

/**
 * SINGLETON: Class used to implement Activation tab on plugin settings page.
 *
 * @since 1.0.0
 *
 **/
final class TabActivation extends Tab {

    /**
     * Slug of current tab.
     *
     * @since 1.0.0
     * @const TAB_SLUG
     **/
    const TAB_SLUG = 'activation';

	/**
	 * Option group name
	 * @const OPTION_GROUP
	 */
	const OPTION_GROUP = 'DeblockerActivationOptionsGroup';

	/**
	 * Remote validation errors transient name
	 */
	const VALIDATION_ERRORS = 'mdp_deblocker_remote_validation_errors';

	/**
	 * The one true PluginActivation.
	 *
	 * @var TabActivation
	 **/
	private static $instance;

    /**
     * Generate Activation Tab.
     *
     * @access public
     **/
    public function add_settings() {

        /** Not show if plugin don't have Envato ID. */
	    if ( ! DEBLOCKER_ENVATO_ID ) { return; }

        /** Activation Tab. */
        register_setting( self::OPTION_GROUP, 'envato_purchase_code_' . DEBLOCKER_ENVATO_ID );
        add_settings_section( 'mdp_deblocker_settings_page_activation_section', '', null, self::OPTION_GROUP );

    }

    /**
     * Render form with all settings fields.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function do_settings() {

        /** No status tab, nothing to do. */
        if ( ! $this->is_enabled( self::TAB_SLUG ) ) { return; }

        /** Render title. */
        $this->render_title( self::TAB_SLUG );

        /** Render fields. */
        settings_fields( 'DeblockerActivationOptionsGroup' );
        do_settings_sections( 'DeblockerActivationOptionsGroup' );

        /** Render Activation fields. */
        $this->render_activation();

    }

	/**
	 * Display Activation Status.
	 *
	 * @access public
	 **/
	public function display_status() {

        /** Disable this method for custom type plugins. */
        if ( 'custom' === Plugin::get_type() ) { return; }

		$activation_tab = admin_url( 'admin.php?page=mdp_deblocker_settings&tab=activation' );
		?>

        <hr class="mdc-list-divider">
        <h6 class="mdc-list-group__subheader"><?php esc_html_e( 'CodeCanyon License', 'deblocker' ); ?></h6>

		<?php if ( Activator::status() ) : ?>
            <a class="mdc-list-item mdp-activation-status activated" href="<?php echo esc_url( $activation_tab ); ?>">
                <i class='material-icons mdc-list-item__graphic' aria-hidden='true'>check_circle</i>
                <span class="mdc-list-item__text"><?php esc_html_e( 'Activated', 'deblocker' ); ?></span>
            </a>
		<?php else : ?>
            <a class=" mdc-list-item mdp-activation-status not-activated" href="<?php echo esc_url( $activation_tab ); ?>">
                <i class='material-icons mdc-list-item__graphic' aria-hidden='true'>remove_circle</i>
                <span class="mdc-list-item__text"><?php esc_html_e( 'Not Activated', 'deblocker' ); ?></span>
            </a>
		<?php endif;

	}

	/**
	 * Render Purchase Code field.
	 *
	 * @access public
	 **/
	public function render_activation() {

        /** Not show if plugin doesn't have Envato ID. */
		if ( ! DEBLOCKER_ENVATO_ID ) { return; }

        ?>
        <div class="mdp-activation">
            <?php

            $this->render_form();
            $this->render_FAQ();
            $this->render_subscribe();

            ?>
        </div>
        <?php

	}

	/**
	 * Render e-sputnik Subscription Form block.
	 *
	 * @access public
	 **/
	public function render_subscribe() {

        ?>
        <div class="mdp-subscribe-form">

            <h3><?php esc_html_e( 'Subscribe to newsletter', 'deblocker' ); ?></h3>
            <p><?php esc_html_e( 'Sign up for the newsletter to be the first to know about news and discounts.', 'deblocker' ); ?></p>
            <p class="mdp-subscribe-form-message"
               data-success="<?php esc_html_e( 'Hurray! We received your Subscription request. Check your inbox for an email from us.', 'deblocker' ); ?>"
               data-warn="<?php esc_html_e( 'Oh! Sorry, but we cannot send messages to this email.', 'deblocker' ); ?>"
               data-error="<?php esc_html_e( 'Oops! Something went wrong. Please try later.', 'deblocker' ); ?>"
            ></p>

            <?php
            /** Render Name. */
            UI::get_instance()->render_input(
                '',
                esc_html__( 'Your Name', 'deblocker' ),
                '',
                [
                    'name' => 'mdp-subscribe-name',
                    'id' => 'mdp-subscribe-name'
                ]
            );

            /** Render e-Mail. */
            UI::get_instance()->render_input(
                '',
                esc_html__( 'Your E-Mail', 'deblocker' ),
                '',
                [
                    'name'  => 'mdp-subscribe-mail',
                    'id'    => 'mdp-subscribe-mail',
                    'type'  => 'email',
                ]
            );

            /** Render button. */
            UI::get_instance()->render_button(
                esc_html__( 'Subscribe', 'deblocker' ),
                '',
                false,
                [
                    "name"  => "mdp-subscribe",
                    "id"    => "mdp-subscribe"
                ]
            );
            ?>

        </div>
        <?php

	}

	/**
	 * Render CodeCanyon Activation Form
	 */
	public function render_form() {

        /** In this option, we store Envato purchase code. */
        $opt_envato_purchase_code = 'envato_purchase_code_' . DEBLOCKER_ENVATO_ID;

        /** Get activation settings. */
        $purchase_code = get_option( $opt_envato_purchase_code );

        ?>
        <div class="mdp-activation-form">
            <?php
            /** Render input. */
            UI::get_instance()->render_input(
                $purchase_code,
                esc_html__( 'Purchase code', 'deblocker'),
                esc_html__( 'Enter your CodeCanyon purchase code. Allowed only one Purchase Code per website.', 'deblocker' ),
                [
                    'name' => $opt_envato_purchase_code,
                    'id' => 'mdp_envato_purchase_code'
                ]
            );
            ?>
        </div>
        <?php

    }

    /**
     * Render FAQ block.
     *
     * @access public
     **/
    public function render_FAQ() {
        ?>
        <div class="mdp-activation-faq">
            <div class="mdc-accordion" data-mdp-accordion="showfirst: true">

                <h3><?php esc_html_e( 'Activation FAQ\'S', 'deblocker' ); ?></h3>

                <div class="mdc-accordion-title">
                    <i class="material-icons">help</i>
                    <span class="mdc-list-item__text"><?php esc_html_e( 'Where is my Purchase Code?', 'deblocker' ); ?></span>
                </div>
                <div class="mdc-accordion-content">
                    <p><?php esc_html_e( 'The purchase code is a unique combination of characters that confirms that you bought the plugin. You can find your purchase code in ', 'deblocker' ); ?>
                        <a href="https://1.envato.market/cc-downloads" target="_blank"><?php esc_html_e( 'your account', 'deblocker' );?></a><?php esc_html_e( 'on the CodeCanyon. Learn more about ', 'deblocker' ); ?>
                        <a href="https://help.market.envato.com/hc/en-us/articles/202822600-Where-Is-My-Purchase-Code-" target="_blank"><?php esc_html_e( 'How to find your purchase code', 'deblocker' );?></a>.
                    </p>
                </div>

                <div class="mdc-accordion-title">
                    <i class="material-icons">help</i>
                    <span class="mdc-list-item__text"><?php esc_html_e( 'Can I use one Purchase Code on multiple sites?', 'deblocker' ); ?></span>
                </div>
                <div class="mdc-accordion-content">
                    <p>
                        <?php esc_html_e( 'No, this is prohibited by license terms. You can use the purchase code on only one website at a time. Learn more about ', 'deblocker' ); ?>
                        <a href="https://1.envato.market/KYbje" target="_blank"><?php esc_html_e( 'Envato License', 'deblocker' );?></a> <?php esc_html_e( 'terms. ', 'deblocker' ); ?>
                    </p>
                </div>

                <div class="mdc-accordion-title">
                    <i class="material-icons">help</i>
                    <span class="mdc-list-item__text"><?php esc_html_e( 'What are the benefits of plugin activation?', 'deblocker' ); ?></span>
                </div>
                <div class="mdc-accordion-content">
                    <p>
                        <?php esc_html_e( 'Activation of the plugin allows you to use all the functionality of the plugin on your site. In addition, in some cases, activating the plugin allows you to access additional features and capabilities of the plugin. Also, using an authored version of the plugin, you can be sure that you will not violate the license.', 'deblocker' ); ?>
                    </p>
                </div>

                <div class="mdc-accordion-title">
                    <i class="material-icons">help</i>
                    <span class="mdc-list-item__text"><?php esc_html_e( 'What should I do if my Purchase Code does not work?', 'deblocker' ); ?></span>
                </div>
                <div class="mdc-accordion-content">
                    <p>
                        <?php esc_html_e( 'There are several reasons why the purchase code may not work on your site. Learn more why your ', 'deblocker' ); ?>
                        <a href="https://merkulove.zendesk.com/hc/en-us/articles/360006100998-Troubleshooting-of-the-plugin-activation" target="_blank"><?php esc_html_e( 'Purchase Code is Not Working', 'deblocker' );?></a>.
                    </p>
                </div>

            </div>
        </div>
        <?php
    }

	/**
	 * Validate Purchase Code. Migration from old version.
     * @depreacted - use Activator::status() instead.
	 * @return bool|WP_Error
	 */
	public function is_activated(): bool
	{
		return Activator::status();
	}

	/**
	 * Main PluginActivation Instance.
	 * Insures that only one instance of PluginActivation exists in memory at any one time.
	 *
	 * @static
     * @since 1.0.0
     * @access public
     *
     * @return TabActivation
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
