<?php
/**
 * DeBlocker
 * Most effective way to detect ad blockers. Ask the visitors to disable their ad blockers.
 * Exclusively on https://1.envato.market/deblocker
 *
 * @encoding        UTF-8
 * @version         3.4.12
 * @copyright       (C) 2018-2024 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 * @license         Envato License https://1.envato.market/KYbje
 **/

namespace Merkulove\Deblocker;

use Merkulove\Deblocker\Unity\Plugin;
use Merkulove\Deblocker\Unity\Settings;
use Merkulove\Deblocker\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove/Deblocker
 */
final class Notice {

	/**
	 * @var Notice
	 **/
	private static $instance;

	/**
	 * @var bool|mixed
	 */
    private static $is_debug = 'off';

	/**
	 * @return void
	 */
	private function __construct() {

        $options = Settings::get_instance()->options;

        self::$is_debug = $options[ 'debug' ] ?? 'off';

		add_action( 'admin_notices', [ $this, 'admin_notice' ] );
        add_action( 'admin_footer', [ $this, 'plugin_notice' ] );

	}

	/**
	 * Notices for WordPress admin area
	 **/
	public function admin_notice() {

		// Get current screen.
		$screen = get_current_screen();
		if ( $screen === null ) { return; }

        // Check if debug mode is enabled
        if ( self::$is_debug === 'on' ) {

	        ?>
            <div class="settings-error notice notice-warning">
                <p><strong><?php esc_html_e( 'DeBlocker debug mode is enabled.', 'deblocker' ); ?></strong></p>
                <p><?php
			        echo wp_sprintf(
				        esc_html__( 'Manage debug mode on the %s tab', 'deblocker' ),
				        '<a href="'  . admin_url( 'admin.php?page=mdp_deblocker_settings&tab=behaviour#mdp_deblocker_behaviour_settings_debug' ) . '">' . esc_html__( 'Behavior', 'deblocker' ) . '</a>'
			        );
			        ?></p>
            </div>
	        <?php

        }

	}

	/**
     * Notice for plugins settings page
	 * @return void
	 */
    public function plugin_notice() {

	    // Get current screen.
	    $screen = get_current_screen();
	    if ( $screen === null ) { return; }

	    // Check if debug mode is enabled
	    if ( self::$is_debug === 'on' && in_array( $screen->base, Plugin::get_menu_bases(), true ) ) {

		    UI::get_instance()->render_snackbar(
			    esc_html__( 'DeBlocker debug mode is enabled. Manage debug mode on the Behavior tab', 'deblocker' ),
			    'warning',
			    -1,
			    true,
			    [
				    [
					    'caption' => esc_html__( 'Behavior', 'deblocker' ),
					    'link' => admin_url( 'admin.php?page=mdp_deblocker_settings&tab=behaviour#mdp_deblocker_behaviour_settings_debug' )
				    ]
			    ]
		    );

	    }

    }

	/**
	 * Main Instance.
	 *
	 * @return Notice
	 **/
	public static function get_instance(): Notice {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
