<?php
/**
 * Contacter
 * Voice feedback form for your website for saving and transcribing user voice messages to text.
 * Exclusively on https://1.envato.market/contacter
 *
 * @encoding        UTF-8
 * @version         1.7.8
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Contacter;

use WP_Query;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class UninstallHelper {

	/**
	 * The one true UninstallHelper.
	 *
     * @since 1.0.0
     * @access private
	 * @var UninstallHelper
	 **/
	private static $instance;

    /**
     * Implement plugin uninstallation.
     *
     * @param string $uninstall_mode - Uninstall mode: plugin, plugin+settings, plugin+settings+data
     *
     * @since  1.0.0
     * @access public
     *
     * @return void
     **/
    public function uninstall( $uninstall_mode ) {

        /** Remove Plugin. */
        if ( 'plugin+settings+data' === $uninstall_mode ) {

			$this->remove_all_forms();
			$this->remove_all_records();
			$this->remove_contacter_attachments();
			$this->remover_all_notifications();
			$this->remove_notification_table();

        }

    }

	/**
	 * Remove all Contacter Forms
	 * @return void
	 */
	public function remove_all_forms() {

		/** Remove all Contacter Forms. */
		$c_forms = get_posts( [
			'post_type' => ContacterForm::POST_TYPE,
			'numberposts' => - 1,
			'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash']
		] );

		foreach ( $c_forms as $c_form ) {
			wp_delete_post( $c_form->ID, true );
		}

	}

	/**
	 * Remove all Contacter Records
	 * @return void
	 */
	public function remove_all_records() {

		$c_records = get_posts( [
			'post_type' => ContacterRecord::POST_TYPE,
			'numberposts' => - 1,
			'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash']
		] );

		foreach ( $c_records as $c_record ) {
			wp_delete_post( $c_record->ID, true );
		}

	}

	/**
	 * Remove all Notification post types
	 * @return void
	 */
	public function remover_all_notifications() {

		$the_query = new WP_Query(
			[
				'post_type' => 'notification',
				'posts_per_page' => -1
			]
		);

		if ( $the_query->have_posts() ) {
			while ( $the_query->have_posts() ) {
				$the_query->the_post();
				wp_delete_post( get_the_ID() );
			}
		}

	}

	/**
	 * Remove Contacter audio attachment from the Media library
	 * @return void
	 */
	public function remove_contacter_attachments() {

		$the_query = new WP_Query(
			[
				'post_type' => 'attachment',
				'post_mime_type' => 'audio',
				'posts_per_page' => -1,
				'post_status' => 'inherit',
				'meta_query' => [
					[
						'key' => 'contacter-attachment',
						'value' => 'wav'
					]
				]
			]
		);

		if ( $the_query->have_posts() ) {
			while ( $the_query->have_posts() ) {
				$the_query->the_post();
				wp_delete_post( get_the_ID() );
			}
		}

	}

	/**
	 * Remove notification associated tables
	 * @return void
	 */
	public function remove_notification_table() {

		global $wpdb;
		$wpdb->query( "DROP TABLE IF EXISTS `wp_notification_logs`" );

	}

	/**
	 * Main UninstallHelper Instance.
	 * Insures that only one instance of UninstallHelper exists in memory at any one time.
	 *
	 * @static
     * @since 1.0.0
     * @access public
     *
	 * @return UninstallHelper
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
