<?php
/**
 * Contacter
 * Voice feedback form for your website for saving and transcribing user voice messages to text.
 * Exclusively on https://1.envato.market/contacter
 *
 * @encoding        UTF-8
 * @version         1.7.8
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Contacter;

use Merkulove\Contacter\Unity\Plugin;
use Merkulove\Contacter\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class Assets {

	/**
	 * The one true Assets.
	 *
     * @since 1.0.0
     * @access private
	 * @var Assets
	 **/
	private static $instance;

	/**
	 * Register all the hooks related to the public-facing functionality.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return void
	 **/
	public function public_assets() {

		add_action( 'wp_enqueue_scripts', [$this, 'styles'] );
		add_action( 'wp_enqueue_scripts', [$this, 'scripts'] );

	}

	/**
	 * Register all the hooks related to the admin area functionality.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return void
	 **/
	public function admin_assets() {

		add_action( 'admin_enqueue_scripts', [ $this, 'admin_styles' ], 100 );
		add_action( 'admin_enqueue_scripts', [ $this, 'admin_scripts' ], 100 );

	}

	/**
	 * Add CSS for admin area.
	 *
	 * @return void
	 * @since 1.0.0
	 **/
	public function admin_styles() {

		/** Get current screen to add styles on specific pages. */
		$screen = get_current_screen();
		if ( null === $screen ) { return; }

		/** Contacter Settings Page. */
		if ( Plugin::get_menu_bases() == $screen->base ) {
			wp_enqueue_style( 'merkulov-ui', Plugin::get_url() . 'css/merkulov-ui' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );
			wp_enqueue_style( 'mdp-contacter-admin', Plugin::get_url() . 'css/admin' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

		/** Contacter popup on update. */
		} elseif ( 'plugin-install' === $screen->base ) {

			/** Styles only for our plugin. */
			if ( isset( $_GET['plugin'] ) && $_GET['plugin'] === 'contacter' ) {
				wp_enqueue_style( 'mdp-contacter-plugin-install', Plugin::get_url() . 'css/plugin-install' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );
			}

		/** Contacter Form edit screen. */
		} elseif ( 'post' === $screen->base && 'contacter_form' === $screen->post_type ) {

			/** Add class .mdc-disable to body. So we can use UI without overrides WP CSS, only for this page.  */
			add_action( 'admin_body_class', [$this, 'add_admin_class'] );

			wp_enqueue_style( 'merkulov-ui', Plugin::get_url() . 'css/merkulov-ui' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );
			wp_enqueue_style( 'mdp-admin-contacter-form-edit', Plugin::get_url() . 'css/admin-contacter-form-edit' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

			wp_dequeue_style( 'ultimate-vc-backend-style' ); // Remove styles ultimate-vc

		/** Contacter Record edit screen. */
		} elseif ( 'post' === $screen->base && 'contacter_record' === $screen->post_type ) {

			wp_enqueue_style( 'mdp-admin-contacter-record-edit', Plugin::get_url() . 'css/admin-contacter-record-edit' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

		/** Contacter Forms list. */
		} elseif ( 'edit' === $screen->base && 'contacter_form' === $screen->post_type ) {

			wp_enqueue_style( 'mdp-admin-contacter-forms', Plugin::get_url() . 'css/admin-contacter-forms' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

		/** Contacter Records list */
		} elseif ( 'edit' === $screen->base && 'contacter_record' === $screen->post_type ) {

			wp_enqueue_style( 'mdp-admin-contacter-records', Plugin::get_url() . 'css/admin-contacter-records' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );
		}

	}

	/**
	 * Add JS for admin area.
	 *
	 * @return void
	 * @since 1.0.0
	 **/
	public function admin_scripts() {

		/** Get current screen to add scripts on specific pages. */
		$screen = get_current_screen();
		if ( null === $screen ) { return; }

		/** Contacter Settings Page. */
		if ( $screen->base == Plugin::get_menu_bases() ) {
			wp_enqueue_script( 'merkulov-ui', Plugin::get_url() . 'js/merkulov-ui' . Plugin::get_suffix() . '.js', [], Plugin::get_suffix(), true );
			wp_enqueue_media(); // WordPress Image library for API Key File.
			wp_enqueue_script( 'mdp-contacter-admin', Plugin::get_url() . 'js/admin' . Plugin::get_suffix() . '.js', ['jquery'], Plugin::get_suffix(), true );

			wp_localize_script('mdp-contacter-admin', 'mdpContacter', [
				'ajaxURL' => admin_url( 'admin-ajax.php' ),
				'nonce' => wp_create_nonce( 'reset_settings' ),
				'contacter_nonce' => wp_create_nonce( 'contacter-nonce' ), // Nonce for security.
			] );

			/** Contacter Form edit screen. */
		} elseif ( 'post' === $screen->base && 'contacter_form' === $screen->post_type ) {

			/** Add class .mdc-disable to body. So we can use UI without overrides WP CSS, only for this page. */
			add_action( 'admin_body_class', [$this, 'add_admin_class'] );

			wp_enqueue_script( 'merkulov-ui', Plugin::get_url() . 'js/merkulov-ui' . Plugin::get_suffix() . '.js', [], Plugin::get_suffix(), true );
			wp_enqueue_script( 'form-builder', Plugin::get_url() . 'js/form-builder.min.js', ['jquery', 'jquery-ui-sortable'], Plugin::get_suffix(), true );
			wp_enqueue_script( 'form-render', Plugin::get_url() . 'js/form-render.min.js', ['jquery', 'jquery-ui-sortable'], Plugin::get_suffix(), true );
			wp_enqueue_script( 'mdp-contacter-clipboard', Plugin::get_url() . 'js/clipboard' . Plugin::get_suffix() . '.js', [], Plugin::get_suffix(), true );
			wp_enqueue_script( 'mdp-admin-contacter-form-edit', Plugin::get_url() . 'js/admin-contacter-form-edit' . Plugin::get_suffix() . '.js', ['jquery', 'form-builder', 'mdp-contacter-clipboard'], Plugin::get_suffix(), true );
			wp_localize_script( 'mdp-admin-contacter-form-edit', 'mdpContacter', [
				'locale' => $this->get_fb_locale(),
				'location' => Plugin::get_url() . 'js/form-builder-languages/'
			] );

			/** Contacter Record edit screen. */
		} elseif ( 'post' === $screen->base && 'contacter_record' === $screen->post_type ) {

			wp_enqueue_script( 'mdp-admin-contacter-record-edit', Plugin::get_url() . 'js/admin-contacter-record-edit' . Plugin::get_suffix() . '.js', [], Plugin::get_suffix(), true );

			/** Set Nonce. */
			$ajax_nonce = wp_create_nonce( 'admin-contacter-nonce' );

			/** Current record ID. */
			$id = get_the_ID();

			wp_localize_script( 'mdp-admin-contacter-record-edit', 'mdpContacter', [
				'record_id'  => $id,
				'ajax_nonce' => $ajax_nonce,
			] );

			/** Contacter Forms list. */
		} elseif ( 'edit' === $screen->base && 'contacter_form' === $screen->post_type ) {

			wp_enqueue_script( 'mdp-contacter-clipboard', Plugin::get_url() . 'js/clipboard' . Plugin::get_suffix() . '.js', [], Plugin::get_suffix(), true );
			wp_enqueue_script( 'mdp-admin-contacter-forms', Plugin::get_url() . 'js/admin-contacter-forms' . Plugin::get_suffix() . '.js', ['jquery', 'mdp-contacter-clipboard'], Plugin::get_suffix(), true );

		}

	}

	/**
	 * Return locale for Form Builder.
	 *
	 * @since 1.0.0
	 * @return string
	 **/
	private function get_fb_locale() {

		/** Get current user Locale. */
		$locale = get_user_locale();

		/** Convert "en_US" to "en-US". */
		$locale = str_replace( '_', '-', $locale );

		/** Do we have translations file for this locale? */
		if ( file_exists( Plugin::get_path() . 'js/form-builder-languages/' . $locale . '.lang' ) ) {
			return $locale;
		}

		return 'en-US';

	}

	/**
	 * Add CSS for the public-facing side of the site.
	 *
	 * @return void
	 * @since 1.0.0
	 **/
	public function styles() {

		/** Frontend CSS for shortcodes. */
		wp_register_style( 'mdp-contacter', Plugin::get_url() . 'css/contacter' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

	}

	/**
	 * Return inline CSS for contacter.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string
	 **/
	public function get_contacter_inline_css() {

		/** Get Plugin Settings. */
		$options = Settings::get_instance()->options;

		return $options[ 'custom_css' ] . "
            :root{
	            --mdp-contacter-margin: {$options['fbutton_margin']}px;
	            --mdp-contacter-padding: {$options['fbutton_padding']}px;
	            --mdp-contacter-radius: {$options['fbutton_border_radius']}px;
	            --mdp-contacter-color: {$options['fbutton_color']};
	            --mdp-contacter-color-hover: {$options['fbutton_color_hover']};
	            --mdp-contacter-bg-color: {$options['fbutton_bgcolor']};
	            --mdp-contacter-bg-color-hover: {$options['fbutton_bgcolor_hover']};
	            --mdp-contacter-size: {$options['fbutton_size']}px;
	            --mdp-contacter-delay: {$options['fbutton_entrance_timeout']}s;
	            --mdp-contacter-overlay-bg: {$options['modal_overlay_color']};
	            --mdp-contacter-modal-bg: {$options['modal_bg_color']};
	            --mdp-contacter-modal-radius: {$options['modal_radius']}px;
	            --mdp-contacter-accent-color: {$options['accent_color']};
	        }
        ";

	}

	/**
	 * Add JavaScript for the public-facing side of the site.
	 *
	 * @return void
	 * @since 1.0.0
	 **/
	public function scripts() {

		$options = Settings::get_instance()->options;

		/** Frontend JS for shortcodes. */
		wp_register_script( 'mdp-contacter-recorder', Plugin::get_url() . 'js/recorder' . Plugin::get_suffix() . '.js', [], Plugin::get_version(), true );
		wp_register_script( 'green-audio-player', Plugin::get_url() . 'js/green-audio-player' . Plugin::get_suffix() . '.js', [], Plugin::get_version(), true );
		wp_register_script( 'mdp-contacter', Plugin::get_url() . 'js/contacter' . Plugin::get_suffix() . '.js', ['mdp-contacter-recorder', 'green-audio-player'], Plugin::get_version(), true );

		/** Pass variables to frontend. */
		wp_localize_script( 'mdp-contacter', 'mdpContacterWP', [
			'nonce' => wp_create_nonce( 'contacter-nonce' ),
			'ajaxurl' => admin_url( 'admin-ajax.php' ),
			'accentColor' => $options['accent_color'],
			'sampleRate' => $options['sample_rate'],
		] );

	}

	/**
	 * Add class to body in admin area.
	 *
	 * @param string $classes - Space-separated list of CSS classes.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function add_admin_class( $classes ) {

		return $classes . ' mdc-disable ';

	}

	/**
	 * Main Assets Instance.
	 * Insures that only one instance of Assets exists in memory at any one time.
	 *
	 * @static
     * @since 1.0.0
     * @access public
     *
	 * @return Assets
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
