<?php
namespace BooklyPackages\Backend\Modules\Packages;

use Bookly\Lib as BooklyLib;
use BooklyPackages\Lib\Entities;
use BooklyPackages\Lib\Notifications;
use Bookly\Backend\Components\Dialogs\Queue\NotificationList;

class Ajax extends BooklyLib\Base\Ajax
{
    /**
     * @inheritDoc
     */
    protected static function permissions()
    {
        return array(
            '_default' => 'supervisor',
            'getPackages' => 'customer',
        );
    }

    /**
     * Get list of packages.
     */
    public static function getPackages()
    {
        $columns = BooklyLib\Utils\Tables::filterColumns( self::parameter( 'columns' ), BooklyLib\Utils\Tables::PACKAGES );
        $order = self::parameter( 'order', array() );
        $filter = self::parameter( 'filter' );
        $is_backend = ! empty( $filter );

        $query = Entities\Package::query( 'p' )
            ->select( '
                p.id,
                p.location_id,
                p.staff_id,
                p.service_id,
                p.created_at,
                p.payment_id,
                ps.title        AS package_title,
                ps.package_size AS package_size,
                s.title         AS service_title,
                c.full_name     AS customer_full_name,
                c.phone         AS customer_phone,
                c.email         AS customer_email,
                st.full_name    AS staff_name,
                pm.paid         AS payment,
                pm.total        AS payment_total,
                pm.type         AS payment_type,
                pm.status       AS payment_status
            ' )
            ->leftJoin( 'Service', 'ps', 'ps.id = p.service_id', '\Bookly\Lib\Entities' )
            ->leftJoin( 'SubService', 'ss', 'ss.service_id = ps.id', '\Bookly\Lib\Entities' )
            ->leftJoin( 'Service', 's', 's.id = ss.sub_service_id', '\Bookly\Lib\Entities' )
            ->leftJoin( 'Customer', 'c', 'c.id = p.customer_id', '\Bookly\Lib\Entities' )
            ->leftJoin( 'Payment', 'pm', 'pm.id = p.payment_id', '\Bookly\Lib\Entities' )
            ->leftJoin( 'Staff', 'st', 'st.id = p.staff_id', '\Bookly\Lib\Entities' );

        if ( $is_backend ) {
            $total = $query->count();
            if ( isset( $filter['id'] ) && $filter['id'] != '' ) {
                $query->where( 'p.id', $filter['id'] );
            }

            if ( isset( $filter['date'] ) ) {
                list ( $start, $end ) = explode( ' - ', $filter['date'], 2 );
                $end = date( 'Y-m-d', strtotime( $end ) + DAY_IN_SECONDS );
                $query->whereBetween( 'p.created_at', $start, $end );
            }

            if ( isset( $filter['staff'] ) && $filter['staff'] != '' ) {
                if ( $filter['staff'] == 0 ) {
                    $query->where( 'p.staff_id', null );
                } else {
                    $query->where( 'p.staff_id', $filter['staff'] );
                }
            }
            if ( isset( $filter['package'] ) && $filter['package'] != '' ) {
                $query->where( 'p.service_id', $filter['package'] );
            }

            if ( isset( $filter['service'] ) && $filter['service'] != '' ) {
                $query->where( 's.id', $filter['service'] );
            }
            if ( isset( $filter['customer'] ) && $filter['customer'] != '' ) {
                $query->where( 'p.customer_id', $filter['customer'] );
            }
        } else {
            $customers = BooklyLib\Entities\Customer::query()->where( 'wp_user_id', get_current_user_id() )->fetchCol( 'id' );
            $query->whereIn( 'p.customer_id', $customers );
            $show_with_completed_payments = get_option( 'bookly_packages_show_with_payments' );
            if ( get_option( 'bookly_packages_show_without_payments' ) ) {
                if ( $show_with_completed_payments === 'completed' ) {
                    $query->whereRaw( 'IF(pm.id IS NULL,true,pm.status = \'completed\')', array() );
                }
            } elseif ( $show_with_completed_payments === 'completed' ) {
                $query->whereRaw( 'IF(pm.id IS NULL,false,pm.status = \'completed\')', array() );
            } else {
                $query->whereNot( 'pm.id', null );
            }

            $total = $query->count();
        }

        foreach ( $order as $sort_by ) {
            $query->sortBy( str_replace( '.', '_', $columns[ $sort_by['column'] ]['data'] ) )
                ->order( $sort_by['dir'] == 'desc' ? BooklyLib\Query::ORDER_DESCENDING : BooklyLib\Query::ORDER_ASCENDING );
        }

        $filtered = $query->count();

        $query->limit( self::parameter( 'length' ) )->offset( self::parameter( 'start' ) );

        $data = array();

        foreach ( $query->fetchArray() as $row ) {
            // Check inactive packages
            $package = new Entities\Package();
            $package->load( $row['id'] );
            $expired_date = $package->getExpiredDate();
            if ( $expired_date && $expired_date < date_create( current_time( 'mysql' ) ) ) {
                $row['package_title'] .= sprintf( ' (%s)', __( 'Expired', 'bookly' ) );
            }

            $payment_title = '';
            if ( $row['payment'] !== null ) {
                $payment_title = BooklyLib\Utils\Price::format( $row['payment'] );
                if ( $row['payment'] != $row['payment_total'] ) {
                    $payment_title = sprintf( __( '%s of %s', 'bookly' ), $payment_title, BooklyLib\Utils\Price::format( $row['payment_total'] ) );
                }

                $payment_title .= sprintf(
                    ' %s <span%s>%s</span>',
                    BooklyLib\Entities\Payment::typeToString( $row['payment_type'] ),
                    $row['payment_status'] == BooklyLib\Entities\Payment::STATUS_PENDING ? ' class="text-danger"' : '',
                    BooklyLib\Entities\Payment::statusToString( $row['payment_status'] )
                );
            }

            $data[] = array(
                'id' => $row['id'],
                'created_at' => BooklyLib\Utils\DateTime::formatDateTime( $row['created_at'] ),
                'expires' => $expired_date ? BooklyLib\Utils\DateTime::formatDateTime( $expired_date->format( 'Y-m-d H:i:s' ) ) : '',
                'staff' => array(
                    'name' => $row['staff_name'] ?: __( 'Unassigned', 'bookly' ),
                ),
                'customer' => array(
                    'full_name' => $row['customer_full_name'],
                    'phone' => $row['customer_phone'],
                    'email' => $row['customer_email'],
                ),
                'service' => array(
                    'title' => $row['service_title'],
                ),
                'package' => array(
                    'title' => $row['package_title'],
                    'size' => $row['package_size'],
                ),
                'payment' => $payment_title,
                'payment_id' => $row['payment_id'],
            );
        }

        unset( $filter['date'] );

        BooklyLib\Utils\Tables::updateSettings( BooklyLib\Utils\Tables::PACKAGES, $columns, $order, $filter );

        wp_send_json( array(
            'draw' => ( int ) self::parameter( 'draw' ),
            'recordsTotal' => $total,
            'recordsFiltered' => $filtered,
            'data' => $data,
        ) );
    }

    /**
     * Delete customer appointments.
     */
    public static function deletePackages()
    {
        $notification = self::parameter( 'notify' );

        $queue = new NotificationList();
        /** @var Entities\Package $package */
        foreach ( Entities\Package::query()->whereIn( 'id', (array) self::parameter( 'data' ) )->find() as $package ) {
            $package->delete();
            if ( $notification ) {
                Notifications\Package\Sender::send( $package, 'deleted', self::parameter( 'reason' ), $queue );
            }
        }
        $response = array();
        $list = $queue->getList();
        if ( $list ) {
            $db_queue = new BooklyLib\Entities\NotificationQueue();
            $db_queue
                ->setData( json_encode( array( 'all' => $list ) ) )
                ->save();

            $response['queue'] = array( 'token' => $db_queue->getToken(), 'all' => $queue->getInfo() );
        }

        wp_send_json_success( $response );
    }

}