<?php
namespace BooklyFiles\Lib\ProxyProviders;

use BooklyFiles\Lib;
use Bookly\Lib as BooklyLib;

abstract class Local extends BooklyLib\Proxy\Files
{
    /******************************************************************************************************************
     * FRONTEND                                                                                                       *
     ******************************************************************************************************************/

    /**
     * @inheritDoc
     */
    public static function getCFIds()
    {
        $custom_fields = BooklyLib\Proxy\CustomFields::getOnly( array( 'file' ) ) ?: array();

        $ids = array();
        foreach ( $custom_fields as $custom_field ) {
            $ids[] = $custom_field->id;
        }

        return $ids;
    }

    /**
     * @inheritDoc
     */
    public static function getCIIds()
    {
        $fields = BooklyLib\Proxy\CustomerInformation::getOnlyFields( array( 'file' ) ) ?: array();

        $ids = array();
        foreach ( $fields as $field ) {
            $ids[] = $field->id;
        }

        return $ids;
    }

    /**
     * @inheritDoc
     */
    public static function attachCFFiles( array $fields, BooklyLib\Entities\CustomerAppointment $ca )
    {
        $fields_with_file = self::getCFIds();

        $customer_appointment_id = $ca->getId();
        foreach ( $fields as $field ) {
            if ( in_array( $field['id'], $fields_with_file ) ) {
                $file = new Lib\Entities\Files();
                if ( $field['value'] && $file->loadBy( array( 'slug' => $field['value'] ) ) ) {
                    $ca_file = new Lib\Entities\CustomerAppointmentFiles();
                    $ca_file->loadBy(
                        array(
                            'customer_appointment_id' => $customer_appointment_id,
                            'file_id' => $file->getId(),
                        ) );
                    if ( ! $ca_file->isLoaded() ) {
                        $ca_file
                            ->setCustomerAppointmentId( $customer_appointment_id )
                            ->setFileId( $file->getId() )
                            ->save();
                        self::renameFile( $file, $ca );
                    }
                }
            }
        }
    }

    /**
     * @inerhitDoc
     */
    public static function attachCIFiles( array $fields, BooklyLib\Entities\Customer $customer )
    {
        $fields_with_file = self::getCIIds();
        foreach ( $fields as $field ) {
            if ( in_array( $field['id'], $fields_with_file ) ) {
                $file = new Lib\Entities\Files();
                if ( $field['value'] && $file->loadBy( array( 'slug' => $field['value'] ) ) ) {
                    self::renameFile( $file, $customer );
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public static function getFileNamesForCustomerInformationFields( array $custom_fields )
    {
        $names = array();
        $fields_with_file = self::getCIIds();
        foreach ( $custom_fields as $field ) {
            if ( in_array( $field['id'], $fields_with_file ) && isset( $field['value'] ) ) {
                $file = Lib\Entities\Files::query()
                    ->select( 'name' )
                    ->where( 'slug', $field['value'] )
                    ->fetchRow();
                $names[ $field['id'] ] = $file ? $file['name'] : $field['value'];
            }
        }

        return $names;
    }

    public static function getFileNamesForCustomFields( array $custom_fields )
    {
        $names = array();
        $fields_with_file = self::getCFIds();
        foreach ( $custom_fields as $field ) {
            if ( in_array( $field['id'], $fields_with_file ) && isset( $field['value'] ) ) {
                $file = Lib\Entities\Files::query()
                    ->select( 'name' )
                    ->where( 'slug', $field['value'] )
                    ->fetchRow();
                $names[ $field['id'] ] = $file ? $file['name'] : $field['value'];
            }
        }

        return $names;
    }

    /******************************************************************************************************************
     * BACKEND                                                                                                        *
     ******************************************************************************************************************/

    /**
     * @inheritDoc
     */
    public static function setFileNamesForCustomFields( $data, array $custom_fields )
    {
        foreach ( $data as &$customer_custom_field ) {
            if ( array_key_exists( $customer_custom_field['id'], $custom_fields ) ) {
                $field = $custom_fields[ $customer_custom_field['id'] ];
                if ( $field->type == 'file' ) {
                    $file = Lib\Entities\Files::query( 'f' )
                        ->select( 'f.name' )
                        ->where('slug', $customer_custom_field['value'] )
                        ->fetchRow();
                    if ( $file ) {
                        $customer_custom_field['value'] = $file['name'];
                    }
                }
            }
        }

        return $data;
    }

    /**
     * Rename file to human friendly name
     *
     * @param Lib\Entities\Files $file
     * @param BooklyLib\Entities\CustomerAppointment|BooklyLib\Entities\Customer $cau
     * @return Lib\Entities\Files
     */
    protected static function renameFile( Lib\Entities\Files $file, $cau )
    {
        $mask = '{a_id}-{ca_id}-{f_id}{random}{extension}';
        $fs = BooklyLib\Utils\Common::getFilesystem();
        $path = $file->getPath();
        $parts = explode( '.', basename( $path ) );
        $dir = dirname( $path );
        $extension = '';
        if ( count( $parts ) > 1 ) {
            $extension .= '.' . end( $parts );
        }
        $random = '';
        if ( $cau instanceof BooklyLib\Entities\CustomerAppointment ) {
            $a_id = sprintf( '%04d', $cau->getAppointmentId() );
            $f_id = $file->getCustomFieldId();
        } else {
            $a_id = 'cust';
            $f_id = $file->getCustomerInformationId();
        }

        do {
            $target = $dir . DIRECTORY_SEPARATOR . strtr( $mask, array(
                '{a_id}' => $a_id,
                '{ca_id}' => sprintf( '%04d', $cau->getId() ),
                '{f_id}' => sprintf( '%05d', $f_id ),
                '{random}' => $random,
                '{extension}' => $extension,
            ) );

            $random = '-' . wp_generate_password( 4, false );
        } while ( $fs->exists( $target ) );

        $fs->move( $file->getPath(), $target );
        $file->setPath( $target )->save();

        return $file;
    }

    /**
     * @return array
     */
    public static function getAllowedExtensions()
    {
        $extensions = strtolower( get_option( 'bookly_files_extensions', '' ) );
        $extensions = preg_replace( '/[^a-z\d,]/', '', $extensions );

        return $extensions === '' ? array() : explode( ',', $extensions );
    }
}