<?php

/**
 * Plugin Name: AX Social Stream
 * Plugin URI: https://axentmedia.com/ax-social-stream/
 * Description: Combine all your social media feeds (Facebook, Twitter, Instagram, Flickr, YouTube, RSS, ...) into one single social stream and display on your website.
 * Tags: social stream, wordpress social plugin, social feed, social tabs, social wall, social timeline, social media, social networks, feed reader, facebook, twitter, tumblr, delicious, pinterest, flickr, instagram, youtube, vimeo, deviantart, rss, soundcloud, vk, vine
 * Version: 3.12.3
 * Author: Axent Media LLC <support@axentmedia.com>
 * Author URI: https://axentmedia.com/
 * License: https://codecanyon.net/licenses/standard
 * 
 * Copyright © 2014-2023 Axent Media LLC, All Rights Reserved.
 * Any type of usages require a commercial license.
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) { exit; }

// Define constants
define( 'AXS_VERSION', '3.12.3' );
define( 'AXS_DOCS', 'https://axentmedia.com/ax-social-stream-docs/' );
define( 'AXS_FILE', __FILE__ );
define( 'AXS_DIRNAME', dirname( AXS_FILE ) );
define( 'AXS_LOGFILE', AXS_DIRNAME.'/sb.log' );
define( 'AXS_UPSERVER', 'https://wpupdate.axentmedia.com/' );

// starting session
add_action('wp', 'axs_start_session', 1);
function axs_start_session() {
    if ( ! session_id() ) {
        session_start();
    }
}

// Localization
load_plugin_textdomain(
    'social-board',
    false,  // deprecated
    basename( AXS_DIRNAME ) . '/language/'  // Relative path to WP_PLUGIN_DIR where the .mo file resides.
);

// Define some global vars
$GLOBALS['sb_scripts'] = array();
$GLOBALS['sb_no_jquery'] = false;
$GLOBALS['sb_apipage_url'] = urlencode( admin_url( 'edit.php?post_type=sb_posts&page=sb_settings&tab=api' ) );

// Include the library file
if ( ! class_exists( 'SB_AdminPageFramework' ) ) {
    include( AXS_DIRNAME . '/library/admin-page-framework/admin-page-framework.php' );
}

// Load dependencies
require AXS_DIRNAME . "/library/autoload.php";
include( AXS_DIRNAME . '/admin/SB_Bootstrap.php' );
include_once AXS_DIRNAME . "/library/class_wordpress.php";
include_once AXS_DIRNAME . "/library/class_instagram.php";
include_once AXS_DIRNAME . '/library/class_tiktok.php';

// Load Twitter library
use Abraham\TwitterOAuth\TwitterOAuth;

// Load cache system
include( AXS_DIRNAME . '/library/SimpleCache.php' );
include( AXS_DIRNAME . '/library/HtmlEmbedCache.php' );

require 'library/Layout.php';

// AX Social Stream main class
class SocialBoard {
    public $sboption = array();
	public $attr, $feed_keys, $protocol, $final, $finalslide;
    public $cache, $forceCrawl, $target, $setoption, $fetchcomments;
    public $echo = true;
    public $args = null;
    private $enabled_layouts;
	private $sbinstagram, $sbwordpress, $sbtiktok;
    
    function __construct() {
        $this->enabled_layouts = axs_enabled_layouts();
    }

    public function start() {
		$this->setoption = SB_AdminPageFramework::getOption( 'AXS_Settings_Page' ) ? SB_AdminPageFramework::getOption( 'AXS_Settings_Page' ) : SB_AdminPageFramework::getOption( 'SB_Settings_Page' );

        // Add assets
        add_action( 'wp_enqueue_scripts', array(&$this, 'add_scripts_header') );
        $js_assets_position = $this->js_assets_position();
        add_action( 'wp_enqueue_scripts', array(&$this, 'add_scripts'), $js_assets_position['order'], 1 );
        
        // Add shortcode
        add_shortcode( 'social_board', array('SocialBoard', 'social_board_shortcode') );

        // Add ajax handlers
        add_action("wp_ajax_sb_fetchcomments", array(&$this, "axs_fetchcomments") );
        add_action("wp_ajax_nopriv_sb_fetchcomments", array(&$this, "axs_fetchcomments") );
    }
    
    // Initialize by property
    function run() {
        $this->init( $this->attr, $this->echo, $this->args );
    }

    // Initialize by function
    function init( $attr, $echo = true, $args = null, $ajax_feed = array(), $loadmore = array() ) {
        $id = (isset($attr['network']) && !empty($attr['network'])) ? substr( sha1( serialize($attr) ), 0, 5 ) : ($attr['id'] ?? 0);
        if (isset($attr['sb_type']) && !empty($attr['sb_type']))
            $attr['type'] = $attr['sb_type'];
        $type = (isset($attr['type']) && !empty($attr['type'])) ? $attr['type'] : 'wall';
        if ( empty($id) ) {
			return '<strong>AX Social Stream: </strong>No social stream is selected!';
			die();
		}

        // get options
        $this->sboption = (isset($attr['network']) && !empty($attr['network'])) ? $attr['network'] : axs_options( $id );
        
        // get default setting & post options
        $defoption = array(
            'setting' => array(
                'theme' => 'sb-modern-light',
                'results' => '30',
                'words' => '40',
                'slicepoint' => '300',
                'commentwords' => '20',
                'titles' => '15',
                'dateformat' => 'friendly',
                'userinfo' => 'top',
                'readmore' => '1',
                'order' => 'date',
                'filters' => '1',
                'filter_ads' => true,
                'display_all' => '',
                'loadmore' => '1',
                'lightboxtype' => 'media',
                'layout_image' => 'imgexpand',
                'links' => '1',
                'nofollow' => '1',
                'https' => true,
                'lazyload' => true,
                'cache' => '720',
                'crawl' => '10',
                'timeout' => '15',
                'debuglog' => '0'
            ),
            'wallsetting' => array(
                'transition' => '400',
                'stagger' => '',
                'filter_search' => true,
                'originLeft' => 'true',
                'wall_width' => '',
                'wall_height' => '',
                'fixWidth' => 'block',
                'breakpoints' => array('5', '4', '4', '3', '2', '2', '1'),
				'itemwidth' => '250',
				'showmore' => true,
                'gutterX' => '10',
                'gutterY' => '10'
            ),
            'gridsetting' => array(
                'height' => '',
                'fixWidth' => 'block',
                'breakpoints' => array('5', '4', '4', '3', '2', '2', '1'),
                'gutterX' => '0',
                'gutterY' => '0',
                'scroll' => false
            ),
            'feedsetting' => array(
                'rotate_speed' => '100',
                'duration' => '4000',
                'direction' => 'up',
                'controls' => '1',
                'autostart' => '1',
                'pauseonhover' => '1',
                'width' => '280'
            ),
            'carouselsetting' => array(
                'cs_speed' => '400',
                'cs_rows' => '2',
                'cs_item' => array('4', '3', '2', '2', '1'),
                'cs_height' => array(
                    'thumb' => '150',
                    'text' => '75',
                    'meta' => '50'
                ),
                'cs_controls' => 'true',
                'cs_rtl' => 'false',
                'cs_auto' => 'false',
                'cs_autospeed' => '2000',
                'cs_pause' => 'true',
                'cs_loop' => 'true',
                'cs_pager' => 'true'
            ),
            'timelinesetting' => array(
                'onecolumn' => 'false'
            )
        );
        
        // merge options
        $setoption = SB_AdminPageFramework::getOption( 'AXS_Settings_Page' ) ? SB_AdminPageFramework::getOption( 'AXS_Settings_Page' ) : SB_AdminPageFramework::getOption( 'SB_Settings_Page' );
        $setoption = array_merge($defoption, $setoption);
        foreach ($defoption as $defoptKey => $defoptVal) {
            if ( isset($setoption[$defoptKey]) )
                $setoption[$defoptKey] = array_merge($defoptVal, $setoption[$defoptKey]);
        }
        
    	// the array of feeds to get
        $filters_order = array(
            0 => array('facebook' => 1),
			array('twitter' => 1),
			array('tumblr' => 1),
			array('delicious' => 1),
			array('pinterest' => 1),
			array('flickr' => 1),
			array('instagram' => 1),
			array('youtube' => 1),
			array('vimeo' => 1),
			array('deviantart' => 1),
			array('rss' => 1),
			array('soundcloud' => 1),
			array('vk' => 1),
			array('vine' => 1),
			14 => array('wordpress' => 1),
			array('tiktok' => 1)
        );
    	if ( ! empty($ajax_feed) && $ajax_feed != 'all' ) {
    		$this->feed_keys = $ajax_feed;
    	} else {
    		if (isset($setoption['setting']['filters_order']) && is_array($setoption['setting']['filters_order']) && !empty($setoption['setting']['filters_order']) && is_array(current($setoption['setting']['filters_order']))) {
	    		// r3.4.0 < compatibility fix
	    		if ( isset($setoption['setting']['filters_order'][0]) && ! is_array($setoption['setting']['filters_order'][0]) ) {
	    			$this->feed_keys = $filters_order;
	    		} else {
					// r3.7.15 < compatibility fix
					if ( isset($setoption['setting']['filters_order'][16]) && ! is_array($setoption['setting']['filters_order'][16]) )
						$setoption['setting']['filters_order'][16] = array('wordpress' => 1);
					// END - r3.7.15 < compatibility fix

                    $this->feed_keys = $setoption['setting']['filters_order'];
				// END - r3.4.0 < compatibility fix
				}
    		} else {
    			$this->feed_keys = $filters_order;
    		}
        }
    	
        // define theme
		if (isset($attr['theme']) && !empty($attr['theme'])) {
			$theme = $attr['theme'];
		} else {
			$theme = $setoption['setting']['theme'];
		}

        $themeoption = axs_options( $theme );

        // set some settings
        if ( $type == 'carousel' ) {
        	$attr['carousel'] = 'on';
        	$type = 'feed';
        }

        $label = (isset($args['widget_id']) && !empty($args['widget_id'])) ? $args['widget_id'] : $type . $id . $themeoption['layout'];
        
        if ( $type == 'feed' ) {
            $is_feed = true;
            $settingsection = (isset($attr['carousel']) && !empty($attr['carousel'])) ? 'carouselsetting' : 'feedsetting';
            $filterlabel = '';
        } else {
            $is_feed = false;
            $settingsection = $type.'setting';
            $filterlabel = ' filter-label';
        }
        $is_timeline = ( $type == 'timeline' ) ? true : false;
        $is_wall = ( $type == 'wall') ? true : false;
        $is_grid = ( $type == 'grid' ) ? true : false;

        $typeoption = $type;
        if ( $is_feed ) {
            if ( isset($attr['position']) && !empty($attr['position']) ) {
                if ( isset($attr['position']) && $attr['position'] != 'normal' )
                    $typeoption = 'feed_sticky';
            }
            if (isset($attr['carousel']) && !empty($attr['carousel']))
                $typeoption = 'feed_carousel';
        }

        // main container id
        if ( isset($label) && !empty($label) )
            $attr_id = ' id="timeline_'.$label.'"';
        $class = array('sboard');

        // for shortcodes to work - fix
        if ( isset($attr['gutterx']) )
            $attr['gutterX'] = $attr['gutterx'];
        if ( isset($attr['guttery']) )
            $attr['gutterY'] = $attr['guttery'];
        if ( isset($attr['originleft']) )
            $attr['originLeft'] = $attr['originleft'];
        if ( isset($attr['fixwidth']) )
            $attr['fixWidth'] = $attr['fixwidth'];
            
        // merge shortcode and widget attributes with related default settings
        if ( empty($setoption[$settingsection]) )
            $setoption[$settingsection] = array();
        $this->attr = $attr = array_merge($setoption['setting'], $setoption[$settingsection], $attr);

        // set results
		if (isset($attr['results']) && !empty($attr['results'])) {
			$results = $attr['results'];
		} else {
			$results = 10;
		}

		if (isset($args['liveresults']) && !empty($args['liveresults'])) {
			if ( $results < $args['liveresults'] ) {
				$results = $args['liveresults'];
			}
		}

		if ( $results > 100 ) {
			$results = 100;
		}
		
        $attr['results'] = $results;
        
        $attr_ajax = json_encode($attr);
        $attr['cache'] = (int)$attr['cache'];

        // set crawl time limit (servers can not fetch a lot of feeds at the same time)
        $GLOBALS['crawled'] = 0;

		$crawl_limit = 0;
		if($attr['cache'] != 0 && isset($attr['crawl'])){
			$crawl_limit = (int) $attr['crawl'];
		}
        
		// Init official classes
		$this->sbwordpress = new \AXS_AX_SOCIAL\WP\AXS_Wordpress();
		$this->sbinstagram = new \AXS_AX_SOCIAL\INSTAGRAM\AXS_Instagram();
		$this->sbtiktok    = new \AXS_AX_SOCIAL\TIKTOK\AXS_Tiktok();

        // Init cache
        $cache = new AXS_SimpleCache;
        $cache->debug_log = $this->sbinstagram->debug_log = $this->sbtiktok->debug_log = (isset($attr['debuglog']) ? $attr['debuglog'] : '');
        $cache->timeout = $attr['timeout'] ?? '';
        $cache->setoption = $setoption;
        
        // check for stream as a widget
        if ( $is_feed ) {
            if (isset($attr['carousel']) && !empty($attr['carousel'])) {
                $class[] = 'sb-carousel';
                if (isset($args['widget_id']) && !empty($args['widget_id']))
                    $class[] = 'sb-widget';
            } else {
                $class[] = 'sb-widget';
            }
        } elseif ($is_wall) {
            $class[] = 'sb-wall';
        }

        // set the block height
        $block_height = $attr['height'] ?? 400;

        // if slideshow is active
        if (isset($attr['lightboxtype']) && $attr['lightboxtype'] == 'slideshow') {
            $slideshow = true;
            $class[] = 'sb-slideshow';
        }

        // load layout
        $layouts_path = AXS_DIRNAME . '/themes/' . $themeoption['layout'] . '.php';
        if ( ! file_exists($layouts_path) ) {
			$upload_dir = wp_upload_dir();
            $layouts_path = $upload_dir['basedir'] . '/ax-social-stream/themes/' . $themeoption['layout'] . '.php';
            if ( ! file_exists($layouts_path) ) {
                unset($layouts_path);
            }
        }

        $output = $ss_output = '';
        if ( isset($layouts_path) && isset($this->enabled_layouts[$attr['type']]) && !empty($this->enabled_layouts[$attr['type']]) ) {
            require_once 'themes/default.php';
            if ($themeoption['layout'] != 'default')
                include_once($layouts_path);

        if ( $themeoption['classname'] != $themeoption['layout'])
            $class[] = 'sb-' . $themeoption['layout'];

        $layoutclass = 'AXS_' . ucfirst($themeoption['layout']) . '_Layout';
        $layoutobj = new $layoutclass;

		$layoutobj->themetypeoption = $themetypeoption = $themeoption["section_$typeoption"] ?? '';
        if(isset($themeoption[$themeoption['layout'] . "_section_$typeoption"]) && ! empty($themeoption[$themeoption['layout'] . "_section_$typeoption"])){
			$layoutobj->themetypeoption = $themetypeoption = $themeoption[$themeoption['layout'] . "_section_$typeoption"];
		}

		$layoutobj->themedefoption = $themedefoption = $themeoption;
        if(isset($themeoption[$themeoption['layout'] . "_section"]) && ! empty($themeoption[$themeoption['layout'] . "_section"])){
			$layoutobj->themedefoption = $themedefoption = $themeoption[$themeoption['layout'] . "_section"];
		}
		
        $layoutobj->layout = $themeoption['layout'];
        $layoutobj->feed_keys = $filters_order;
        $layoutobj->type = $attr['type'];
        $layoutobj->attr = $attr;

        // load slide layout
        if ( isset($slideshow) ) {
            if ( has_filter( "axs_create_slideitem_{$layoutobj->layout}", array( $layoutobj, 'axs_create_slideitem_callback') ) ) {
                $slidelayoutobj = $layoutobj;
                $slidelayoutobj->attr = $attr;
                $slidelayoutobj->layout = $themeoption['layout'];
            } else {
                $slidelayoutobj = new AXS_Default_Layout;
                $slidelayoutobj->attr = $attr;
                $slidelayoutobj->layout = 'default';
            }
        }

        // Check the theme supports the layout
        if ( ! in_array( $attr['type'], $layoutobj->supported_layouts() ) ) {
            return axs_throw_error( 'The selected theme is not created for this layout!' );
        }

        $target = '';
        // nofollow links
        if (isset($attr['nofollow']) && !empty($attr['nofollow']))
            $target .= ' rel="nofollow noopener"';
        
        // open links in new window
        if (isset($attr['links']) && !empty($attr['links']))
            $target .= ' target="_blank"';
        $this->target = $layoutobj->target = $target;
        if ( isset($slideshow) )
            $slidelayoutobj->target = $target;
        
        // use https
        $protocol = $this->protocol = (isset($attr['https']) && !empty($attr['https'])) ? 'https' : 'http';
        // set output
        if ( ! empty($this->sboption['output']) ) {
            if ( ! is_array($this->sboption['output']) )
                $feedoutput = axs_explode($this->sboption['output']);
            else
				$feedoutput = $this->sboption['output'];
		} else {
            $feedoutput = array( 'title' => true, 'thumb' => true, 'text' => true,
                'comments' => true, 'likes' => true, 'user' => true,
                'share' => true, 'info' => true, 'stat' => true, 'meta' => true, 'tags' => false
            );
		}
        $layoutobj->output = $feedoutput;
        if ( isset($slideshow) )
            $slidelayoutobj->output = $feedoutput;
        
        // Add theme classes
        $class[] = $layoutobj->axs_theme_classes();

        if ( ! $ajax_feed) {
        // do some styling stuffs
        $dotboard = "#timeline_$label.sboard";

        $location = null;
        if ( $is_feed ) {
            $class[] = (isset($attr['position']) && ! empty($attr['position'])) ? $attr['position'] : '';
            if ( isset($attr['position']) && $attr['position'] != 'normal' ) {
                $class[] = (isset($attr['location']) && ! empty($attr['location'])) ? $attr['location'] : '';
                if ( !isset($attr['autoclose']) || empty($attr['autoclose']) ) {
                    $class[] = 'open';
                    $active = ' active';
                }
                
                $locarr = explode('_', str_replace('sb-', '', $attr['location'] ?? '') );
                $location = $locarr[0];
            }
        }

        $args = array(
            'label' => $label,
            'is_feed' => $is_feed,
            'is_timeline' => $is_timeline,
            'is_wall' => $is_wall,
            'is_grid' => $is_grid,
            'dotboard' => $dotboard,
            'location' => $location
        );
        $style = $layoutobj->axs_inline_styles( $args );

        if ($is_wall || $is_grid) {
            $dotitem2 = '.sb-item';
            $sbitem2 = "$dotboard $dotitem2";
            $sbgutter2 = "$dotboard .sb-gsizer";
            $sbgrid2 = "$dotboard .sb-isizer";
            $gutterX = isset($attr['gutterX']) ? $attr['gutterX'] : ($is_grid ? 0 : 10);
            $gutterY = isset($attr['gutterY']) ? $attr['gutterY'] : ($is_grid ? 0 : 10);
            if (isset($attr['itemwidth']) && !empty($attr['itemwidth'])) {
                $itemwidth = isset($attr['itemwidth']) ? $attr['itemwidth'] : $defoption['wallsetting']['itemwidth'];
            }
            // if wall is set to block
            if (isset($attr['fixWidth']) && ($attr['fixWidth'] == 'false' || $attr['fixWidth'] == 'block')) {
                // calculate breakpoints
                $bpsizes = array(1200, 960, 768, 600, 480, 320, 180);
                if ( !isset($attr['breakpoints']) || ! is_array($attr['breakpoints']) ) {
                    if (!isset($attr['breakpoints']) || $attr['breakpoints'] == '')
                        $breakpoints = $defoption['wallsetting']['breakpoints'];
                    else
                        $breakpoints = explode(',', trim($attr['breakpoints']) );
                } else {
                    $breakpoints = $attr['breakpoints'];
                    if (count($breakpoints) != count($bpsizes) ) {
                        foreach ($defoption['wallsetting']['breakpoints'] as $breakKey => $breakVal)
                            if (empty($breakpoints[$breakKey]) )
                                $breakpoints[$breakKey] = $breakVal;
                    }
                }

                foreach ($breakpoints as $bpkey => $breakpoint) {
                    if ($is_grid && $attr['columns_style'] == "1-2") {
                        $breakpoint = round( $breakpoint - ($breakpoint / 3) );
                    }

                    $gut = ($gutterX) ? number_format( ($gutterX * 100) / $bpsizes[$bpkey], 3, '.', '') : 0;
                    if ($gutterY) {
                        $yut = number_format( ($gutterY * 100) / $bpsizes[$bpkey], 3, '.', '');
                        $bpyut = number_format($bpsizes[$bpkey] / (100/$yut), 3, '.', '');
                    } else {
                        $bpyut = 0;
                    }
                    $tw = number_format(100 - ( ($breakpoint - 1) * $gut), 3, '.', '');
                    if ($tw < 100 || ! $gutterX) {
                        $bpgrid = number_format($tw / $breakpoint, 3, '.', '');
                        $bpgut = $gut;
                        $bpgridtwo = number_format( ($bpgrid * 2) + $bpgut, 3, '.', '');
                        $bpgridthree = number_format( ($bpgrid * 3) + ($bpgut * 2), 3, '.', '');
                    } else {
                        $bpgrid = 100;
                        $bpgut = 0;
                        $bpgridtwo = $bpgridthree = 100;
                    }
                    $bpcol[$bpkey] = '';
                    if (isset($attr['fixWidth']) && $attr['fixWidth'] == 'false') {
                        if ($is_grid) {
                            $sbcolumn2 = "$dotboard .sb-column";
                            $bpcol[$bpkey] .= "$sbcolumn2 { width: $bpgrid%; margin-bottom: {$bpyut}px; margin-right: {$bpgut}%; }
                            $sbcolumn2:nth-child({$breakpoint}n) { margin-right: 0; }";
                            if ($attr['columns_style'] == "1-2") {
                                $bpcol[$bpkey] .= "
                                $sbitem2 { width: 50%; height: 40%; margin-bottom: {$bpyut}px; }
                                $sbitem2.sb-twofold { width: 100%; height: 60%; }";
                            }
                        } else {
	                        $bpcol[$bpkey] .= "$sbitem2, $sbgrid2 { width: $bpgrid%; margin-bottom: {$bpyut}px; }
	                        $sbitem2.sb-twofold { width: $bpgridtwo%; }
	                        $sbitem2.sb-threefold { width: $bpgridthree%; }
                            $sbgutter2 { width: $bpgut%; }";
                        }
                    } else {
                        if ($is_grid) {
							$sbcolumn2 = "$dotboard .sb-column";
							$bpcol[$bpkey] .= "$sbcolumn2 { width: $bpgrid%; margin-bottom: {$bpyut}px; margin-right: {$bpgut}%; }
                            $sbcolumn2:nth-child({$breakpoint}n) { margin-right: 0; }";
							if ($attr['columns_style'] == "1-2") {
								$bpcol[$bpkey] .= "
                                $sbitem2 { width: 50%; height: 40%; margin-bottom: {$bpyut}px; }
                                $sbitem2.sb-twofold { width: 100%; height: 60%; }";
							}
                        } else {
                            $bpcol[$bpkey] .= '$("'.$sbitem2.', '.$sbgrid2.'").css({ "width": "'.$bpgrid.'%", "margin-bottom": "'.$bpyut.'px" });
                            $("'.$sbitem2.'.sb-twofold").css({ "width": "'.$bpgridtwo.'%" });
                            $("'.$sbitem2.'.sb-threefold").css({ "width": "'.$bpgridthree.'%" });
                            $("'.$sbgutter2.'").css({ "width": "'.$bpgut.'%" });';
                        }
					}
                }
                
                if (isset($attr['fixWidth']) && ($attr['fixWidth'] == 'false' || $attr['fixWidth'] == 'block')) {
                	$mediaqueries = "$bpcol[0]
@media (min-width: 960px) and (max-width: 1200px) { $bpcol[1] }
@media (min-width: 768px) and (max-width: 959px) { $bpcol[2] }
@media (min-width: 600px) and (max-width: 767px) { $bpcol[3] }
@media (min-width: 480px) and (max-width: 599px) { $bpcol[4] }
@media (min-width: 320px) and (max-width: 479px) { $bpcol[5] }
@media (max-width: 319px) { $bpcol[6] }";
                }
                
                if ($is_grid) {
                    $grid_height = $attr['grid_height'] ?? 340;
                    if ($attr['columns_style'] == "1-2") {
                        $grid_height *= 2;
                    }
                    $style[$sbcolumn2][] = 'height: '.$grid_height.'px';
                }
            } else {
                $style[$sbitem2][] = 'width: '.$itemwidth.'px';
                $style[$sbitem2][] = 'margin-bottom: '.$gutterY.'px';
            }
        }

        if (!empty($attr['carousel']) && !empty($attr['tabable']))
            unset($attr['tabable']);

        if (isset($attr['carousel']) && !empty($attr['carousel']))
            $attr['layout_image'] = 'imgnormal';

        if (isset($attr['tabable']) && !empty($attr['tabable']))
            $class[] = 'tabable';

        if ( (!empty($attr['filters']) or !empty($attr['controls'])) && empty($attr['carousel'])) {
            $style[$dotboard.' .sb-content'][] = 'border-bottom-left-radius: 0 !important;border-bottom-right-radius: 0 !important';
        }
        if ( (!empty($attr['showheader']) || ($location == 'bottom' && empty($attr['tabable'])) ) && $is_feed) {
            $style[$dotboard.' .sb-content'][] = 'border-top: 0 !important;border-top-left-radius: 0 !important;border-top-right-radius: 0 !important';
        }
        if ( $location == 'left' )
            $style[$dotboard.' .sb-content'][] = 'border-top-left-radius: 0 !important';
        if ( $location == 'right' )
            $style[$dotboard.' .sb-content'][] = 'border-top-right-radius: 0 !important';

        if ($is_feed) {
            if (isset($attr['width']) && $attr['width'] != '')
                $style["$dotboard"][] = 'width: '.$attr['width'].'px';
        }
        
        if ( isset($attr['height']) && $attr['height'] != '' && ! $is_feed ) {
            $style[$csskey][] = 'height: '.$attr['height'].'px';
            if ( ! $is_feed ) {
                $style[$csskey][] = 'overflow: scroll';
                if ( $is_timeline )
                    $style[$csskey][] = 'padding-right: 0';
                $style[$dotboard][] = 'padding-bottom: 30px';
            }
        }
        } // end no ajax
        
        if ( isset($theme) && !empty($theme) ) {
            $class['theme'] = $themeoption['classname'];
        }
        
    	if ( ! $order = $attr['order'] )
            $order = 'date';
        
        if ( ! $ajax_feed)
            $output .= "\n<!-- AX Social Stream Plugin By Axent Media -->\t";

    	// Make sure feeds are getting local timestamps
    	if ( get_option( 'timezone_string' ) && strpos( get_option( 'timezone_string' ), 'UTC' ) === false )
    		date_default_timezone_set( get_option( 'timezone_string' ) );

		$GLOBALS['islive'] = false;

        if ($GLOBALS['islive']) {
            // Live update need cache to be disabled
            $forceCrawl = true;
        } else {
            // If a cache time is set in the admin AND the "cache" folder is writeable, set up the cache.
            if ( $attr['cache'] > 0 && is_writable( AXS_DIRNAME . '/cache/' ) ) {
                $cache->cache_path = AXS_DIRNAME . '/cache/';
                $cache->cache_time = $attr['cache'] * 60;
        		$forceCrawl = false;
        	} else {
        		// cache is not enabled, call local class
                $forceCrawl = true;
        	}
        }
        $this->cache = $this->sbinstagram->cache = $this->sbtiktok->cache = $cache;
        $this->forceCrawl = $this->sbinstagram->forceCrawl = $forceCrawl;
        $this->setoption = $setoption;

        if ($this->fetchcomments) {
            return null;
		}

        // If it is an ajax request
        if ( ! empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            if (!empty($_SESSION["$label-temp"])) {
            	$_SESSION[$label] = $_SESSION["$label-temp"];
                $_SESSION["$label-temp"] = array();
            }
            if (isset($_REQUEST['action']) && $_REQUEST['action'] == "sb_liveupdate") {
                $_SESSION["$label-temp"] = $_SESSION[$label] ?? '';
                $_SESSION[$label] = array();
            }
        } else {
            $_SESSION[$label] = array();
            unset($_SESSION["$label-temp"]);
		}

        // Check which feeds are specified
        $feeds = array();
        $firstTab = false;
		$lightbox = false;
		$twitter_i = 0;
		$twitter_app = trim($setoption['section_twitter']['twitter_app'] ?? '');
        foreach ($this->feed_keys as $value) {
            $key = key($value);
            for ($i = 1; $i <= 6; $i++) {
        		if ( !empty($this->sboption['section_'.$key][$key.'_id_'.$i]) && $keyitems = $this->sboption['section_'.$key][$key.'_id_'.$i] ) {
                    foreach ($keyitems as $key2 => $eachkey) {
						// Limit Twitter to maximum 3 feeds if using our App
						if ($twitter_app == 'yes') {
							if ($key == 'twitter') {
								$twitter_i++;
								if ($twitter_i > 4) {
									continue;
								}
							}
						}
                        $load_stop = (isset($_REQUEST['action']) && $_REQUEST['action'] == "sb_loadmore"
							&& ! empty($_SESSION[$label]['loadcrawl']) && empty($_SESSION[$label]['loadcrawl'][$key.$i.$key2]) ) ? true : false;
                        // r3.6.0 < compatibility fix
                        if ( ! is_array($eachkey) ) {
                            $eachkey = ['feed' => $eachkey, 'limit' => 5];
						}
                        // END - r3.6.0 < compatibility fix
                        if ( empty($load_stop) && ! empty($eachkey) && ! empty($eachkey['feed']) ) {
         			        if ( empty($attr['tabable']) || $ajax_feed ) {
                                if ( $crawl_limit && $GLOBALS['crawled'] >= $crawl_limit ) {
                                    break;
								}
                                if ( $feed_data = $this->axs_get_feed( $key, $i, $key2, $eachkey, intval($results), $setoption, $this->sboption['section_'.$key], $cache, $label ) ) {
                                    $filteractive = (isset($attr['default_filter']) && $attr['default_filter'] == $key) ? ' active' : '';
                                    $feeds[$key][$i][$key2] = $feed_data;
                                    if ( isset($value[$key]) && !empty($value[$key]) ) {
                                    	$filterItems[$key] = ($is_feed) ? '<span class="sb-hover sb-'.$key.$filterlabel.$filteractive.'" data-filter=".sb-'.$key.'"><i class="sb-micon sb-'.$key.'"></i></span>' : '<span class="sb-hover sb-'.$key.$filterlabel.$filteractive.'" data-filter=".sb-'.$key.'"><i class="sb-icon sb-'.$key.'"></i></span>';
									}
                                }
                            } else {
                                $activeTab = '';
                                if ( (isset($attr['position']) && $attr['position'] == 'normal') || (!empty($attr['slide']) && empty($attr['autoclose'])) ) {
                                    if ( ! $firstTab ) {
                                        if ( $feed_data = $this->axs_get_feed( $key, $i, $key2, $eachkey, intval($results), $setoption, $this->sboption['section_'.$key], $cache, $label ) ) {
                                            $feeds[$key][$i][$key2] = $feed_data;
                                        }
                                        $firstTab = true;
                                        $activeTab = ' active';
                                    }
                                }
                                
                                $fi = '
                    			<li class="'.$key.($activeTab ?? '').'" data-feed="'.$key.'">';
                                if (isset($attr['position']) && $attr['position'] == 'normal') {
                                    $fi .= '
                                    <span><i class="sb-icon sb-'.$key.'"></i></span>';
                                } else {
                                	$fi .= '
                                    <i class="sb-icon sb-'.$key.'"></i>';
                                    if ( $location != 'bottom' )
                                        $fi .= ' <span>'.ucfirst($key).'</span>';
                                }
                    			$fi .= '</li>';
                                $filterItems[$key] = $fi;
                            }
                        }
                    }
                }
            }
			
			if ($lightbox == false) {
				if ( !empty($this->sboption['section_'.$key][$key.'_iframe']) && !empty($feeds[$key]) ) {
					$lightbox = true;
				}
			}
        }
        
        // set wall custom size if defined
        if ($is_wall) {
            if (isset($attr['wall_height']) && $attr['wall_height'] != '')
            	$style['#sb_'.$label][] = 'height: '.$attr['wall_height'].'px;overflow-y: scroll';
            if (isset($attr['wall_width']) && $attr['wall_width'] != '')
            	$style['#sb_'.$label][] = 'width: '.$attr['wall_width'].'px';
		}
		
        // set timeline style class
        if ( $is_timeline ) {
            $class[] = ($attr['onecolumn'] == 'true') ? 'timeline onecol' : 'timeline';
            $class[] = 'animated';
        }

		if ( !empty( $attr['full_item_link'] ) ) {
			$style['#timeline_'.$label.'.sboard .sb-item'][] = 'cursor: pointer';
		}
        
        if ( ! $ajax_feed) {
            if ( isset($style) && !empty($style) ) {
                $output .= '<style type="text/css">';
                if ( isset($themeoption['custom_css']) && !empty($themeoption['custom_css']) )
                    $output .= $themeoption['custom_css']."\n";
                if ( isset($mediaqueries) && !empty($mediaqueries) )
                    $output .= $mediaqueries."\n";
                foreach ($style as $stKey => $stItem) {
                    $output .= $stKey.'{'.implode(';', $stItem).'}';
                }
                $output .= '</style>';
            }

            if ($is_wall || $is_timeline || $is_grid)
                $output .= '<div id="sb_'.$label.'">';
                
            if ( ! $is_feed && ((isset($attr['position']) && $attr['position'] == 'normal') || ! $is_timeline && empty($attr['tabable']) ) ) {
                $filter_box = '';
                if ( ! empty($this->sboption['filtering_tabs']) ) {
					foreach ($this->sboption['filtering_tabs'] as $filtertab) {
						$filteractive = (isset($filtertab['default_filter']) && !empty($filtertab['default_filter'])) ? ' active' : '';
						if ( isset($filtertab['search_term']) && !empty($filtertab['search_term']) ) {
							$filterItems[] = '<span class="sb-hover sb-filter'.$filteractive.'" data-filter="'.$filtertab['search_term'].'"><i class="sb-filter-icon">'.(!empty($filtertab['filter_icon']) ? '<img class="sb-img" src="' . $filtertab['filter_icon'] . '" alt="" title="'.$filtertab['tab_title'] ?? ''.'">' : $filtertab['tab_title'] ?? '').'</i></span>';
						}
					}
				}
                if ( !empty($attr['filters']) && !empty($filterItems) ) {
                    if (!isset($attr['display_all']) || $attr['display_all'] != 'disabled') {
                    	$display_all = '<span class="sb-hover filter-label'.(!empty($attr['default_filter']) ? '' : ' active').'" data-filter="*" title="'.__( 'Show All', 'social-board' ).'"><i class="sb-icon sb-ellipsis-h"></i></span>';
	                    if (isset($attr['display_all']) && $attr['display_all'] == '1')
	                    	$display_all_last = $display_all ?? '';
	                    else
                    		$display_all_first = $display_all ?? '';
                    }
                    $filter_box .= ($display_all_first ?? '').implode("\n", $filterItems).($display_all_last ?? '');
                }
                if ( isset($attr['filter_search']) && !empty($attr['filter_search']) ) {
					if ($is_grid && !empty($attr['filters'])) {
						$filter_box .= '<input type="text" id="GridSearchFilter" class="sb-search" placeholder="'.__( 'Search...', 'social-board' ).'" />';
					}
					else {
						$filter_box .= '<input type="text" class="sb-search" placeholder="'.__( 'Search...', 'social-board' ).'" />';
					}
				}
                    
                if ($filter_box) {
                    $output .= '
            		<div class="filter-items sb-'.$themeoption['layout'].'">
                        '.$filter_box.'
            		</div>';
                }
            }

            $output .= '<div' . ($attr_id ?? '') . ' class="' . implode(' ', (array) $class ) . '" data-columns>' . "\n";
            if ($is_wall) {
                $output .= '<div class="sb-gsizer"></div><div class="sb-isizer"></div>';
            }
			if ($is_feed) {
                if ( isset($attr['tabable']) && !empty($attr['tabable']) && !empty($filterItems) ) {
					$minitabs = (count($filterItems) > 5) ? ' minitabs' : '';
					$output .= '
					<div class="sb-tabs'.$minitabs.'">
						<ul class="sticky" data-nonce="'.wp_create_nonce( 'tabable' ).'">
						'.implode("\n", $filterItems).'
						</ul>
					</div>';
                }
                if ($is_feed) {
                    if ( empty($attr['tabable']) && !empty($attr['slide']) ) {
                        if ($location == 'left' || $location == 'right') {
                            $opener_image = (!empty($themetypeoption['opener_image'])) ? $themetypeoption['opener_image'] : plugins_url( 'public/img/opener.png', AXS_FILE );
                            $output .= '<div class="sb-opener'.($active ?? '').'" title="'.($attr['label'] ?? '').'"><img class="sb-img" src="'.$opener_image.'" alt="'.($attr['label'] ?? '').'" /></div>';
                        } else {
                            $upicon = '<i class="sb-arrow"></i>';
                        }
                    }
                    if ( !empty($attr['showheader']) || ( $location == 'bottom' && empty($attr['tabable']) ) )
                        $output .= '<div class="sb-heading'.($active ?? '').'">'.($attr['label'] ?? '').($upicon ?? '').'</div>';
                }
                
                $content_style = (empty($attr['carousel'])) ? ' style="height: '.$block_height.'px"' : '';
                if (isset($attr['carousel']) && !empty($attr['carousel'])) {
                    $carousel_classes = $layoutobj->axs_carousel_classes();
                    $output .= '<div id="ticker_'.$label.'" class="ax-slider'.$carousel_classes.'">';
                } else {
                    $output .= '<div class="sb-content"'.$content_style.'>';
                    $output .= '<ul id="ticker_'.$label.'">';
                }
            }
        }

        // Parsing the combined feed items and create a unique feed
        if ( ! empty($feeds) ) {
            foreach ($feeds as $feed_class => $feeditem) {
                foreach ($feeditem as $i => $feeds2) {
                foreach ($feeds2 as $ifeed => $feed) {

                // Facebook
                if ($feed_class == 'facebook') {
                    if (!empty($feed)) {
                        $iframe = $this->sboption['section_facebook']['facebook_iframe'] ?? '';

                    if ($i != 3 && $i != 4 && ! empty($feed['data']) ) {
                    	$feednext = $feed['next'];
                    	$feed = $feed['data'];
                    }
                    // loop through feeds
                    if (!empty($feed))
                    foreach ($feed as $data) {
                    if (!empty($data)) {
                        if ($i == 3 || $i == 4) {
                        	if(isset($data->paging->cursors->after) && ! empty($data->paging->cursors->after)){
								$loadcrawl[$feed_class.$i.$ifeed] = $data->paging->cursors->after;
							}
                        	else{
								$loadcrawl[$feed_class.$i.$ifeed] = "";
							}
                            
                            $data = $data->data;
                        }
                    
                    // loop through feed items
                    foreach ($data as $entry) {
                        $url = $play = $text = $embed = $object_id = $source = $image2 = $mediasize = $type2 = '';

                        if ( !empty($entry->attachments) ) {
                            if ( $i == 1 && (empty($this->sboption['section_facebook']['facebook_pagefeed']) || $this->sboption['section_facebook']['facebook_pagefeed'] != 'tagged') ) {
                                $entry->type = $entry->attachments->data[0]->media_type ?? '';
                                $entry->description = $entry->attachments->data[0]->description ?? '';
                                $entry->link = $entry->attachments->data[0]->unshimmed_url ?? '';
                                $entry->name = $entry->attachments->data[0]->title ?? '';
								$entry->object_id = ($entry->attachments->data[0]->media_type == 'photo' && isset( $entry->attachments->data[0]->target->id) ) ? $entry->attachments->data[0]->target->id : '';
                                $entry->source = $entry->attachments->data[0]->media->source ?? '';
                                $entry->caption = $entry->attachments->data[0]->title ?? '';
                            }
                        }
						
                        if ( isset($entry->type) ) {
							if ( $entry->type == "link" && ( ! $entry->object_id && ! $entry->link ) ) {
								continue;
							}
						}

						// Get real post id
						$idparts = array();
						$restricted_content = false;
						if(isset($entry->id) && ! empty($entry->id)){
							$idparts = explode('_', $entry->id);

							if ($i == 1
								&& (@$this->sboption['section_facebook']['facebook_pagefeed'] == 'posts')
								&& $idparts[0] != $entry->from->id) {
								$restricted_content = true;
							}
						}
                        
                        // define link
                        if ( isset($entry->permalink_url) ) {
                            $link = $entry->permalink_url;
                        } else {
                            if ( !empty($idparts) && count($idparts) > 1 )
                                $link = 'https://www.facebook.com/'.$idparts[0].'/posts/'.$idparts[1];
                            elseif (!empty($entry->from->id) && !empty($entry->id))
                                $link = 'https://www.facebook.com/'.$entry->from->id.'/posts/'.$entry->id;
                        }
                        
                        if ( ! $link)
                            $link = $entry->link ?? '';
                        if ( ! $link)
                            $link = $entry->source ?? '';

                        // Fixing the links when permalink_url does not contain facebook.com
                        $link = trim($link);
                        if ( 0 !== stripos( $link, 'http' ) ) {
                            if ( 0 !== stripos( $link, '/' ) ) {
                                $link = '/'.$link;
                            }
                            $link = 'https://www.facebook.com'.$link;
                        }

                        if ( $this->make_remove($link) && empty($restricted_content) ) {
                        // body text
                        $content = '';
                        if ( isset($entry->message) && !empty($entry->message) ) {
                            $content .= $entry->message;
                        } elseif ( isset($entry->description) && !empty($entry->description) ) {
                            $content .= $entry->description;
                        } elseif ( isset($entry->story) && !empty($entry->story) ) {
                            $content .= $entry->story;
						}

                        if ( !$this->make_filters( array( $content ) ) ) {
							continue;
						}
                        
                        $text = (!empty($attr['words'])) ? $this->word_limiter($content, $link) : $this->format_text($content);

                        // link hashtags
                        if ( isset($text) && !empty($text) ) {
                            $text = $this->add_links('facebook', $text);
                        }

                        $meta = array();

                        // comments
                        $count = 0;
                        $meta['comments']['data'] = '';
                        $comments_count = ( isset($this->sboption['section_facebook']['facebook_comments']) && $this->sboption['section_facebook']['facebook_comments'] > 0 ) ? $this->sboption['section_facebook']['facebook_comments'] : 0;

						$comments = false;
						if((isset($entry->comments) && is_array($entry->comments))){
							$comments = $entry->comments;
						}
						if(isset($entry->comments->data) && $comments == false){
							$comments = $entry->comments->data;
						}	
                        if ( $comments != false && $comments_count ) {
                            foreach ( $comments as $comment ) {
                                $count++;
                                $meta['comments']['data'] .= $this->facebook_get_comments($comment, $link);
                                // display replies
                                $replies = is_array($comment->comments) ? $comment->comments : $comment->comments->data;
                                if ( ! empty($replies) ) {
                                    foreach ( $replies as $reply ) {
										if ( $count >= $comments_count ) break;
                                        $meta['comments']['data'] .= $this->facebook_get_comments($reply, $link);
										$count++;
                                    }
                                }
                                if ( $count >= $comments_count ) break;
                            }
                        }
                        $meta['comments']['total'] = [
                            'count' => $entry->comments->summary->total_count ?? 0
                        ];

                        // likes
                        $count = 0;
                        $meta['likes']['data'] = '';
                        $likes_count = ( isset($this->sboption['section_facebook']['facebook_likes']) && $this->sboption['section_facebook']['facebook_likes'] > 0 ) ? $this->sboption['section_facebook']['facebook_likes'] : 0;
                        if ( isset($entry->likes) ) {
                            $likes = is_array($entry->likes) ? $entry->likes : $entry->likes->data;
                            if ( isset($likes) && $likes_count ) {
                                foreach ( $likes as $like ) {
                                    $count++;
                                    $like_title = (isset($like->name) && !empty($like->name)) ? ' title="' . $like->name . '"' : '';
                                    $like_image = (isset($like->pic) && !empty($like->pic)) ? $like->pic : $protocol.'://graph.facebook.com/' . $like->id . '/picture?type=square';
                                    $meta['likes']['data'] .= '<a href="' . (!empty($like->link) ? $like->link : 'https://www.facebook.com/'.$like->id) . '"'.$target.'><img class="sb-img" src="' . $like_image . '"' . $like_title . ' alt="'.( $like->name ?? '' ).'"></a>';
                                    if ( $count >= $likes_count ) break;
                                }
                            }
                            $meta['likes']['total'] = [
                                'count' => $entry->likes->summary->total_count ?? 0
                            ];
                        }
                        
                        $image_width = $this->sboption['section_facebook']['facebook_image_width'] ?? 300;
                        $source = $entry->picture ?? '';
                        $object_id = $entry->object_id ?? '';
                        if ($iframe) {
                            $url = $source;
                            $image_width_iframe = 800;
                        }
                        if ( ! empty($entry->images) ) {
                            if ($image_width) {
                                // get closest image width
                                $closest = null;
                                foreach ($entry->images as $image) {
                                    if ($closest === null || abs($image_width - $closest) > abs($image->width - $image_width)) {
                                        $closest = $image->width;
                                        $source = $image->source;
                                    }
                                }
                            }
                            // set iframe image
                            if ($iframe) {
                                $closest = null;
                                foreach ($entry->images as $image2) {
                                    if ($closest === null || abs($image_width_iframe - $closest) > abs($image2->width - $image_width_iframe)) {
                                        $closest = $image2->width;
                                        $url = $image2->source;
                                        $mediasize = $image2->width.','.$image2->height;
                                    }
                                }
                            }
                        } else {
                            if (isset($entry->attachments) && !empty($entry->attachments)) {
                                $type2 = $entry->attachments->data[0]->type ?? '';
                                if ($type2 == 'multi_share') {
									$image2 = $entry->attachments->data[0]->subattachments->data[0]->media->image ?? '';
								} else {
	                                $image2 = $entry->attachments->data[0]->media->image ?? '';
	                                if ( ! $image2 ) {
	                                    $image2 = $entry->attachments->data[0]->subattachments->data[0]->media->image ?? '';
	                                } else {
		                                if ( empty($mediasize))
		                                	$mediasize = $image2->width.','.$image2->height;
									}
                                }
								if (isset($image2->src) && !empty($image2->src)) {
									$source = $image2->src;
									if ($iframe) {
										$url = $source;
									}
								}
                            }

                            // get or create thumb
                            if ($image_width > 180 && (empty($type2) || $type2 != 'multi_share')) {
                                if (isset($entry->full_picture) && !empty($entry->full_picture)) {
                                    $urlArr = explode('&url=', $entry->full_picture);
                                    if (isset($urlArr[1]) && $urlfb = $urlArr[1]) {
                                        if (stristr($urlfb, 'fbcdn') == TRUE || stristr($urlfb, 'fbstaging') == TRUE) {
                                            $source = $entry->full_picture."&w=$image_width&h=$image_width";
                                        } else {
                                            $urlfbArr = explode('&', $urlfb);
                                            if (isset($attr['image_proxy']) && !empty($attr['image_proxy'])) {
                                            	$token = md5(($urlfbArr[0] ?? '').($_SERVER['SERVER_ADDR'] ?? '').($_SERVER['SERVER_ADMIN'] ?? '').($_SERVER['SERVER_NAME'] ?? '').($_SERVER['SERVER_PORT'] ?? '').($_SERVER['SERVER_PROTOCOL'] ?? '').strip_tags($_SERVER['SERVER_SIGNATURE'] ?? '').($_SERVER['SERVER_SOFTWARE'] ?? '').($_SERVER['DOCUMENT_ROOT'] ?? ''));
                                            	$imgStr = 'resize='.$image_width.'&refresh=3600&token='.$token.'&src='.($urlfbArr[0] ?? '');
                                            	$source = esc_url( add_query_arg( 'sbimg', base64_encode($imgStr), site_url('index.php') ) );
                                            } else {
                                                $source = (empty($type2) || $type2 != 'share') ? urldecode($urlfbArr[0] ?? '') : $entry->full_picture;
											}
                                        }
                                    } else {
                                        $source = $entry->full_picture;
                                    }
                                } else {
                                    if ( $object_id && empty($source) )
                                        $source = $protocol.'://graph.facebook.com/'.$object_id.'/picture?type=normal';
                                }
                            }
                            if (empty($source) ) {
                                if (isset($image2->src) && !empty($image2->src))
                                    $source = $image2->src;
                            }

                            // set iframe image
                            if ($iframe) {
                                if (isset($type2) && $type2 == 'multi_share' && !empty($image2->src)) {
                                	$url = $image2->src;
                                } else {
	                                if (isset($entry->full_picture) && !empty($entry->full_picture)) {
	                                    $url = $entry->full_picture;
	                                } else {
	                                    if ($object_id) {
	                                        $url = $protocol.'://graph.facebook.com/'.$object_id.'/picture?type=normal';
										}
	                                }
                                }
                            }
                        }

                        // set video
                        if ( (isset($entry->type) && $entry->type == 'video')
							|| (isset($entry->status_type) && $entry->status_type == 'added_video')
							|| $i == 4
							|| stristr($type2, 'animated_image') == TRUE) {
                            $play = true;
                            $video_width = $this->sboption['section_facebook']['facebook_video_width'] ?? 720;
                            if (isset($entry->format) && !empty($entry->format)) {
                                // get closest video width
                                $videosize = null;
                                foreach ($entry->format as $eformat) {
                                    if (abs($eformat->width) >= abs($video_width) ) {
                                        $videosize = $eformat;
                                        $source = $eformat->picture;
                                        break;
                                    }
                                }
                                if ( ! $videosize)
                                	$videosize = end($entry->format);
                                $mediasize = $videosize->width.','.$videosize->height;
                            }
                            if (empty($mediasize))
                                $mediasize = '640,460';
                            if ($iframe) {
                                if (stristr($type2, 'animated_image') == TRUE) {
                                    if (isset($entry->link) && stristr($entry->link, '.gif') == TRUE)
                                        $url = $entry->link ?? '';
                                    else {
                                        if (isset($entry->link) && stristr($entry->link, 'giphy.com') == TRUE) {
                                            $giphyID = substr($entry->link, strrpos($entry->link, '-') + 1);
                                            $url = 'https://i.giphy.com/'.$giphyID.'.gif';
                                        }
                                    }
                                } else {
                                    if (isset($entry->status_type) && $entry->status_type == 'shared_story') {
                                        $url = $entry->link;
                                    } else {
                                        $msize = explode(',', $mediasize);
                                        if (isset($entry->link) && !empty($entry->link)){
											$vlink = $entry->link;
										} elseif(isset($entry->status_type) && $entry->status_type == 'added_video'){
											if(isset($entry->attachments) && isset($entry->attachments->data)){
												if(!empty($entry->attachments->data) && is_array($entry->attachments->data)){
													foreach ($entry->attachments->data as $r){
														if(isset($r->media_type) && $r->media_type == "video"){
															$vlink = $r->url;
															break;
														}
													}
												}
											}
										} else {
                                            $vlink = 'https://www.facebook.com/'.($entry->from->id ?? 'facebook').'/videos/'.$entry->id.'/';
                                        }
                                        $url = 'https://www.facebook.com/plugins/video.php?href='.urlencode($vlink).'&show_text=0&width='.$msize[0].'&height='.($msize[1] ?? '');
                                    }
                                }

								if (isset($entry->source) && stristr($entry->source, 'youtube.com/embed') == TRUE) {
									$url = $entry->source;
									$entry->status_type = 'added_video';
								} else {
									// if shared from youtube without embed
									if (strpos($url, 'youtube.com') == TRUE or strpos($url, 'youtu.be') == TRUE) {
										if (stristr($url, 'youtube.com/embed') === FALSE) {
											$url = $this->youtube_get_embedurl($url) . '?rel=0&wmode=transparent';
										}
									}
								}
                            }
                        }

                        switch ($i) {
                            case 3:
                                $itemtype = 'image';
                                $type_icon = $themedefoption['type_icons'][4] ?? '';
                            break;
                            case 4:
                                $itemtype = 'video-camera';
                                $type_icon = $themedefoption['type_icons'][6] ?? '';
                            break;
                            default:
                                $itemtype = 'pencil';
                                $type_icon = $themedefoption['type_icons'][0] ?? '';
                            break;
                        }
                        
                        // Set title
                        $title = $entry->name ?? ($entry->title ?? '');
                        if (empty($title) ) {
                            if (isset($entry->attachments) && !empty($entry->attachments)) {
                                $title = $entry->attachments->data[0]->title ?? '';
                            }
                        }
                        if ($title == "'s cover photo") {
                            $title = (!empty($entry->from->name)) ? $entry->from->name . $title : '';
                        }

                        // If is a shared link and the picture is very small
                        if (isset($entry->type) && $entry->type == 'link' && ! empty($entry->link) && isset($image2->width) && $image2->width < 150) {
                            $source = '';
                            if (isset($this->sboption['section_facebook']['facebook_embed'])
								&& !empty($this->sboption['section_facebook']['facebook_embed'])) {
                                $embed = $this->get_embed($entry->link, true);
                            } else {
                                $embed = '<a class="sb-html-embed" href="' . $entry->link . '" target="_blank">';
                                $embed .= '<div class="sb-embed-user">';
                                if ( ! empty($entry->picture) && !empty($entry->name) ) {
                                    $embed .= '
                                    <div class="sb-embed-uthumb">
                                        <img class="sb-img" alt="' . $entry->name . '" src="' . $entry->picture . '">
                                    </div>';
                                }
                                $embed .= '<div class="sb-embed-uinfo">' . $entry->name . '</div>
                                </div>';
                                if ( ! empty($entry->description) )
                                    $embed .= '<span class="sb-text">' . $this->word_limiter($entry->description, $entry->link) . '</span>';
                                $embed .= '</a>';
                            }
                            $text .= $embed;
                            $title = $entry->story ?? '';
                        }

                        $entryPicture = isset($entry->from->picture->data) ? $entry->from->picture->data : ($entry->from->picture ?? '');

                        $thetime = $entry->created_time ?? $entry->updated_time;
                        $sbi = $this->make_timestr($thetime, $link);
                        $itemdata = array(
                        'title' => (isset($title) && !empty($title)) ? '<a href="' . ( (isset($entry->link) && $entry->link && (stristr($entry->link, 'http://') == TRUE || stristr($entry->link, 'https://') == TRUE) ) ? $entry->link : $link) . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($title) : $title) . '</a>' : '',
                        'thumb' => $source ?? '',
                        'thumburl' => $url ? $url : $link,
                        'text' => $text ?? '',
                        'meta' => $meta ?? '',
                        'url' => $link ?? '',
                        'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? ( ( (isset($entry->type) && $entry->type == 'video') || (isset($entry->status_type) && $entry->status_type == 'added_video') || $i == 4 ) ? 'iframe' : 'icbox' ) : '',
                        'date' => $thetime,
                        'user' => array(
                            'name' => $entry->from->name ?? '',
                            'url' => $entry->from->link ?? (!empty($entry->from->id) ? 'https://www.facebook.com/' . $entry->from->id : ''),
                            'image' => $entryPicture->url ?? (!empty($entry->from->id) ? $protocol.'://graph.facebook.com/' . $entry->from->id . '/picture?type=square' : ''),
                            'status' => ""
                        ),
                        'type' => $itemtype,
                        'play' => $play ?? '',
                        'icon' => array($themedefoption['social_icons'][0] ?? '', $type_icon)
                        );

							if (isset($mediasize) && !empty($mediasize) && ($iframe || isset($slideshow) ) )
                                $itemdata['size'] = $mediasize;

                            $this->final[$sbi] = array( $feed_class, $itemdata, $attr, $sbi, $i, $ifeed );
                            if ( isset($slideshow) ) {
                                $content = $this->format_text($content ?? '');
                                $itemdata['text'] = $this->add_links('facebook', $content);
                                if ($url)
                                    $itemdata['thumb'] = $url;
                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                            }
                        }
                    } // end foreach
                    
                    if ($i != 3 && $i != 4) {
                        // facebook get last item date
                        if ( ! empty($feednext) && isset($thetime) )
                        	$loadcrawl[$feed_class.$i.$ifeed] = strtotime($thetime)-1;
                    }
                    
                    } // end $data
                    }
                    }
                }
        		// Twitter
                elseif ( $feed_class == 'twitter' ) {
                    if (isset($feed) && !empty($feed->data)) {
                        $iframe = $this->sboption['section_twitter']['twitter_iframe'] ?? '';

						// Private api result
                        $twitter_app = trim($setoption['section_twitter']['twitter_app'] ?? '');
                        if ($twitter_app == 'yes') {
                            foreach ($feed->data as $data) {
                                if (isset($data->timestamp) ) {
                                    if (isset($_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed])
                                        && $_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed] == $data->id) {
                                        continue;
                                    }
                                $link = $data->permanentUrl;
                                if ($this->make_remove($link) ) {
                                    $url = $thumb = $mediasize = $play = $meta = $embed = '';
                                    if ( !empty($data->text) ) {
                                        $text = $data->text;
                                    } else {
                                        $text = $data->html;
                                    }
                                    $text = str_replace("\n", " \n", $text);
                                    $text = (isset($attr['words']) && ! empty($attr['words'])) ? $this->word_limiter($text, $link) : $this->format_text($text);

                                    if ( !empty($data->videos[0]->preview) ) {
                                        $thumb = $data->videos[0]->preview;
                                        $url = $data->videos[0]->url;
                                        $play = true;
                                    } elseif ( !empty($data->photos[0]->url) ) {
                                        $thumb = $data->photos[0]->url;
                                        $url = $thumb;
                                    }

                                    $meta = array();
                                    $meta['likes']['total'] = [
                                        'count' => $data->likes ?? 0,
                                        'url' => 'https://twitter.com/intent/favorite?tweet_id=' . $data->id,
                                        'class' => 'star-o'
                                    ];
                                    $meta['comments']['total'] = [
                                        'count' => $data->replies ?? 0,
                                        'url' => 'https://twitter.com/intent/tweet?in_reply_to=' . $data->id . '&via=' . $data->username,
                                        'class' => 'reply'
                                    ];
                                    $meta['retweets']['total'] = [
                                        'count' => $data->retweets ?? 0,
                                        'url' => 'https://twitter.com/intent/retweet?tweet_id=' . $data->id . '&via=' . $data->username
                                    ];

                                    if (isset($data->isRetweet) && ! empty($data->isRetweet)) {
                                        $meta['retweeted']['data'] = '
                                            <span class="sb-meta sb-tweet">
                                                <a href="https://twitter.com/' . $data->username . '"'.$target.'><img class="sb-img" src="' . $data->name . '" alt="' . $data->username . '"> ' . $data->name . ' '.ucfirst(__( 'retweeted', 'social-board' )).'</a>
                                            </span>';
                                    }

                                    $sbi = $this->make_timestr($data->timestamp, $link);
                                    $itemdata = array(
                                        'thumb' => $thumb,
                                        'thumburl' => $url,
                                        'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? (!empty($play) ? 'iframe' : 'icbox') : '',
                                        'text' => $text,
                                        'meta' => $meta ?? '',
                                        'url' => $link,
                                        'date' => $data->timeParsed,
                                        'user' => array(
                                            'name' => $data->username,
                                            'url' => 'https://twitter.com/'.$data->username,
                                            'title' => $data->name,
                                            'image' => ''
                                        ),
                                        'type' => 'pencil',
                                        'play' => $play ?? '',
                                        'icon' => array($themedefoption['social_icons'][1] ?? '', $themedefoption['type_icons'][0] ?? '')
                                    );
                                    
                                    if (isset($mediasize) && ! empty($mediasize) && ($iframe || isset($slideshow) ) ) {
                                        $itemdata['size'] = $mediasize;
                                    }
                                    $this->final[$sbi] = array( $feed_class, $itemdata, $attr, $sbi, $i, $ifeed );
                                    if ( isset($slideshow) ) {
                                        $text = $data->text ?? $data->html;
                                        $text = $this->format_text($text ?? '');
                                        $itemdata['text'] = $this->twitter_add_links($text, $embed_urls, $referenced_tweet);
                                        $itemdata['size'] = $mediasize ?? '';
                                        if ($url) {
                                            $itemdata['thumb'] = $url;
                                        }
                                        $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                                    }
                                }
                                }
                            } // end foreach
                        } else {
						$user = $feed->includes->users[0];
						foreach ($feed->includes->users as $userobj) {
							$users[$userobj->id] = $userobj;
						}
						if (isset($feed->includes->media)) {
							foreach ($feed->includes->media as $mediaobj) {
								$media[$mediaobj->media_key] = $mediaobj;
							}
						}
						if (isset($feed->includes->tweets)) {
							foreach ($feed->includes->tweets as $reftweetobj) {
								$reftweet[$reftweetobj->id] = $reftweetobj;
							}
						}
	                    foreach ( $feed->data as $data ) {
	                        if ( isset($data->created_at) ) {
		                        if (isset($_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed])
									&& $_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed] == $data->id) {
		                            continue;
								}
	                        $link = $protocol.'://twitter.com/' . $user->username . '/status/' . $data->id;
	                        if ( $this->make_remove($link) ) {
                                $url = $thumb = $mediasize = $play = $text = '';
	                            if ( ! empty($data->text) ) {
									$text = $data->text;
								}
                                $text = str_replace("\n", " \n", $text);
								if ( !$this->make_filters( array( $text ) ) ) {
									continue;
								}
                                $text = (!empty($attr['words'])) ? $this->word_limiter($text, $link) : $this->format_text($text);

		                        // get attachments
								$mediaobj = '';
								if (!empty($data->attachments->media_keys)) {
									$mediakey = $data->attachments->media_keys[0];
									if (!empty($media[$mediakey])) {
										$mediaobj = $media[$mediakey];
										// get video
										if ((isset($mediaobj->type) && $mediaobj->type == 'video')
											|| (isset($mediaobj->type) && $mediaobj->type == 'animated_gif')) {
											$twitter_videos = $this->sboption['section_twitter']['twitter_videos'] ?? 'small';
											$thumb = $mediaobj->preview_image_url . ':' . $twitter_videos;
											$play = true;
											if ($iframe) {
												if (isset($mediaobj->height) && isset($mediaobj->width)) {
													// get the video size
													$mediasize = $mediaobj->width.','.$mediaobj->height;
												}
												$url = $mediaobj->variants[0]->url;
											}
										} else {
											// get image
											$twitter_images = $this->sboption['section_twitter']['twitter_images'] ?? '';
											$thumb = $mediaobj->url . ':' . $twitter_images;
											if ($iframe) {
												$url = $mediaobj->url . ':large';
												$mediasize = $mediaobj->width.','.$mediaobj->height;
											}
										}
									}
								}

                                // Convert hashtags to links
                                // Convert shared links to embed only if there is no media tag set
								$embed_urls = null;
								if (isset($data->entities)) {
									$entities = $data->entities;
                                	if (!empty($this->sboption['section_twitter']['twitter_embed']) && !empty($entities->urls) && empty($mediaobj)) {
										$embed_urls = $entities->urls;
									}
								}
								$referenced_tweet = null;
								if ( isset($data->referenced_tweets) && !empty($data->referenced_tweets[0]) ) {
									$referencedkey = $data->referenced_tweets[0]->id;
									$reftweet[$referencedkey]->user = $users[$reftweet[$referencedkey]->author_id];
									$referenced_tweet = $reftweet[$referencedkey];
								}
								$text = $this->twitter_add_links($text, $embed_urls, $referenced_tweet);

                                $meta = array();
                                $meta['likes']['total'] = [
                                    'count' => $data->public_metrics->like_count ?? 0,
                                    'url' => 'https://twitter.com/intent/favorite?tweet_id=' . $data->id,
                                    'class' => 'star-o'
                                ];
                                $meta['comments']['total'] = [
                                    'count' => $data->public_metrics->reply_count ?? 0,
                                    'url' => 'https://twitter.com/intent/tweet?in_reply_to=' . $data->id . '&via=' . $user->username,
                                    'class' => 'reply'
                                ];
                                $meta['retweets']['total'] = [
                                    'count' => $data->public_metrics->retweet_count ?? 0,
                                    'url' => 'https://twitter.com/intent/retweet?tweet_id=' . $data->id . '&via=' . $user->username
                                ];
								/*
                                if (isset($data->retweeted_status) && !empty($data->retweeted_status)) {
                                    $meta['retweeted']['data'] = '
                                        <span class="sb-meta sb-tweet">
                                            <a href="https://twitter.com/' . $user->username . '"'.$target.'><img class="sb-img" src="' . $data->user->profile_image_url . '" alt="' . $data->user->name . '"> ' . $data->user->name . ' '.ucfirst(__( 'retweeted', 'social-board' )).'</a>
                                        </span>';
                                }*/
                                
                                $sbi = $this->make_timestr($data->created_at, $link);
                                $itemdata = array(
                                'thumb' => $thumb,
                                'thumburl' => $url ? $url : $link,
                                'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? (!empty($play) ? 'iframe' : 'icbox') : '',
                                'text' => $text,
                                'meta' => $meta ?? '',
                                'url' => $link,
                                'date' => $data->created_at,
                                'user' => array(
                                    'name' => $user->username,
                                    'url' => 'https://twitter.com/'.$user->username,
                                    'title' => $user->name,
                                    'image' => $user->profile_image_url
                                    ),
                                'type' => 'pencil',
                                'play' => $play ?? '',
                                'icon' => array($themedefoption['social_icons'][1] ?? '', $themedefoption['type_icons'][0] ?? '')
                                );
								if (isset($mediasize) && !empty($mediasize) && ($iframe || isset($slideshow) ) ) {
									$itemdata['size'] = $mediasize;
								}
								$this->final[$sbi] = array( $feed_class, $itemdata, $attr, $sbi, $i, $ifeed );
	                            if ( isset($slideshow) ) {
	                                $text = $data->text ?? $data->full_text;
	                                $text = $this->format_text($text ?? '');
	                                $itemdata['text'] = $this->twitter_add_links($text, $embed_urls, $referenced_tweet);
									$itemdata['size'] = $mediasize ?? '';
	                                if ($url) {
	                                    $itemdata['thumb'] = $url;
									}
	                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
	                            }
	                        }
	                        }
	                    } // end foreach
                        } // end official api result
	                    
                    	// twitter get last id
						if(isset($data->id)){
							$loadcrawl[$feed_class.$i.$ifeed] = $data->id;
						}
                    }
        		}
                elseif ( $feed_class == 'tumblr' ) {
                    $keyTypes = array( 'text' => array('pencil', 0), 'quote' => array('quote-right', 2), 'link' => array('link', 3), 'answer' => array('reply', 1), 'video' => array('video-camera', 6), 'audio' => array('youtube-play', 7), 'photo' => array('image', 4), 'chat' => array('comment', 9) );
                    $iframe = $this->sboption['section_tumblr']['tumblr_iframe'] ?? '';
                    
                    // tumblr next page start
                    $loadcrawl[$feed_class.$i.$ifeed] = (!empty($_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed])) ? $_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed] + $results : $results;
                    
                    // blog info
                    $blog = $feed->response->blog;
                    
                    if (isset($feed->response->posts) && !empty($feed->response->posts)) {
                        
                    foreach ($feed->response->posts as $item) {
                        $title = $thumb = $text = $textlong = $body = $object = $tags = $url = $mediasrc = $mediasize = $play = '';
                        
                        $link = $item->post_url ?? '';
                        if ( $this->make_remove($link) ) {
                        
                        // tags
                        if ( isset($feedoutput['tags']) && !empty($feedoutput['tags']) ) {
                            if ( isset($item->tags) && !empty($item->tags) ) {
                                $tags = implode(', ', $item->tags);
                            }
                        }
                        
                        if ( isset($item->title) && !empty($item->title) ) {
                            $title = '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($item->title) : $item->title) . '</a>';
                        }
                        
                        // set image
						$photoItem = '';
                        if (isset($item->photos[0]) && !empty($item->photos[0])) {
							$photoItem = $item->photos[0];
                            if (isset($photoItem->alt_sizes) && !empty($photoItem->alt_sizes)) {
                                foreach ($photoItem->alt_sizes as $photo) {
                                    if (isset($this->sboption['section_tumblr']['tumblr_thumb']) && $photo->width == $this->sboption['section_tumblr']['tumblr_thumb'])
                                        $thumb = $photo->url;
                                }
                            }
                            // set iframe image
                            if ($iframe) {
								$original = '';
                                if (isset($photoItem->original_size) && !empty($photoItem->original_size)) {
									$original = $photoItem->original_size;
                                    $url = $mediasrc = $original->url;
                                    $mediasize = $original->width.','.$original->height;
                                }
                            }
                        }
                        
                        if ($item->type == 'photo') {
                            $body = $item->caption ?? '';
                        }
                        elseif ($item->type == 'video') {
                            $url = (isset($item->video_type) && $item->video_type == 'tumblr') ? ($item->video_url ?? '') : ($item->permalink_url ?? '');
                            if (isset($item->thumbnail_url) && !empty($item->thumbnail_url))
                                $thumb = $item->thumbnail_url;
                                
                            if ($iframe) {
                                // set player
                                if (isset($item->player) && !empty($item->player)) {
                                    foreach ($item->player as $player) {
                                        if (isset($this->sboption['section_tumblr']['tumblr_video']) && $player->width == $this->sboption['section_tumblr']['tumblr_video']) {
                                            $object = $player->embed_code;
											if (isset($original->height) && !empty($original->height)) {
												$player_height = number_format( ($player->width * $original->height) / $original->width, 0, '.', '');
												$mediasize = $player->width.','.$player_height;
											}
											break;
										}
                                    }
                                }
                            }
                            $body = $item->caption ?? '';
                            $play = true;
                        }
                        elseif ( $item->type == 'audio') {
                            $tit = ($item->artist ?? '') . ' - ' . ($item->album ?? '') . ' - ' . ($item->id3_title ?? '');
                            $title = '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($tit) : $tit) . '</a>';
                            if (empty($thumb))
                                $thumb = plugins_url( 'public/img/thumb-audio.png', AXS_FILE );
                            $body = $item->caption ?? '';
                            $object = $item->embed ?? '';
                        }
                        elseif ( $item->type == 'link') {
                            $title = '<a href="' . $item->url . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($item->title) : $item->title) . '</a>';
                            if (isset($item->excerpt) && !empty($item->excerpt))
                                $excerpt = $item->excerpt." \n";
                            $text = $body = ($excerpt ?? '').($item->description ?? '');
                            if (empty($thumb))
                                $thumb = $item->link_image ?? '';
                            if ( ! $url) {
								$url = $item->link_image;
								$mediasize = $item->link_image_dimensions->width.','.$item->link_image_dimensions->height;
							}
                        }
                        elseif ( $item->type == 'answer') {
                            $text = $body = ($item->question ?? '')." \n".($item->answer ?? '');
                        }
                        elseif ( $item->type == 'quote') {
                            if (isset($item->source) && !empty($item->source))
                                $source = $item->source;
                            $text = $textlong = '<span class="sb-title">'.($item->text ?? '').'</span>'.($source ?? '');
                        }
                        elseif ( $item->type == 'chat') {
                            $text = $body = $item->body ?? '';
                        }
                        // type = text
                        else {
                            if ( isset($item->body) && !empty($item->body) )
                                $text = $body = $item->body;
                            
                            // find img
                            if (empty($thumb)) {
                                $thumbarr = axs_getsrc($body);
                                $thumb = isset($thumbarr['src']) ? $thumbarr['src'] : '';
                            }
                        }
                        
                        if ($iframe) {
                            if ( empty($url) )
                                $url = $thumb ?? '';
                        }
                        
                        if ( empty($text) ) {
							$text = $item->summary ?? '';
						}

                        if ( !$this->make_filters( array( $text ) ) ) {
							continue;
						}
                        
                        $text = (!empty($attr['words'])) ? $this->word_limiter($text, $link) : $this->format_text($text ?? '');
                        
                        if ( isset($slideshow) && ! empty($body) && empty($textlong) ) {
                            $textlong = $this->format_text($body ?? '');
                        }
                        
                        $sbi = $this->make_timestr($item->timestamp, $link);
                        $itemdata = array(
                        'title' => $title ?? '',
                        'thumb' => $thumb ?? '',
                        'thumburl' => $url ? $url : $link,
                        'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? ( (isset($item->type) && ($item->type == 'video' || $item->type == 'audio')) ? 'inline' : 'icbox') : '',
                        'text' => $text ?? '',
                        'tags' => $tags ?? '',
                        'url' => $link ?? '',
						'object' => $object ?? '',
                        'date' => $item->date,
                        'play' => $play ?? '',
                        'user' => array(
                            'name' => $blog->name,
                            'title' => $blog->title,
                            'url' => $blog->url,
                            'image' => $protocol.'://api.tumblr.com/v2/blog/'.$blog->name.'.tumblr.com/avatar/64'
                            ),
                        'type' => $keyTypes[$item->type][0],
                        'icon' => array($themedefoption['social_icons'][3] ?? '', $themedefoption['type_icons'][$keyTypes[$item->type][1]] ?? '')
                        );
							if (isset($mediasize) && !empty($mediasize) && ($iframe || isset($slideshow) ) )
								$itemdata['size'] = $mediasize;
							$this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                            if ( isset($slideshow) ) {
                                $itemdata['text'] = $textlong;
                                $itemdata['object'] = $object ?? '';
                                if ($mediasrc)
                                    $itemdata['thumb'] = $mediasrc;
                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                            }
                        }
                    }
                    }
                }
                elseif ( $feed_class == 'delicious' ) {
                    if (isset($feed) && !empty($feed)) {
                    foreach ($feed as $item) {
                        $link = $item->u ?? '';
                        if ( $this->make_remove($link) ) {
                        // tags
                        $tags = '';
                        if ( isset($feedoutput['tags']) && !empty($feedoutput['tags']) ) {
                            $tags = '';
                            if ( isset($item->t) && !empty($item->t) ) {
                                $tags = implode(', ', $item->t);
                            }
                        }
                        
                        $itemN = (isset($item->n) && ! empty($item->n)) ? $item->n : '';

                        if ( !$this->make_filters( array( $itemN ) ) ) {
							continue;
						}
                        
                        $sbi = $this->make_timestr($item->dt, $link);
                        $itemdata = array(
                        'title' => '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($item->d) : $item->d) . '</a>',
                        'text' => (!empty($attr['words'])) ? $this->word_limiter($itemN , $link) : $this->format_text($itemN ),
                        'tags' => $tags,
                        'url' => $link,
                        'date' => $item->dt,
                        'user' => array('name' => $item->a),
                        'type' => 'pencil',
                        'icon' => array($themedefoption['social_icons'][4] ?? '', $themedefoption['type_icons'][0] ?? '')
                        );
                        $this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                            if ( isset($slideshow) ) {
                                $itemdata['text'] = $this->format_text($itemN);
                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                            }
                        }
                    }
                    }
                }
                elseif ( $feed_class == 'pinterest' ) {
                    $iframe = $this->sboption['section_pinterest']['pinterest_iframe'] ?? '';
                    
                    $fcount = $ikey = 0;
                    $pinuser = $feed[0]->data->user ?? '';
                    $pinuser_url = $pinuser->profile_url ?? '';
					$pinuser_image = "";
					if(isset($pinuser->image_small_url) && ! empty($pinuser->image_small_url)){
						$pinuser_image = str_replace('_30.', '_140.', $pinuser->image_small_url );
					}

					foreach ($feed[0]->data->pins as $pinitem) {
						$pins[$pinitem->id] = $pinitem;
					}

                    if (isset($feed[1]->channel->item) && $items = $feed[1]->channel->item)
                    foreach ($items as $item) {
                        if ( isset($item->guid) ) {
							$id = substr($item->guid, strrpos($item->guid, 'pin') + 4, 18);
						}
						$pin = $pins[$id];
						$link = $item->link ?? '';
                        if ( $this->make_remove($link) ) {
                        $fcount++;
        
                        $cats = array();
                        if (isset($item->category) && !empty($item->category)) {
                            foreach($item->category as $category) {
                                $cats[] = (string) $category;
                            }
                        }
                        
						$description = $pin->description ?? '';

                        // Fix the links in description
                        $pattern = "/(?<=href=(\"|'))[^\"']+(?=(\"|'))/";
                        if (preg_match($pattern, $description, $url1)) {
                            $description = preg_replace($pattern, "https://www.pinterest.com$url1[0]", $description);
                        }

                        // Find img
                        $mediasrc = $mediasize = $thumb = '';
                        $image_width = $this->sboption['section_pinterest']['pinterest_image_width'] ?? 237;
                        if ( isset($pin->images->{'237x'}) ) {
							$thumbobj = $pin->images->{'237x'};
                            $thumb = $thumbobj->url;
                            $bigthumb = str_replace('237x', '564x', $thumb);
                            // r3.9.13 < compatibility fix
							if ($image_width == '736') {
								$image_width == '564';
							}
							// END - r3.9.13 < compatibility fix
							if ($image_width == '564')
                                $thumb = $bigthumb;
                            if ($iframe) {
                                $mediasrc = $bigthumb;
                                $newwidth = 450;
                                $newheight = number_format( ($newwidth * $thumbobj->height) / $thumbobj->width, 0, '.', '');
                                $mediasize = $newwidth.','.$newheight;
                            }
                        }
                        
                        if (!empty($pin->is_video) && !empty($pin->embed) && $iframe) {
                            $mediasrc = $pin->embed->src ?? '';
                            $mediasize = (!empty($pin->embed->width) && !empty($pin->embed->height)) ? $pin->embed->width.','.$pin->embed->height : $newwidth.','.$newheight;
                        }

                        // add meta
						$meta = array();
                        if (isset($feedoutput['stat']) && !empty($feedoutput['stat'])) {
                            $meta['likes']['total'] = [
                                'count' => $pin->like_count ?? '',
                                'class' => 'thumbs-up'
                            ];
                            $meta['repin']['total'] = [
                                'count' => $pin->repin_count ?? '',
                                'class' => 'star-o'
                            ];
                        }

                        if ( !$this->make_filters( array( $description ) ) ) {
							continue;
						}
                        
                        $sbi = $this->make_timestr($item->pubDate, $link);
                        $itemdata = array(
                        'title' => '<a href="' . ($pin->link ?? '') . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($item->title) : $item->title) . '</a>',
                        'text' => (string) (!empty($attr['words'])) ? $this->word_limiter($description, $link) : $this->format_text($description),
                        'thumb' => $thumb,
                        'thumburl' => $mediasrc ? $mediasrc : $link,
                        'tags' => isset($cats) && is_array($cats) ? implode(', ', $cats) : "",
                        'url' => $link,
                        'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? (!empty($pin->is_video) ? 'iframe' : 'icbox') : '',
                        'date' => $item->pubDate,
                        'meta' => $meta ?? '',
                        'user' => array(
                            'name' => $pinuser->full_name ?? '',
                            'url' => $pinuser_url ?? '',
                            'image' => $pinuser_image ?? ''
                            ),
                        'type' => 'pencil',
                        'play' => (!empty($pin->is_video)) ? true : false,
                        'icon' => array($themedefoption['social_icons'][5] ?? '', $themedefoption['type_icons'][0] ?? '')
                        );

							if (!empty($mediasize) && ($iframe || isset($slideshow) ) )
                                $itemdata['size'] = $mediasize;

							$this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                            if ( isset($slideshow) ) {
                                $itemdata['text'] = $this->format_text($description);
                                if ($mediasrc)
                                    $itemdata['thumb'] = $mediasrc;
                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                            }
                            
							if ( $fcount >= $results ) break;
                        }
                    }
                }
                elseif ( $feed_class == 'flickr' ) {
                    $feed = ($i == 3) ? ($feed->photoset ?? '') : ($feed->photos ?? '');
                    
                    // flickr next page
                    $loadcrawl[$feed_class.$i.$ifeed] = !empty($feed->page) ? $feed->page+1 : 1;
                    
                    if (isset($feed->photo) && !empty($feed->photo)) {
                        $iframe = $this->sboption['section_flickr']['flickr_iframe'] ?? '';
                        
                        foreach ($feed->photo as $media) {
                            $owner = ($i == 3) ? ($feed->owner ?? '') : $media->owner;
                            $link = 'https://flickr.com/photos/' . $owner . '/' . $media->id;
                            if ( $this->make_remove($link) ) {
                                $text = $image = $url = $mediasize = '';
                                
                                // tags
                                $tags = '';
                                if ( isset($feedoutput['tags']) && !empty($feedoutput['tags']) ) {
                                    if ( isset($media->tags) && !empty($media->tags) ) {
                                        $tags = $media->tags;
                                    }
                                }

								if ( !$this->make_filters( array( $media->title ) ) ) {
									continue;
								}
                                
                                if (isset($attr['carousel']) && !empty($attr['carousel']))
                                    $text = (!empty($attr['words'])) ? $this->word_limiter($media->title, $link) : $media->title;
                                
                                $flickr_thumb = (!empty($this->sboption['section_flickr']['flickr_thumb'])) ? '_'.$this->sboption['section_flickr']['flickr_thumb'] : '';
                                $image = $protocol.'://farm' . $media->farm . '.staticflickr.com/' . $media->server . '/' . $media->id . '_' . $media->secret . $flickr_thumb . '.jpg';
                                $author_icon = (isset($media->iconserver) && $media->iconserver > 0) ? $protocol.'://farm' . $media->iconfarm . '.staticflickr.com/' . $media->iconserver . '/buddyicons/' . $owner . '.jpg' : 'https://www.flickr.com/images/buddyicon.gif';
                                if ($iframe) {
                                    $url = $protocol.'://farm' . $media->farm . '.staticflickr.com/' . $media->server . '/' . $media->id . '_' . $media->secret . '_c.jpg';
                                    $mediasize = (!empty($media->width_c) && !empty($media->height_c)) ? $media->width_c.','.$media->height_c : '';
                                }
                                
                                $mediadate = (!empty($media->dateadded)) ? $media->dateadded : $media->dateupload;
                                
                                $sbi = $this->make_timestr($mediadate, $link);
                                $itemdata = array(
                                    'thumb' => $image,
                                    'thumburl' => $url ? $url : $link,
                                    'title' => '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($media->title) : $media->title) . '</a>',
                                    'text' => $text,
                                    'tags' => $tags,
                                    'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? 'icbox' : '',
                                    'url' => $link,
                                    'date' => $media->datetaken,
                                    'user' => array(
                                        'name' => $media->ownername ?? '',
                                        'url' => $protocol.'://www.flickr.com/people/' . $owner . '/',
                                        'image' => $author_icon
                                        ),
                                    'type' => 'image',
                                    'icon' => array($themedefoption['social_icons'][6] ?? '', $themedefoption['type_icons'][5] ?? '')
                                );
                                $this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                                if ( isset($slideshow) ) {
                                    $itemdata['text'] = $media->title;
                                    $itemdata['size'] = $mediasize ?? '';
                                    if ($url)
                                        $itemdata['thumb'] = $url;
                                    $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                                }
                            }
                        }
                    }
                }
                // Instagram
                elseif ( $feed_class == 'instagram' ) {
                    $keyTypes = array(
                        'image' => array('camera', 5),
                        'video' => array('video-camera', 6),
                        'VIDEO' => array('video-camera', 6),
                        'carousel' => array('camera', 5)
                    );
                    $iframe = $this->sboption['section_instagram']['instagram_iframe'] ?? '';
                    
                    // Instagram next page
                    if ($i != 4) {
                        $next_id = isset($feed->pagination->next_max_id) ? $feed->pagination->next_max_id : "";
                        $loadcrawl[$feed_class.$i.$ifeed] = isset($next_id) ? $next_id : "";
                    }

                    // loop in feed items
                    if ( ! empty($feed->data) ) {
                        foreach ($feed->data as $item) {
							$link = $url = isset($item->link) ? $item->link : "";
                            if ( $this->make_remove($link) ) {
								$thumb = $mediasrc = $mediasize = '';
								$instagram_images = $this->sboption['section_instagram']['instagram_images'] ?? 'low_resolution';
								if (isset($item->images) && !empty($item->images)) {
									$thumb = $item->images->{$instagram_images}->url ?? ($item->images->thumbnail->url ?? '');
									
									// set iframe image
									$instagram_images_iframe = 'high_resolution';
									if ($iframe) {
										$itemimages = $item->images->{$instagram_images_iframe} ?? '';
										$url = $mediasrc = !empty($itemimages->url) ? $itemimages->url : ($item->images->standard_resolution->url ?? '');
										if (!empty($itemimages->width) && $itemimages->height)
											$mediasize = $itemimages->width.','.$itemimages->height;
									}
								}

								if (isset($item->type) && in_array( $item->type, array('video', 'VIDEO') ) && $iframe) {
									$instagram_videos = $this->sboption['section_instagram']['instagram_videos'] ?? 'low_resolution';
									$itemvideos = $item->videos->{$instagram_videos} ?? '';
									if ( strlen($link) <= 40 || (isset($item->product_type) && $item->product_type == "IGTV")
										|| strpos($link, '/tv/') !== false || strpos($link, '/reel/') !== false) {
										// Some reels having media_url
										if (isset($itemvideos->url) && !empty($itemvideos->url)) {
											$url = $mediasrc = $itemvideos->url;
										} else {
											$url = $mediasrc = rtrim($link, '/\\') . '/embed/';
										}
									} else {
										// embed does not work for non-private posts
										if (isset($itemvideos->url) && !empty($itemvideos->url)) {
											$url = $mediasrc = $itemvideos->url;
										}
									}
									if (!empty($itemvideos->width) && $itemvideos->height) {
										$mediasize = $itemvideos->width.','.($itemvideos->height + 65);
									} else {
										if ( ! empty($item->height) && ! empty($item->width) ) {
											$mediasize = $item->width.','.($item->height + 65);
										} else {
											$mediasize = '640,640';
										}
									}
								}

								// tags
								$tags = '';
								if ( isset($feedoutput['tags']) && !empty($feedoutput['tags']) ) {
									if ( isset($item->tags) && !empty($item->tags) ) {
										$tags = implode(', ', $item->tags);
									}
								}
								
								$meta = array();

								// comments
								if (isset($item->comments->count) && !empty($item->comments->count)) {
									if ( ! empty($item->comments->data) ) {
										$meta['comments']['data'] = $this->instagram_parse_comments($item->comments->data, $link);
									} elseif ($i == 1) {
										$meta['comments']['data'] = 'fetch';
									}
								}
								$meta['comments']['total'] = [
									'count' => $item->comments->count ?? 0
								];

								// likes
								$count = 0;
								$meta['likes']['data'] = '';
								$likes_count = ( isset($this->sboption['section_instagram']['instagram_likes']) && $this->sboption['section_instagram']['instagram_likes'] > 0 ) ? $this->sboption['section_instagram']['instagram_likes'] : 0;
								if ( ! empty($item->likes->data) && $likes_count ) {
									foreach ( $item->likes->data as $like ) {
										$count++;
										$meta['likes']['data'] .= '<img class="sb-img" src="' . $like->profile_picture . '" title="' . $like->full_name . '" alt="' . $like->full_name . '">';
										if ( $count >= $likes_count ) break;
									}
								}
								$meta['likes']['total'] = [
									'count' => $item->likes->count ?? 0
								];

								if ( !$this->make_filters( array( $item->caption->text ) ) ) {
									continue;
								}
                            
								$textlong = '';
								$text = (!empty($attr['words'])) ? $this->word_limiter($item->caption->text , $link) : $this->format_text($item->caption->text );
								if ( isset($slideshow) ) {
									$textlong = $this->format_text($item->caption->text );
								}
                            
								// Add links to all hash tags
								if ( isset($item->tags) && !empty($item->tags) ) {
									$text = $this->add_links('instagram', $text);

									if ( isset($slideshow) ) {
										$textlong = $this->add_links('instagram', $textlong);
									}
								}
								
								// create item
								$sbi = $this->make_timestr($item->created_time, $link);
								$itemdata = array(
									'id' => $item->id ?? '',
									'thumb' => $thumb ?? '',
									'thumburl' => $url ? $url : $link,
									'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? (isset($item->type) && in_array($item->type, array('video','VIDEO')) ? 'iframe' : 'icbox') : '',
									'text' => $text,
									'meta' => $meta ?? '',
									'tags' => $tags,
									'url' => $link,
									'date' => $item->created_time,
									'user' => array(
										'name' => $item->user->username ?? '',
										'title' => $item->user->full_name ?? '',
										'url' => (isset($item->user->username) && ! empty($item->user->username)) ? 'https://instagram.com/'.$item->user->username .'/' : 'https://instagram.com/',
										'image' => $item->user->profile_picture ?? ''
									),
									'type' => isset($keyTypes[$item->type][0]) ? $keyTypes[$item->type][0] : 'camera',
									'play' => (isset($item->type) && in_array($item->type, array('video','VIDEO'))) ? true : false,
									'icon' => array("", "")
								);

								if (isset($themedefoption['social_icons'][7]) && ! empty($themedefoption['social_icons'][7])) {
									$itemdata["icon"][0] = $themedefoption['social_icons'][7];
								}
								if (isset($keyTypes[$item->type][1]) && isset($themedefoption['type_icons'][$keyTypes[$item->type][1]]) && ! empty($themedefoption['type_icons'][$keyTypes[$item->type][1]])) {
									$itemdata["icon"][1] = $themedefoption['type_icons'][$keyTypes[$item->type][1]];
								}
								
								if ( !empty($item->sidecar_media) ) {
									if ( is_array($item->sidecar_media) ) {
										$itemdata["child"]["medias"] = $item->sidecar_media;
									}
								}
                                if (isset($mediasize) && !empty($mediasize) && ($iframe || isset($slideshow) ) ) {
                                    $itemdata['size'] = $mediasize;
								}

                                $this->final[$sbi] = array( $feed_class, $itemdata, $attr, $sbi, $i, $ifeed );
                                if ( isset($slideshow) ) {
                                    $itemdata['text'] = $textlong;
                                    if ($mediasrc) {
                                        $itemdata['thumb'] = $mediasrc;
									}
                                    $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                                }
                            }
                        } // end foreach
                    
                        // next page timestamp only for /media/search
                        if ($i == 4 && empty($next_id)) {
                            $loadcrawl[$feed_class.$i.$ifeed] = $item->created_time ?? '';
						}
                        // next page get last id only for hashtag search
                        if ($i == 2 && empty($next_id)) {
                            $loadcrawl[$feed_class.$i.$ifeed] = isset($item->id) ? $item->id : "";
						}
                    }
                }
				elseif ($feed_class == 'tiktok') {
					$iframe = $this->sboption['section_tiktok']['tiktok_iframe'] ?? '';
					if ( !empty($feed) && isset($feed->data) && !empty($feed->data) ) {
						if (isset($feed->cursore) && !empty($feed->cursore)) {
							$loadcrawl[$feed_class.$i.$ifeed] = $feed->cursore;
						}
						foreach ($feed->data as $item) {
							$link = $item->share_url;
							if ( $this->make_remove($link) ) {
								$title = !empty($item->title) ? '<a href="' . $link . '"'.$target.'>' . (isset($attr['titles']) && ! empty($attr['titles']) ? $this->title_limiter($item->title) : $item->title) . '</a>' : '';
								
								$text = '';
								if (isset($item->video_description) && !empty($item->video_description)) {
									$text = (isset($attr['words']) && !empty($attr['words'])) ? $this->word_limiter($item->video_description, $link) : $this->format_text($item->video_description);
								}

								$author_link = '';
								if (preg_match('/https:\/\/www.tiktok.com\/@.*?\//', $link, $match, PREG_OFFSET_CAPTURE, 0)) {
									if (!empty($match[0][0])) {
										$author_link = $match[0][0];
									}
								}

								$meta = array();
								$meta['comments']['total'] = [
									'count' => $item->comment_count,
									'class' => 'comment'
								];
								$meta['likes']['total'] = [
									'count' => $item->like_count,
									'class' => 'thumbs-up'
								];
								$meta['shares']['total'] = [
									'count' => $item->share_count,
									'class' => 'retweet'
								];
                                $meta['views']['total'] = [
                                    'count' => $item->view_count,
                                    'class' => 'play-circle'
                                ];

								$sbi = $this->make_timestr($item->create_time, $link);
								$itemdata = array(
									'id'       => isset( $item->id ) ? $item->id : '',
									'thumb'    => isset( $item->cover_image_url ) ? $item->cover_image_url : '',
									'thumburl' => $item->embed_link,
									'iframe'   => $iframe ? 'iframe' : '',
									'title'	   => $title,
									'text'     => $text,
									'meta'     => $meta,
									'tags'     => '',
									'url'      => $link,
									'date'     => $item->create_time,
									'user'     => array(
										'name'  => isset( $feed->user->display_name ) ? $feed->user->display_name : '',
										'url'   => $author_link,
										'image' => isset( $feed->user->avatar ) ? $feed->user->avatar : ''
									),
									'type'     => 'video-camera',
									'play'     => true,
									'size' => "{$item->width},{$item->height}",
									'icon'     => array(
										isset( $themedefoption['social_icons'][17] ) ? $themedefoption['social_icons'][17] : '',
										isset( $themedefoption['type_icons'][17] ) ? $themedefoption['type_icons'][17] : ''
									)
								);
								
								$this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
								if ( isset($slideshow) ) {
									$itemdata['text'] = $title;
									$this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
								}
							}
						}
					}
				}
                elseif ( $feed_class == 'youtube' ) {
                    $iframe = $this->sboption['section_youtube']['youtube_iframe'] ?? '';

                    // youtube next page
                    $loadcrawl[$feed_class.$i.$ifeed] = $feed->nextPageToken ?? '';
                    
                    if (isset($feed->items) && !empty($feed->items))
                    foreach ($feed->items as $item) {
                        $watchID = ($i == 3) ? ($item->id->videoId ?? '') : ($item->snippet->resourceId->videoId ?? '');
                        $link = $protocol.'://www.youtube.com/watch?v='.$watchID;
                        $snippet = $item->snippet;
                        if ( $this->make_remove($link) ) {
                        $dateof = $item->contentDetails->videoPublishedAt ?? '';
                        $title = $snippet->title ?? '';
                        $text = $snippet->description ?? '';

						// Remove private videos of the playlist
						if ( isset($item->status->privacyStatus) ) {
							if ( $item->status->privacyStatus !== 'public' ) {
								continue;
							}
						}

						if ( !$this->make_filters( array( $text ) ) ) {
							continue;
						}
                        
                        $text = (isset($attr['words']) && !empty($attr['words'])) ? $this->word_limiter($text , $link) : $this->format_text($text );

                        $thumb = $mediasrc = $mediasize = '';
                        if ($iframe) {
                            $mediasrc = $protocol.'://www.youtube.com/embed/' . $watchID . '?rel=0&wmode=transparent';
                        }
                        $thumbnail = $snippet->thumbnails->{$this->sboption['section_youtube']['youtube_thumb']} ?? '';
                        if ( ! $thumbnail )
                            $thumbnail = $snippet->thumbnails->{'medium'} ?? '';
                        $thumb = $thumbnail->url ?? '';

                        // comments
                        $meta = array();
                        $meta['comments']['data'] = 'fetch';
                        
						// user info
						$userdata = array(
							'name' => $snippet->channelTitle,
							'url' => 'https://www.youtube.com/channel/'.$snippet->channelId
						);
						if (isset($feed->userInfo->thumbnails) && !empty($feed->userInfo->thumbnails))
							$userdata['image'] = $feed->userInfo->thumbnails->default->url ?? '';
						
						
                        $sbi = $this->make_timestr($dateof, $link);
                        $itemdata = array(
                            'id' => $watchID,
                            'thumb' => $thumb,
							'thumburl' => $mediasrc ? $mediasrc : $link,
							'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? 'iframe' : '',
							'title' => '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($title) : $title) . '</a>',
							'text' => $text,
							'url' => $link,
							'date' => $dateof,
                            'user' => $userdata,
                            'meta' => $meta ?? '',
							'type' => 'youtube-play',
							'play' => true,
							'icon' => array($themedefoption['social_icons'][8] ?? '', $themedefoption['type_icons'][6] ?? '')
                        );

                        $youtube_video = $this->sboption['section_youtube']['youtube_video'] ?? '640-360';
                        $ytvsize = explode('-', $youtube_video);
                        $mediasize = "$ytvsize[0],".($ytvsize[1] + 100);
							if (isset($mediasize) && !empty($mediasize) && ($iframe || isset($slideshow) ) )
								$itemdata['size'] = $mediasize;
							$this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                            if ( isset($slideshow) ) {
                                $itemdata['text'] = $this->format_text($snippet->description );
                                if ($mediasrc)
                                    $itemdata['thumb'] = $mediasrc;
                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                            }
                        }
                    }
                }
                elseif ( $feed_class == 'vimeo' ) {
                    $iframe = $this->sboption['section_vimeo']['vimeo_iframe'] ?? '';
                    
                    if (isset($feed) && !empty($feed)) {
                        // vimeo next page
                        $loadcrawl[$feed_class.$i.$ifeed] = !empty($feed->page) ? $feed->page+1 : 1;
                        
                        $vimeo_thumb = $this->sboption['section_vimeo']['vimeo_thumb'] ?? '295';

                        if (isset($feed->data) && $data = $feed->data)
                        foreach ($data as $item) {
                            $link = $item->link ?? '';
                            if ( $this->make_remove($link) ) {
                                $thumb = $mediasrc = $mediasize = '';
                                if (isset($item->pictures->sizes) && $pictures = $item->pictures->sizes) {
                                    foreach ($pictures as $photo) {
                                        if ($photo->width == $vimeo_thumb) {
                                            $thumb = $photo->link;
                                            break;
                                        }
                                    }
                                }
                                
                                $title = $item->name;
                                $id = preg_replace('/\D/', '', $item->uri);
                                if ($iframe || $slideshow) {
                                    $url = $mediasrc = 'https://player.vimeo.com/video/'. $id;
                                    $mediasize = $item->width.','.$item->height;
                                } else {
                                    $url = $link;
                                }
                                
                                $datetime = $item->created_time ?? ($item->modified_time ?? '');
                                $connections = $item->metadata->connections ?? '';

                                $meta = array();
                                $meta['likes']['total'] = [
                                    'count' => $connections->likes->total ?? 0,
                                    'class' => 'thumbs-up'
                                ];
                                $meta['comments']['total'] = [
                                    'count' => $connections->comments->total ?? 0,
                                    'class' => 'comment'
                                ];
                                $meta['views']['total'] = [
                                    'count' => $item->stats->plays ?? 0,
                                    'class' => 'play-circle'
                                ];
                                $meta['secs']['total'] = [
                                    'count' => $item->duration ?? 0,
                                    'class' => 'clock-o'
                                ];

                                $user_name = $item->user->name ?? '';
                                $user_url = $item->user->link ?? '';
                                $user_image = $item->user->pictures->sizes[1]->link ?? '';

								if ( !$this->make_filters( array( $item->description ) ) ) {
									continue;
								}
                                
                                $sbi = $this->make_timestr($datetime, $link);
                                $itemdata = array(
                                'thumb' => $thumb ?? '',
                                'thumburl' => $url ? $url : $link,
                                'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? 'iframe' : '',
                                'title' => '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($title) : $title) . '</a>',
                                'text' => (!empty($attr['words'])) ? $this->word_limiter($item->description, $link) : $item->description,
                                'meta' => (!empty($feedoutput['share'])) ? $meta : null,
                                'url' => $link,
                                'date' => $datetime,
                                'user' => array(
                                    'name' => $user_name,
                                    'url' => $user_url,
                                    'image' => $user_image
                                    ),
                                'type' => 'video-camera',
                                'play' => true,
                                'icon' => array($themedefoption['social_icons'][9] ?? '', $themedefoption['type_icons'][6] ?? '')
                                );
								if (isset($mediasize) && !empty($mediasize) && ($iframe || isset($slideshow) ) )
									$itemdata['size'] = $mediasize;
                                $this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                                if ( isset($slideshow) ) {
                                    $itemdata['text'] = $item->description;
                                    $itemdata['thumb'] = $mediasrc;
                                    $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                                }
                            } // end if $link
                        } // end foreach $data
                    }
                }
                elseif ( $feed_class == 'deviantart' ) {
                    $fcount = 0;
                    $channel = $feed->channel ?? '';
                    if (isset($channel->item) && $channel->item)
                    foreach($channel->item as $item) {
                        $link = $item->link ?? '';
                        if ( $this->make_remove($link) ) {
                        $fcount++;

                        $description = $item->children('media', true)->description;
                        if (isset($item->children('media', true)->thumbnail->{1}) && $thumbObj = $item->children('media', true)->thumbnail->{1})
                            $thumb = $thumbObj->attributes()->url ?? '';

                        if ( !$this->make_filters( array( $description ) ) ) {
							continue;
						}
                        
                        $sbi = $this->make_timestr($item->pubDate, $link);
                        $itemdata = array(
                        'thumb' => $thumb ?? '',
                        'title' => '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($item->title) : $item->title) . '</a>',
                        'text' => (!empty($attr['words'])) ? $this->word_limiter($description, $link) : $description,
                        'tags' => '<a href="' . $item->children('media', true)->category . '"'.$target.'>' . $item->children('media', true)->category->attributes()->label . '</a>',
                        'url' => $link,
                        'date' => $item->pubDate,
                        'user' => array(
                            'name' => $item->children('media', true)->credit->{0},
                            'url' => $item->children('media', true)->copyright->attributes()->url,
                            'image' => $item->children('media', true)->credit->{1}),
                        'type' => 'image',
                        'icon' => array($themedefoption['social_icons'][11] ?? '', $themedefoption['type_icons'][4] ?? '')
                        );
                        $this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                        if ( isset($slideshow) ) {
                            $itemdata['text'] = $description;
                            $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                        }
                        
                        if ( $fcount >= $results ) break;
                        }
                    }
                }
                elseif ( $feed_class == 'rss' ) {
                    $iframe = $this->sboption['section_rss']['rss_iframe'] ?? '';
                    // RSS next page
					$startP = null;
					if ( isset($_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed])
						&& !empty($_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed]) ) {
						$startP = $_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed];
					}
                    $loadcrawl[$feed_class.$i.$ifeed] = $nextP = $startP + $results;
                    $fcount = 0;
                    $MIMETypes = array('image/jpeg', 'image/jpg', 'image/gif', 'image/png');
                    if ( isset($feed->channel) && $channel = $feed->channel ) { // rss
                        if (isset($channel->item) && !empty($channel->item))
                        foreach($channel->item as $item) {
                            $link = $item->link ?? '';
                            if ( $this->make_remove($link) ) {
                            	$fcount++;
                            
                            if ($fcount > $startP) {

                            $thumb = $url = '';
                            if (!empty($item->children('media', true)->thumbnail))
                            foreach($item->children('media', true)->thumbnail as $thumbnail) {
                                $thumb = $thumbnail->attributes()->url ?? '';
                            }
                            if ( ! $thumb && !empty($item->children('media', true)->content)) {
                                foreach($item->children('media', true)->content as $content) {
                                    $thumb = $content->children('media', true)->thumbnail->attributes()->url ?? '';
                                    if ( isset($content->attributes()->type) && isset($MIMETypes) && in_array($content->attributes()->type, (array) $MIMETypes) )
                                        $url = $content->attributes()->url ?? '';
                                }
                                if ( ! $thumb && $url) {
                                    $thumb = $url;
                                }
                            }
                            
                            if ( ! $thumb) {
                                if ( isset($item->enclosure) && isset($item->enclosure->attributes()->type)
									&& isset($MIMETypes) && in_array($item->enclosure->attributes()->type, (array) $MIMETypes) ) {
                                    $thumb = $item->enclosure->attributes()->url ?? '';
								}
                            }
                            
                            if ( !empty($item->category) && !empty($feedoutput['tags']) ) {
								foreach($item->category as $category) {
									$cats[] = (string) $category;
								}
							}
                            
                            // set Snippet or Full Text
                            $text = $description = '';
                            if (isset($this->sboption['section_rss']['rss_text']) && !empty($this->sboption['section_rss']['rss_text']))
                                $description = $item->description ?? '';
                            else
                                $description = $item->children("content", true)->encoded ?? ($item->description ?? '');
                                
                            if (isset($description) && !empty($description)) {
                                $description = preg_replace("/<script.*?\/script>/s", "", $description);
                                if (isset($attr['words']) && !empty($attr['words'])) {
                                    if ( ! $thumb) {
                                        $thumbarr = axs_getsrc($description);
										$thumb = isset($thumbarr['src']) ? $thumbarr['src'] : '';
                                    }
                                    $text = $this->word_limiter($description, $link);
                                } else {
                                    $text = $description;
                                }
                            }
                            if ($iframe) {
                                if ( ! $url)
                                    $url = $thumb ?? '';
                            }
                            
                            // resize thumbnails
                            $image_width = $this->sboption['section_rss']['rss_image_width'] ?? '';
                            if (!empty($thumb) && ! empty($image_width) ) {
                            	$token = md5(urlencode($thumb).($_SERVER['SERVER_ADDR'] ?? '').($_SERVER['SERVER_ADMIN'] ?? '').($_SERVER['SERVER_NAME'] ?? '').($_SERVER['SERVER_PORT'] ?? '').($_SERVER['SERVER_PROTOCOL'] ?? '').strip_tags($_SERVER['SERVER_SIGNATURE'] ?? '').($_SERVER['SERVER_SOFTWARE'] ?? '').($_SERVER['DOCUMENT_ROOT'] ?? ''));
                            	$imgStr = 'resize='.$image_width.'&refresh=3600&token='.$token.'&src='.$thumb;
                            	$thumb = esc_url( add_query_arg( 'sbimg', base64_encode($imgStr), site_url('index.php') ) );
                            }

                            if ( !$this->make_filters( array( $description ) ) ) {
								continue;
							}
                            
                            $sbi = $this->make_timestr($item->pubDate, $link);
                            $itemdata = array(
                            'thumb' => $thumb ?? '',
                            'thumburl' => $url ? $url : $link,
                            'title' => '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($item->title) : $item->title) . '</a>',
                            'text' => $text,
                            'tags' => isset($cats) && is_array($cats) ? implode(', ', $cats) : "",
                            'url' => $link,
                            'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? 'icbox' : '',
                            'date' => $item->pubDate,
                            'user' => array(
                                'name' => $channel->title,
                                'url' => $channel->link,
                                'image' => $channel->image->url ?? ''
                                ),
                            'type' => 'pencil',
                            'icon' => array($themedefoption['social_icons'][12] ?? '', $themedefoption['type_icons'][0] ?? '')
                            );
	                            $this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
	                            if ( isset($slideshow) ) {
	                                $itemdata['text'] = $this->format_text($description ?? '');
	                                if ($url)
	                                    $itemdata['thumb'] = $url;
	                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
	                            }
							} // end start point

                            	if ($fcount >= $nextP) break;
                            } // end make remove
                        } // end foreach
                    } elseif ( isset($feed->entry) && $entry = $feed->entry ) { // atom
                        // get feed link
                        if (isset($feed->link) && !empty($feed->link))
                        foreach($feed->link as $link) {
                            if (isset($link->attributes()->rel) && $link->attributes()->rel == 'alternate')
                                $user_url = $link->attributes()->href ?? '';
                        }
                        foreach($feed->entry as $item) {
                            $link = $item->link[0]->attributes()->href ?? '';
                            if ( $this->make_remove($link) ) {
                            	$fcount++;
                            
                            if ($fcount > $startP) {

                            $title = (string) $item->title ?? '';
                            $thumb = $url = '';
                            if (isset($item->media) && !empty($item->media))
                            foreach($item->media as $thumbnail) {
                                $thumb = $thumbnail->attributes()->url ?? '';
                            }
                            if ( ! $thumb && !empty($item->link)) {
                                foreach($item->link as $linkitem) {
                                    if (isset($linkitem->attributes()->rel) && $linkitem->attributes()->rel == 'enclosure') {
                                        if ( isset($linkitem->attributes()->type) && in_array($linkitem->attributes()->type, $MIMETypes) )
                                            $thumb = $content->attributes()->url ?? '';
                                    }
                                }
                            }
                            
                            $cats = '';
                            if (!empty($item->category) && !empty($feedoutput['tags'])) {
                                foreach($item->category as $category) {
                                    $cats .= ($category->attributes()->term ?? '') .', ';
                                }
                                $cats = rtrim($cats, ", ");
                            }

                            // set Snippet or Full Text
                            $text = $description = '';
                            if (isset($this->sboption['section_rss']['rss_text']) && !empty($this->sboption['section_rss']['rss_text'])) {
                                $description = (string) $item->summary ?? '';
                            } else {
                                $content = (string) $item->content ?? '';
                                $description = ($content) ? $content : (string) $item->summary ?? '';
                            }

                            if (isset($description) && !empty($description)) {
                                if (isset($attr['words']) && !empty($attr['words'])) {
                                    if ( ! $thumb) {
                                        $thumbarr = axs_getsrc($description);
                                        $thumb = isset($thumbarr['src']) ? $thumbarr['src'] : '';
                                    }
                                    $text = $this->word_limiter($description, $link);
                                }
                                else {
                                    $text = $description;
                                }
                            }
                            if ($iframe)
                                $url = $thumb ?? '';
                                
                            // resize thumbnails
                            $image_width = $this->sboption['section_rss']['rss_image_width'] ?? '';
                            if (!empty($thumb) && ! empty($image_width) ) {
                            	$token = md5(urlencode($thumb).($_SERVER['SERVER_ADDR'] ?? '').($_SERVER['SERVER_ADMIN'] ?? '').($_SERVER['SERVER_NAME'] ?? '').($_SERVER['SERVER_PORT'] ?? '').($_SERVER['SERVER_PROTOCOL'] ?? '').strip_tags($_SERVER['SERVER_SIGNATURE'] ?? '').($_SERVER['SERVER_SOFTWARE'] ?? '').($_SERVER['DOCUMENT_ROOT'] ?? ''));
                            	$imgStr = 'resize='.$image_width.'&refresh=3600&token='.$token.'&src='.$thumb;
                            	$thumb = esc_url( add_query_arg( 'sbimg', base64_encode($imgStr), site_url('index.php') ) );
                            }

                            if ( !$this->make_filters( array( $description ) ) ) {
								continue;
							}
                            
                            $sbi = $this->make_timestr($item->published, $link);
                            $itemdata = array(
                            'thumb' => $thumb ?? '',
                            'thumburl' => $url ? $url : $link,
                            'title' => '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($title) : $title) . '</a>',
                            'text' => $text ?? '',
                            'tags' => $cats ?? '',
                            'url' => $link,
                            'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? 'icbox' : '',
                            'date' => $item->published,
                            'user' => array(
                                'name' => $feed->title,
                                'url' => $user_url ?? '',
                                'image' => $feed->logo ?? ''
                                ),
                            'type' => 'pencil',
                            'icon' => array($themedefoption['social_icons'][12] ?? '', $themedefoption['type_icons'][0] ?? '')
                            );
                            $this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                            if ( isset($slideshow) ) {
                                $itemdata['text'] = $this->format_text($description  ?? '');
                                if ($url)
                                    $itemdata['thumb'] = $url;
                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                            }
									} // end start point
                            	if ($fcount >= $nextP) break;
                            } // end make remove
                        }
                    }
                }
                elseif ( $feed_class == 'soundcloud' ) {
                    // Next page
                    if ( ! empty($feed->next_href) ) {
						$loadcrawl[$feed_class.$i.$ifeed] = (!empty($_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed])) ? $_SESSION[$label]['loadcrawl'][$feed_class.$i.$ifeed] + $results : $results;
					}

                    if (isset($feed) && !empty($feed->collection))
                    foreach ($feed->collection as $item) {
                        $link = $item->permalink_url ?? '';
                        if ( $this->make_remove($link) ) {
                        // tags
                        $tags = '';
                        if ( isset($feedoutput['tags']) && !empty($feedoutput['tags']) ) {
                            if (isset($item->tag_list) && !empty($item->tag_list))
                                $tags .= $item->tag_list;
                        }
                        
                        // convert duration to mins
                        $duration = '';
                        if (isset($item->duration) && !empty($item->duration)) {
                            $seconds = floor($item->duration / 1000);
                            $duration = floor($seconds / 60);
                        }
                        
                        $download = '';
                        if (isset($item->download_url) && !empty($item->download_url)) {
                            $download .= '<span class="download"><i class="sb-bico sb-cloud-download"></i> <a href="'.$item->download_url.'">' . $item->download_count ?? '' . '</a></span>';
                        }
                        
                        if (isset($item->artwork_url) && !empty($item->artwork_url)) {
                        	$artwork_url = str_replace('-large', '-t300x300', $item->artwork_url);
                        } else {
							$artwork_url = '';
						}

                        $meta = array();
                        $meta['likes']['total'] = [
                            'count' => $item->favoritings_count ?? 0,
                            'class' => 'thumbs-up'
                        ];
                        $meta['views']['total'] = [
                            'count' => $item->playback_count ?? 0,
                            'class' => 'play-circle'
                        ];
                        $meta['comments']['total'] = [
                            'count' => $item->comment_count ?? 0,
                            'class' => 'comment'
                        ];
                        $meta['mins']['total'] = [
                            'count' => $duration ?? 0,
                            'class' => 'clock-o'
                        ];
                        $meta['download']['data'] = $download;

                        if ( !$this->make_filters( array( $item->description ) ) ) {
							continue;
						}
                        
                        $sbi = $this->make_timestr($item->created_at, $link);
                        $itemdata = array(
                        'title' => '<a href="' . $link . '"'.$target.'>' . (!empty($attr['titles']) ? $this->title_limiter($item->title) : $item->title) . '</a>',
                        'text' => (!empty($attr['words'])) ? $this->word_limiter($item->description ?? '', $link) : ($item->description ?? ''),
                        'thumb' => $artwork_url,
                        'tags' => $tags,
                        'url' => $link,
                        'meta' => (!empty($feedoutput['meta'])) ? $meta : null,
                        'date' => $item->created_at,
                        'user' => array(
                            'name' => $item->user->username,
                            'url' => $item->user->permalink_url,
                            'image' => $item->user->avatar_url
                            ),
                        'type' => 'youtube-play',
                        'icon' => array($themedefoption['social_icons'][13] ?? '', $themedefoption['type_icons'][7] ?? '')
                        );
                        $this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                            if ( isset($slideshow) ) {
                                $itemdata['text'] = $item->description ?? '';
                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                            }
                        }
                    }
                }
                elseif ( $feed_class == 'vk' ) {
                    if (isset($feed) && !empty($feed)) {
                        $iframe = $this->sboption['section_vk']['vk_iframe'] ?? '';
                    // vk next page start
                    $offset = $feed->offset ?? '';
                    $loadcrawl[$feed_class.$i.$ifeed] = ($offset == 0) ? $results : $results + $offset;
                    
                    if (isset($feed->response->groups) && $groups = $feed->response->groups) {
                        foreach ($feed->response->groups as $group) {
                            $groupdata['-'.$group->id] = $group;
                        }
                    }
                    if (isset($feed->response->profiles) && $profiles = $feed->response->profiles) {
                        foreach ($feed->response->profiles as $profile) {
                            $userdata[$profile->id] = $profile;
                        }
                    }
                    if (isset($feed->response) && !empty($feed->response))
                    foreach ($feed->response->items as $entry) {
                        $link = $protocol.'://vk.com/wall'.($entry->owner_id ?? '').'_'.($entry->id ?? '');
                        if ( $this->make_remove($link) ) {
                        // body text
                        $text = $entry->text ?? '';
                        if ( ! $text) {
                            if (isset($entry->copy_history) && !empty($entry->copy_history))
                                $text = $entry->copy_history[0]->text;
                        }
                        if ( isset($slideshow) ) {
                            $textlong = $this->format_text($text );
                            $textlong = preg_replace('/#([^\s]+)/', '<a href="'.$protocol.'://vk.com/feed?section=search&q=%23$1"'.$target.'>#$1</a>', $textlong);
                        }

                        if ( !$this->make_filters( array( $text ) ) ) {
							continue;
						}
                        
                        $text = (!empty($attr['words'])) ? $this->word_limiter($text, $link) : $this->format_text($text );
                        // Add links to all hash tags
                        $text = preg_replace('/#([^\s]+)/', '<a href="'.$protocol.'://vk.com/feed?section=search&q=%23$1"'.$target.'>#$1</a>', $text);
                        
                        // user info
                        $user = (!empty($userdata[$entry->from_id])) ? $userdata[$entry->from_id] : $groupdata[$entry->from_id];
                        $user_name = (!empty($user->name)) ? $user->name : $user->first_name.' '.$user->last_name;
                        $user_image = $user->photo_50;
                        $user_url = isset($user->screen_name) ? $protocol.'://vk.com/' . $user->screen_name : $protocol.'://vk.com/id' . $entry->from_id;
                        
                        // get image
                        $image_type = $this->sboption['section_vk']['vk_image_width'] ?? 'x';
                        $attachments = $entry->attachments ?? '';
                        if ( ! $attachments) {
                            if (isset($entry->copy_history) && !empty($entry->copy_history))
                                $attachments = $entry->copy_history[0]->attachments ?? '';
                        }
                        $source = $iframe2 = $play = $url = $mediasrc = $mediasize = '';
                        if ( ! empty($attachments) ) {
                            if ($image_type) {
                                foreach ($attachments as $attach) {
                                    if ($attach->type == 'photo') {
                                    	$get_source = $this->vk_get_photo($attach->photo->sizes,$image_type);
                                    	if(!empty($get_source) && isset($get_source->url)){
                                    		$source = $get_source->url;
											$mediasize = $get_source->width.','.$get_source->height;
                                    		$url = $this->vk_get_photo($attach->photo->sizes,"w")->url;
											break;
										}
                                    } 
                                    elseif ($attach->type == 'link') {
                                    	if(isset($attach->link->photo) && isset($attach->link->photo->sizes)){
											$get_source = $this->vk_get_photo($attach->link->photo->sizes,$image_type);
											if(!empty($get_source) && isset($get_source->url)){
												$source = $get_source->url;
												$mediasize = $get_source->width.','.$get_source->height;
												$url = $this->vk_get_photo($attach->link->photo->sizes,"w")->url;
												break;
											}
										}   
                                    } 
                                    elseif ($attach->type == 'video') {
										if(isset($attach->video->image)){
											$get_source = $this->vk_get_photo($attach->video->image,$image_type,"video");
											if(!empty($get_source) && isset($get_source->url)){
												$source = $get_source->url;
												$mediasize = $get_source->width.','.$get_source->height;
												$url = $this->vk_get_photo($attach->video->image,"w","video")->url;
												$play = true;
												break;
											}
										}
                                    } elseif ($attach->type == 'doc') {
										if(isset($attach->doc->preview) && isset($attach->doc->preview->photo) && isset($attach->doc->preview->photo->sizes)){
											$get_source = $this->vk_get_photo($attach->doc->preview->photo->sizes,$image_type);
											if(!empty($get_source) && isset($get_source->src)){
												$source = $get_source->src;
												$mediasize = $get_source->width.','.$get_source->height;
												$url = $this->vk_get_photo($attach->doc->preview->photo->sizes,"w")->src;
												break;
											}
										}
                                    }
                                }
                            }
                        }
                        
                        if($iframe && !empty($mediasize)){
							$iframe2 = $iframe;
						}
                        
                        $meta = array();
                        $meta['comments']['total'] = [
                            'count' => $entry->comments->count,
                            'class' => 'comment'
                        ];
                        $meta['likes']['total'] = [
                            'count' => $entry->likes->count,
                            'class' => 'thumbs-up'
                        ];
                        $meta['shares']['total'] = [
                            'count' => $entry->reposts->count,
                            'class' => 'retweet'
                        ];
                        
                        $sbi = $this->make_timestr($entry->date, $link);
                        $itemdata = array(
                        'thumb' => $source ?? '',
                        'thumburl' => $url ? $url : $link,
                        'text' => $text ?? '',
                        'meta' => !empty($feedoutput['stat']) ? ($meta ?? '') : '',
                        'url' => $link ?? '',
                        'iframe' => ( !empty($iframe2) && empty($this->attr['full_item_link']) ) ? 'icbox' : '',
                        'date' => $entry->date,
                        'user' => array(
                            'name' => $user_name ?? '',
                            'url' => $user_url,
                            'image' => $user_image ?? ''
                            ),
                        'type' => 'pencil',
                        'play' => $play ?? '',
                        'icon' => array($themedefoption['social_icons'][14] ?? '', ($i == 2) ? ($themedefoption['type_icons'][4] ?? '') : ($themedefoption['type_icons'][0] ?? '') )
                        );
							$this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);
                            if ( isset($slideshow) ) {
                                $itemdata['text'] = $textlong ?? '';
                                $itemdata['size'] = $mediasize ?? '';
                                if ($mediasrc)
                                    $itemdata['thumb'] = $mediasrc;
                                $this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
                            }
                        }
                    } // end foreach
                    } // end $feed
                }
				elseif ( $feed_class == "wordpress" ) {
					$iframe = $this->sboption['section_wordpress']['wordpress_iframe'] ?? '';
					$image_size = $this->sboption['section_wordpress']['wordpress_image_width'] ?? "medium";
					foreach ($feed as $post){
						$link = get_post_permalink($post->ID);
						
						if ( !$this->make_remove($link) ) continue;
						
						$link = get_post_permalink($post->ID);
						$thumb = get_the_post_thumbnail_url($post->ID,$image_size);
						$thumb_url = get_the_post_thumbnail_url($post->ID,"full");
						$title = (!empty($attr['titles'])) ? $this->title_limiter($post->post_title) : $post->post_title;

						if ( !$this->make_filters( array( $post->post_content ) ) ) {
							continue;
						}
						
						$text = (!empty($this->sboption['section_wordpress']['wordpress_text']) && !empty($attr['words'])) ? $this->word_limiter($post->post_content, $link) : $post->post_content;
						$cats = array();
						if(is_array(wp_get_post_categories( $post->ID ))){
							foreach(wp_get_post_categories( $post->ID ) as $c){
								$cat = get_category( $c );
								$cats[] = $cat->name;
							}
						}
						$date = $post->post_date;
						
						$sbi = $this->make_timestr($post->post_date, $link);
						$itemdata = array(
							'thumb' => $thumb ?? "",
							'thumburl' => $thumb_url ? $thumb_url : $link,
							'title' => '<a href="' . $link . '"'.$target.'>' . $title . '</a>',
							'text' => $text,
							'tags' => isset($cats) && is_array($cats) ? implode(', ', $cats) : "",
							'url' => $link,
							'iframe' => ( $iframe && empty($this->attr['full_item_link']) ) ? 'icbox' : '',
                            'date' => $date,
                            'user' => array(
                                'name' => get_the_author_meta( "display_name",$post->post_author ),
                                'url' => get_the_author_meta( "user_url",$post->post_author ),
                                'image' => get_avatar_url( $post->post_author ),
                                ),
							'type' => 'pencil',
							'icon' => array($themedefoption['social_icons'][16] ?? '', $themedefoption['type_icons'][0] ?? '')
						);
						$this->final[$sbi] = array($feed_class, $itemdata, $attr, $sbi, $i, $ifeed);

						if ( isset($slideshow) ) {
							$itemdata['text'] = $this->format_text($post->post_content );
							if(isset($itemdata["thumburl"]) && !empty($itemdata["thumburl"])) $itemdata['thumb'] = $itemdata["thumburl"];
							$this->finalslide[$sbi] = $slidelayoutobj->create_slideitem($feed_class, $itemdata, $sbi, $i, $ifeed);
						}
						
					}
				}
					
                $final = $this->final;
				
				// each network sorting
                if ( ! empty($final) ) {
                    if ( $order == 'date' )
                        krsort($final);
                    elseif ( $order == 'date_asc' )
                        ksort($final);

                    reset($final);
                    $ifeedclass = $feed_class.$i.$ifeed;
                    
                    if ( ! empty($loadmore) ) {
                        // filter last items
                        if ( $lastloaditem = $loadmore[$ifeedclass] ) {
                            $final_keys = array_keys($final);
                            $loadremovefrom = array_search( $lastloaditem, $final_keys );
                            if (empty($loadremovefrom))
                                $loadremovefrom = 0;
                            if ( empty($_SESSION[$label]['loadcrawl'][$ifeedclass])
                            || $final_keys[$loadremovefrom] == $_SESSION[$label]['loadmore'][$ifeedclass] )
                                $loadremovefrom++;
                            $final = array_slice($final, $loadremovefrom);
                        }
                    }

                    $finals[$ifeedclass] = $final;
                    $rankcount[$ifeedclass] = count($final);

                    if (isset($this->sboption['limiting']) && $this->sboption['limiting'] == 'custom') {
                        $fresults[$ifeedclass] = $this->sboption['section_'.$feed_class][$feed_class.'_id_'.$i][$ifeed]['limit'];
                    } else {
                        $ranking[key($final)] = $ifeedclass;
                    }
                }
                $final = $this->final = array();
                
                } // end foreach
                }
            } // end foreach $feeds

            if (!isset($this->sboption['limiting']) || $this->sboption['limiting'] != 'custom') {
                if ( isset($ranking) && !empty($ranking) ) {
                    // defining limits by recent basis
                    krsort($ranking);
                    $rsum = 0;
                    $rnum = count($ranking);
                    for ($i = 1; $i <= $rnum; $i++) {
                        $rsum += $i;
                    }
                    $i = $rnum;
                    foreach ($ranking as $cfeed) {
                        $rank[$cfeed] = number_format( ($i * 100) / $rsum, 0, '.', '');
                        $i--;
                    }
                }
                if ( isset($rankcount) && !empty($rankcount) ) {
                    $maxcountkey = array_search(max($rankcount), $rankcount);
                    foreach ($rankcount as $rkey => $rval) {
                        if (isset($rank[$rkey])) {
							$fresults[$rkey] = number_format($rank[$rkey] * $results / 100, 0, '.', '');
						}
                    }
                    foreach ($rankcount as $rkey => $rval) {
                        if ( isset($fresults[$rkey]) && $fresults[$rkey] > $rval ) {
                            $diffrankcount = $fresults[$rkey] - $rval;
                            $fresults[$rkey] -= $diffrankcount;
                            $fresults[$maxcountkey] += $diffrankcount;
                        }
                    }
                }
            }
            
            if ( isset($finals) && !empty($finals) ) {
                // filnal sorting and adding
                $rescount = 0;
                foreach ($finals as $fkey => $fval) {
                    $fcount = 0;
                    // limit last result
                    foreach ($fval as $key => $val) {
                        $rescount++;
                        $fcount++;
                        $final[$key] = $val;
                        $loadmore[$fkey] = $key;
                        if ( $fcount >= $fresults[$fkey] || $rescount == $results ) break;
                    }
                }

                // set next pages if exist
                if ( array_sum($rankcount) <= $results && ! $is_feed && ( ! $GLOBALS['islive'] || (isset($_REQUEST['action']) && $_REQUEST['action'] == "sb_loadmore") ) ) {
                    foreach ($rankcount as $rkey => $rval) {
                        if (isset($loadcrawl[$rkey]) && !empty($loadcrawl[$rkey]))
                            $_SESSION[$label]['loadcrawl'][$rkey] = $loadcrawl[$rkey];
                        else
                            $_SESSION[$label]['loadcrawl'][$rkey] = null;
                    }
                }
                
                if ( $order == 'date' )
                    krsort($final);
                elseif ( $order == 'date_asc' )
                    ksort($final);
                elseif ( $order == 'random' )
                    $final = axs_shuffle_assoc($final);
                    
                // get related ads
                $args = array(
                    'meta_key' => 'board_id',
                    'meta_value' => $attr['id'],
                    'post_type' => 'sb_ads',
                    'posts_per_page' => 999999
                );
                $adposts = get_posts($args);
                foreach ($adposts as $adpost) {
                    $post_id = $adpost->ID;
                    $adData = array();
                    foreach( ( array ) get_post_custom_keys( $post_id ) as $sKey ) {
                        $adData[ $sKey ] = get_post_meta( $post_id, $sKey, true );
                    }
                    $ad_position = (!empty($adData['ad_position'])) ? $adData['ad_position'] : 0;
                    $ads[$ad_position][] = $adData;
                }

                // Output board items
                if (isset($attr['loadmore']) && !empty($attr['loadmore']) && ! $ajax_feed) {
                    $_SESSION[$label]['loadcount'] = 0;
                }
                
                $display_ads = false;
                if (isset($attr['display_ads']) && !empty($attr['display_ads'])) {
                    if (isset($attr['display_ads'][$type]) && !empty($attr['display_ads'][$type])) {
                        $display_ads = true;
                    } else {
                        if (!empty($attr['carousel']) && !empty($attr['display_ads']['carousel']))
                            $display_ads = true;
                    }
                }
                
                $i = (!empty($_SESSION[$label]['loadcount'])) ? $_SESSION[$label]['loadcount'] : 0;
                foreach ($final as $key => $arrval) {
                    if ( ! empty($ads[$i]) && $display_ads) {
                        foreach ($ads[$i] as $ad) {
                            $adstyle = '';
                            if (!empty($ad['ad_height']) && empty($attr['carousel']) && ! $is_grid) {
                                $adstyle .= 'height: '.$ad['ad_height'].'px;';
                            }
                            if ( isset($ad['ad_border_size']) ) {
                                $adstyle .= 'border-style: solid;';
                                $adstyle .= 'border-width: '.$ad['ad_border_size'].'px;';
                            }
                            if (!empty($ad['ad_border_color']) && !empty($ad['ad_border_size']))
                                $adstyle .= 'border-color: '.$ad['ad_border_color'].';';
                            if (isset($ad['ad_background_color']) && !empty($ad['ad_background_color']))
                                $adstyle .= 'background-color: '.$ad['ad_background_color'].';';
                            if (isset($ad['ad_text_align']) && !empty($ad['ad_text_align']))
                                $adstyle .= 'text-align: '.$ad['ad_text_align'].';';
                            if ($ad['ad_type'] == 'image')
                                $adstyle .= 'line-height: 0;';
                            $adgrid = (!empty($ad['ad_grid_size'])) ? ($ad['ad_grid_size'] == 'solo' ? '' : ' sb-'.$ad['ad_grid_size']) : '';
                            // get ad content
                            $adcontent = '';
                            switch ($ad['ad_type']) {
                                case "text":
                                    $adcontent = nl2br($ad['ad_text']);
                                break;
                                case "code":
                                    $adcontent = $ad['ad_custom_code'];
                                break;
                                case "image":
                                    if (isset($ad['ad_text_align']) && !empty($ad['ad_text_align'])) {
                                        if ($ad['ad_text_align'] == 'left')
                                            $admargin = 'margin-right: auto;';
                                        elseif ($ad['ad_text_align'] == 'right')
                                            $admargin = 'margin-left: auto;';
                                        else
                                            $admargin = 'margin: auto;';
                                    }
                                    if ( ! empty($ad['ad_image']) ) {
                                        $ad_link_target = (isset($ad['ad_link_target']) && $ad['ad_link_target'] == 'blank') ? ' target="_blank"' : '';
                                        $ad_image = '<img class="sb-img" src="'.$ad['ad_image'].'"'.(!empty($admargin) ? ' style="'.$admargin.'max-height: 100%;"' : '').'>';
                                        $adcontent = ! empty($ad['ad_link']) ? '<a href="'.$ad['ad_link'].'"'.$ad_link_target.'>'.$ad_image.'</a>' : $ad_image;
                                    }
                                break;
                            }
                            $adout = $layoutobj->create_ad($adgrid, $adstyle, $adcontent, $ad);
                            $output .= $adout;
                        }
                    }
                    $i++;
                    $val = $layoutobj->create_item(...$arrval);
                    $output .= $val;
                    
                    if ( isset($slideshow) ) {
                        $ss_output .= $this->finalslide[$key];
                    }
                } // end foreach $final
                if (isset($attr['loadmore']) && !empty($attr['loadmore'])) {
                    $_SESSION[$label]['loadcount'] = $i;
                }
            } else {
                if ( empty($loadmore) )
                    $output_error = axs_throw_error( 'There is no feed data to display!' );
            }
        } else {
            if ( empty($loadmore) )
                $output_error = axs_throw_error( 'There is no feed to show or there is a connectivity problem!' );
        }
			
        } else {
            if ( empty($loadmore) )
                $output_error = axs_throw_error( 'The theme does not exist or the layout is not enabled!' );
        }

        if (isset($attr['loadmore']) && !empty($attr['loadmore'])) {
            $_SESSION[$label]['loadmore'] = $loadmore;
        }
        
        if ($ajax_feed && $is_feed) {
            if (isset($output_error) && !empty($output_error))
                $output .= $output_error;
        }
        
    	if ( ! $ajax_feed ) {
            if ( $is_feed ) {
                if (isset($output_error) && !empty($output_error)) {
                    $output .= $output_error;
                }
                $output .= (!empty($attr['carousel'])) ? "</div>" : "</ul></div>";

                if (empty($output_error)) {
                    if (empty($attr['carousel'])) {
                        if (isset($attr['autostart']) && !empty($attr['autostart'])) {
                            $play_none = ' style="display: none;"';
                        } else {
                            $pause_none = ' style="display: none;"';
                        }
                        $controls = (isset($attr['controls']) && !empty($attr['controls'])) ? '
                            <div class="control">
                                <span class="sb-hover" id="ticker-next-'.$label.'"><i class="sb-bico sb-wico sb-arrow-down"></i></span>
                                <span class="sb-hover" id="ticker-prev-'.$label.'"><i class="sb-bico sb-wico sb-arrow-up"></i></span>
                                <span class="sb-hover" id="ticker-pause-'.$label.'"'.($pause_none ?? '').'><i class="sb-bico sb-wico sb-pause"></i></span>
                                <span class="sb-hover" id="ticker-play-'.$label.'"'.($play_none ?? '').'><i class="sb-bico sb-wico sb-play"></i></span>
                            </div>' : '';
                            
                        $filters = '';
                        if ( empty($attr['tabable']) && !empty($filterItems) && !empty($feeds) ) {
                            $filters = (!empty($attr['filters'])) ? '
                            <div class="filter">
                                <span class="sb-hover'.(!empty($attr['default_filter']) ? '' : ' active').'" data-filter="all"><i class="sb-bico sb-wico sb-ellipsis-h" title="'.__( 'Show All', 'social-board' ).'"></i></span>
                                '.implode("\n", $filterItems).'
                            </div>' : '';
                        }
                        
                        if (!empty($attr['filters']) or !empty($attr['controls']))
                        $output .= '
                        <div class="toolbar">
                            '.$controls.'
                            '.$filters.'
                        </div>'."\n";
                    }
                }
            }
        }

        if ($is_wall || $is_timeline || $is_grid) {
            if (isset($output_error) && !empty($output_error)) {
                $output = str_replace(' timeline ', ' ', $output);
                $output .= $output_error;
            }
        }
        
        if ( ! $ajax_feed) {
            $output .= "</div>\n";
            $loadmoretxt = (isset($attr['loadmore']) && $attr['loadmore'] == 2) ? '' : '<p>'.__( 'Load More', 'social-board' ).'</p>';
            $loadmorecls = (isset($attr['loadmore']) && $attr['loadmore'] == 2) ? ' sb-infinite' : '';
            if ( ( ! $is_feed && empty($output_error) ) && !empty($attr['loadmore']) ) {
                $output .= '<div class="sb-loadmore'.$loadmorecls.'" data-nonce="'.wp_create_nonce( 'loadmore' ).'">'.$loadmoretxt.'</div>'."\n";
            }
            if ($is_wall || $is_timeline || $is_grid)
                $output .= "</div>\n";

            $iframe_output = $iframe_slideshow = $iframe_media = '';
            if (isset($attr['lightboxtype']) && $attr['lightboxtype'] == 'slideshow') {
			$iframe_output = $iframe_slideshow = '
                $(".sb-inline").colorbox({
                    inline:true,
                    rel:"sb-inline",
                    href: function(){
                      return $(this).data("href");
                    },
					maxHeight:"95%",
                    width:"85%",
                    current:"slide {current} of {total}",
                    onComplete: function() {
                        var href, attrwidth, aspectratio, newheight = "";
						var winCurrentWidth = $(window).width();
                        var winCurrentHeight = $(window).height();
                        href = $(this).data("href");
						if (winCurrentWidth >= 768) {
							thumbimg = $(href + " .sb-inner .sb-thumb img," + href + " .sb-inner .sb-thumb iframe");
							attrwidth = thumbimg.attr("width");
							if (!attrwidth) {
								sizearrY = thumbimg.height();
								sizearrX = thumbimg.width();
								if (sizearrY) {
									var gapHeight = Math.round((winCurrentHeight * 5) / 100);
									var currentHeight = winCurrentHeight-gapHeight-30;
									if (currentHeight < sizearrY) {
										var newheight = currentHeight;
										
										aspectratio = sizearrX * newheight;
										newwidth = Math.round(aspectratio / sizearrY);
										sizearrX = newwidth;
										sizearrY = newheight;
										
                                        thumbimg.height(newheight);
									} else {
										var newheight = "500";
									}
									$(href + " .sb-inner .sb-body").innerHeight(newheight);
									
									if (thumbimg.height() > 500) {
										thumbimg.height(newheight);
									}
								}
								jQuery(this).colorbox.resize({innerHeight:newheight});
							}
                        }
                        /* Auto-trigger comments */
                        if ( $(href + " .sb-fetchcomments a.sb-triggercomments").length )
                            $(href + " .sb-fetchcomments a.sb-triggercomments").trigger("click");
                    },
                    onLoad:function(){
                        $(".sb-slide .sb-thumb").empty();
                        var sizestr, href, inner, type, media, size = "";
						var wsize = sb_getwinsize();
						var bheight = (wsize.newHeight < 500) ? wsize.newHeight : 500;
                        href = $(this).data("href");
                        inner = $(href + " .sb-inner");
                        type = inner.data("type");
                        if (type) {
                            media = inner.data("media");
                            size = inner.data("size");
                            sizearr = size.split(",");
                            sizearrX = sizearr[0];
                            sizearrY = sizearr[1];
							thumb = inner.children(".sb-thumb");
							newConWidth = Math.round((wsize.newWidth * 70) / 100);
                            
							if ( (sizearrX && sizearrY) && (sizearrX > 400 || sizearrY > 400) ) {
								if (wsize.winCurrentWidth > 768) {
									if (sizearrY < 400) {
										thumb.width("50%");
										inner.children(".sb-body").width("50%").children(".sb-slide-footer").width("50%");
									}
									
									if (wsize.winCurrentHeight < sizearrY || newConWidth < sizearrX) {
										aspectratio = sizearrX * wsize.newHeight;
										sizearrX = Math.round(aspectratio / sizearrY);
										sizearrY = wsize.newHeight;
										
										if (sizearrX > newConWidth) {
											aspectratio = sizearrY * newConWidth;
											sizearrY = Math.floor(aspectratio / sizearrX);
											sizearrX = newConWidth;
											$(href + " .sb-inner .sb-body").innerHeight(sizearrY);
										} else {
											$(href + " .sb-inner .sb-body").innerHeight(wsize.newHeight);
										}
									} else {
										if (sizearrY && sizearrY > 400) {
											$(href + " .sb-inner .sb-body").innerHeight(sizearrY);
										}
									}
								} else {
									if (wsize.newWidth < sizearrX) {
										aspectratio = sizearrY * wsize.newWidth;
										sizearrY = Math.round(aspectratio / sizearrX);
										sizearrX = wsize.newWidth;
									} else if (newConWidth < sizearrX) {
										aspectratio = sizearrY * newConWidth;
										sizearrY = Math.round(aspectratio / sizearrX);
										sizearrX = newConWidth;
                                    }
                                    $(href + " .sb-inner .sb-body").innerHeight("auto");
								}
							} else {
								sizestr = "";
								thumb.width("50%");
								inner.children(".sb-body").width("50%").children(".sb-slide-footer").width("50%");
							}
							
                            if (type == "image") {
                                if ( (sizearrX && sizearrY) && (sizearr[0] > 400 || sizearr[1] > 400) ) {
                                    sizestr = " style=\'width:" + sizearrX + "px;height:" + sizearrY + "px\' width=\'" + sizearr[0] + "\' height=\'" + sizearr[1] + "\'";
                                    thumb.html("<img decoding=\"async\" class=\"sb-img\" src=\"" + media + "\"" + sizestr + " alt=\"\">");
                                } else {
                                    thumb.html("<span><img decoding=\"async\" src=\"" + media + "\" class=\"sb-img sb-imgholder\" alt=\"\"></span>");
                                }
                            } else if (type == "video") {
                                if ( (sizearrX && sizearrY) && (sizearrX > 400 || sizearrY > 400) ) {
										sizestr = " style=\'width:" + sizearrX + "px;height:" + sizearrY + "px\' width=\'" + sizearr[0] + "\' height=\'" + sizearr[1] + "\'";
                                } else {
                                    sizestr = " width=\'560\' height=\'100%\'";
                                }
                                var imedia = "<iframe" + sizestr + " src=\"" + media + "\" allowfullscreen=\"\" webkitallowfullscreen=\"\" mozallowfullscreen=\"\" autoplay=\"0\" wmode=\"opaque\" frameborder=\"0\"></iframe>";
								if (sizearr[1] && sizearr[1] > 400) {
									thumb.html(imedia);
								} else {
									thumb.html("<span>" + imedia + "</span>");
									$(href + " .sb-inner .sb-body").innerHeight(bheight);
								}
                            } else {
                                if (sizearrY && sizearrY > 400) {
									thumb.html(media);
								} else {
									thumb.html("<span>" + media + "</span>");
									if (wsize.winCurrentWidth > 768)
										$(href + " .sb-inner .sb-body").innerHeight(bheight);
								}
                            }
                        } else {
							$(href + " .sb-inner .sb-body").innerHeight(bheight);
						}
						var _childs = href.replace("inline","childs");
						if($(_childs).length){
							var _childItems = $(_childs).html();
							var _ImgClassAttr;
							if(!thumb.children("span").length){
								_ImgClassAttr = thumb.children("img").first().attr("class");
								thumb.html("<span></span>");
							}
							else{
								_ImgClassAttr = thumb.children("span").children("img").first().attr("class");
							}
							thumb.children("span").append(_childItems + "<i></i><i></i>");
							thumb.children("span").children("img").addClass(_ImgClassAttr);
							thumb.children("span").children("img").first().show();
							thumb.children("span").children("i").first().on("click",function(){
							var _ThisImgs = thumb.children("span").children("img");
								_ThisImgs.each(function(i,v){
									if($(this).css("display") != "none"){
										if(i == 0){
											$(this).hide();
											_ThisImgs.last().show();
										}
										else{
											$(this).hide().prev().show();
										}
										return false;
									}
								});
							});
							thumb.children("span").children("i").last().on("click",function(){
							var _ThisImgs = thumb.children("span").children("img");
								_ThisImgs.each(function(i,v){
									if($(this).css("display") != "none"){
										if(_ThisImgs.last().index() == i){
											$(this).hide();
											_ThisImgs.first().show();
										}
										else{
											$(this).hide().next().show();
										}
										return false;
									}
								});
							});
						}
                    },
                    onClosed:function(){ $(".sb-slide .sb-thumb").empty(); }
                });';
            } else {
                $iframe_output = '
				$(".sboard .sb-thumb .iframe").colorbox({
					iframe: true,
                    maxWidth: "85%",
                    maxHeight: "95%",
					width: function() {
                        var size = $(this).data("size");
                        if (size) {
                            sizearr = size.split(",");
				            return parseInt(sizearr[0])+10;
                        } else {
                            return 640;
                        }
					},
					height: function() {
                        var size = $(this).data("size");
                        if (size) {
                            sizearr = size.split(",");
                            return parseInt(sizearr[1])+10;
                        } else {
                            return 460;
                        }
					},
					onComplete: function() {
						var size = $(this).data("size");
                        if (size) {
    						var sizearr = size.split(",");
    						var iframebox = $( "#cboxLoadedContent iframe" );
    						if (iframebox.length) {
    							iframebox.attr("width", sizearr[0]).attr("height", sizearr[1]);
    						}
                        }
					}
				});';
                $iframe_output .= '
                $(".sboard .icbox, .sboard .sb-thumb .icbox").colorbox({
                    photo: true,
                    href: function() {
                        return $(this).attr("href") ? $(this).attr("href") : $(this).data("href");
                    },
                    maxWidth: "95%",
                    maxHeight: "95%",
                    onComplete: function() {
						if( $(this).closest(".timeline-row").attr("id") ){
							var _childs = "#childs_" + $(this).closest(".timeline-row").attr("id");
						}
						else{
							var _childs = "#childs_" + $(this).closest(".sb-item").attr("id");
						}
						if($(_childs).length){
							var _childItems = $(_childs).html();
							$("#cboxLoadedContent").append(_childItems + "<i></i><i></i>");
							$("#cboxLoadedContent").children("i").first().on("click",function(){
							var _ThisImgs = $("#cboxLoadedContent").children("img");
								_ThisImgs.each(function(i,v){
									if($(this).css("display") != "none"){
										if(i == 0){
											$(this).hide();
											_ThisImgs.last().show();
										}
										else{
											$(this).hide().prev().show();
										}
										return false;
									}
								});
							});
							$("#cboxLoadedContent").children("i").last().on("click",function(){
							var _ThisImgs = $("#cboxLoadedContent").children("img");
								_ThisImgs.each(function(i,v){
									if($(this).css("display") != "none"){
										if(_ThisImgs.last().index() == i){
											$(this).hide();
											_ThisImgs.first().show();
										}
										else{
											$(this).hide().next().show();
										}
										return false;
									}
								});
							});
						}
					}
                });
                $(".sboard .inline, .sboard .sb-thumb .inline").colorbox({
                    inline: true,
                    maxWidth: "95%",
                    maxHeight: "95%"});';
                $iframe_media = $iframe_output;
            }
        
        // Lazy load images
        $lazyload_output = '';
        $layout_container = (isset($attr['wall_height']) && $attr['wall_height'] != '') ? '"#sb_'.$label.'"' : 'window';
        if (isset($attr['lazyload']) && !empty($attr['lazyload'])) {
            $lazyload_output = '
                $(".sb-thumb img").lazyload({
                    effect: "fadeIn",
                    skip_invisible: true,';
                if ( $is_wall ) {
                    if (isset($attr['wall_height']) && $attr['wall_height'] != '')
                        $lazyload_output .= 'container: $("#sb_'.$label.'"),';
                    $lazyload_output .= '
                        appear: function() {
                            $wall.isotope("layout");
                        }';
                }
                $lazyload_output .= '
                });';
        }
		
        // loadmore ajax function
        $more_output = '';
        if (isset($attr['loadmore']) && !empty($attr['loadmore'])) {
            if (isset($attr['loadmore']) && $attr['loadmore'] == 2) {
                $fail_func = 'console.log';
                $more_output .= '
                var sbwin = $(window);
                /* Each time the user scrolls */
                sbwin.scroll(function() {
					var bottom_of_element = $("#timeline_'.$label.'").offset().top + $("#timeline_'.$label.'").innerHeight();
					var bottom_of_screen = sbwin.scrollTop() + sbwin.innerHeight();
					/* End of the document reached? */
                    if ( (Math.ceil(bottom_of_screen) >= Math.ceil(bottom_of_element) - 200) && !$("#timeline_'.$label.'").hasClass("loadmore-active") ) {
						$("#timeline_'.$label.'").addClass("loadmore-active");';
            } else {
                $fail_func = 'alert';
                $more_output .= '
                jQuery("#sb_'.$label.'").on("click", ".sb-loadmore", function() {';
            }
            $lazyload_output_append = (!empty($attr['lazyload'])) ? str_replace('$(".sb-thumb img")', 'lmdata.find(".sb-thumb img")', $lazyload_output) : '$('.$layout_container.').trigger("resize");';
            $more_output .= '
                lmobj = $("#sb_'.$label.' .sb-loadmore");
                lmnonce = lmobj.attr("data-nonce");';
            $more_output .= "$('#sb_".$label." .sb-loadmore').html('<p class=\"sb-loading\">&nbsp;</p>');";
            $more_output .= '
                $.ajax({
                type: "post",
                url: "'.admin_url( 'admin-ajax.php' ).'",
                data: {
                    action: "sb_loadmore",
                    attr: '.$attr_ajax.',
                    nonce: lmnonce,
                    label: "'.$label.'"
                },
                cache: false
                })
                .done(function( response ) {
                    /* append and layout items */';
                if ( $is_wall ) {
                    $more_output .= '
                    var lmdata = $(response);
                    var $items = lmdata.filter(".sb-item");
                    var $slides = lmdata.filter(".sb-slide");
                    $wall.append( $items ).isotope( "appended", $items );
                    $("#sb_slides_'.$label.'").append( $slides );
                    $(window).one("transitionend webkitTransitionEnd oTransitionEnd otransitionend MSTransitionEnd", function(e) {
                        '.$lazyload_output_append.'
                        $wall.one( "layoutComplete", function() {
                            $('.$layout_container.').trigger("resize");
                        });
                        $wall.isotope("layout");
                    });';
                } elseif ( $is_grid ) {
                    $more_output .= '
                    $("#timeline_'.$label.'").append(response).find(".sb-thumb img").lazyload();
                    $('.$layout_container.').trigger("resize");';
                } else {
                    $more_output .= '
					$("#timeline_'.$label.'").append(response).find(".sb-thumb img").lazyload();
					$('.$layout_container.').trigger("scroll");';
                }
                $more_output .= $iframe_output . '
					$("#sb_'.$label.' .sb-loadmore").html("'.$loadmoretxt.'");
					$("#timeline_'.$label.'").removeClass("loadmore-active");
                })
                .fail(function() {
                    '.$fail_func.'("Problem reading the feed data!");
                });';
            if (isset($attr['loadmore']) && $attr['loadmore'] == 2) {
                $more_output .= '
                  }';
            }
                $more_output .= '
                });';
        }
        
        // Fetch comments ajax function
        $more_output .= '
        jQuery(".sboard").on("click", ".sb-fetchcomments a.sb-triggercomments", function() {';
        $more_output .= '
            fcobj = $(this).parent();
            fcnonce = fcobj.attr("data-nonce");';
        $more_output .= "fcobj.html('<p class=\"sb-loading\">&nbsp;</p>');";
        $more_output .= '
            $.ajax({
                type: "post",
                url: "'.admin_url( 'admin-ajax.php' ).'",
                data: {
                    action: "sb_fetchcomments",
                    network: fcobj.attr("data-network"),
                    attr: '.$attr_ajax.',
                    id: fcobj.attr("data-id"),
                    feed: fcobj.attr("data-feed"),
                    link: fcobj.attr("data-link"),
                    nonce: fcnonce,
                    label: "'.$label.'"
                },
                cache: false
            })
            .done(function( response ) {
                /* replace comments */';
            if ( $is_wall ) {
                $more_output .= '
                /* re-layout wall */
                fcobj.html(response).promise().done(function() {
                    $wall.isotope("layout");
                    $('.$layout_container.').trigger("scroll");
                });';
            } else {
                $more_output .= '
                fcobj.html(response);';
            }
            $more_output .= '
            })
            .fail(function() {
                fcobj.html(\'<a href="javascript:void(0)" class="sb-triggercomments">'.__( 'Show Comments', 'social-board' ).'</a>\');
                alert("Problem reading the comments feed data!");
            });';
        $more_output .= '
        });';
        // END: fetch comments ajax function

        $output .= '
        <script type="text/javascript">
            jQuery(document).ready(function($) {
				function sb_getwinsize() {
					var wsize = {
						winCurrentWidth: $(window).width(),
						newWidth: 0,
						winCurrentHeight: $(window).height(),
						newHeight: 0
					};
					var gapWidth = Math.round((wsize.winCurrentWidth * 15) / 100);
					var currentWidth = wsize.winCurrentWidth-gapWidth;
					wsize.newWidth = currentWidth-10;
					
					var gapHeight = Math.round((wsize.winCurrentHeight * 5) / 100);
					var currentHeight = wsize.winCurrentHeight-gapHeight;
					wsize.newHeight = currentHeight-30;
					return wsize;
				}';

        $ticker_id_t = '';
        if ( $is_feed ) {
            if (isset($attr['carousel']) && !empty($attr['carousel'])) {
                if ( isset($attr['cs_item']) && ! is_array($attr['cs_item']) ) {
                    if ($attr['cs_item'] == '') {
                        $attr['cs_item'] = $defoption['carouselsetting']['cs_item'];
                    } else {
                        $attr['cs_item'] = explode(',', trim($attr['cs_item']) );
					}
                }

				$output .= '
				    $("#ticker_'.$label.'").on("init",function(e,s){
				        if($("#ticker_'.$label.' .sb-item > .sb-container").length){
				        	let _top = 0;
				        	let _changeTicker = false;
				            $("#ticker_'.$label.' .sb-item > .sb-container").each(function(){
								_top = $(this).bind().height() > _top ? $(this).bind().height() : _top;
							});
				            $("#ticker_'.$label.' .sb-item > .sb-container").each(function(){
				                let _innerFixHeight = _top - $(this).bind().height();
				                let _innerFixEl = ($(this).find(".sb-text").length == 1) ? $(this).find(".sb-text") : $(this).find(".sb-inner");
				                
				                if(_innerFixEl.length){
				                	_changeTicker = true;
				                	$(this).height(_top);
				                	$(this).find(_innerFixEl).height(_innerFixHeight + _innerFixEl.height());
				                }
				                
				            });
				            if(_changeTicker){
				                $("#ticker_'.$label.'").height($("#ticker_'.$label.' .slick-slide.slick-current.slick-active > div").first().bind().height() * s.options.rows + 10);
				            }
				        }
				    });
				';

                $output .= '
                var slider_options = {
                    accessibility: true,
                    waitForAnimate: false,
                    variableWidth: false,
                    initialSlide: 0,
                    adaptiveHeight: false,
                    infinite: '.(!empty($attr['cs_loop']) ? 'true' : 'false').',
                    dots: '.(!empty($attr['cs_pager']) ? 'true' : 'false').',
                    arrows: '.(!empty($attr['cs_controls']) ? 'true' : 'false').',
                    speed: '.(!empty($attr['cs_speed']) ? $attr['cs_speed'] : '400').',
                    autoplay: '.(!empty($attr['cs_auto']) ? 'true' : 'false').',
                    autoplaySpeed: '.(!empty($attr['cs_autospeed']) ? $attr['cs_autospeed'] : '2000' ).',
                    pauseOnHover: '.(!empty($attr['cs_pause']) ? 'true' : 'false').',
                    rows: '.(!empty($attr['cs_rows']) ? $attr['cs_rows'] : '1' ).',
                    slidesPerRow: 1,
                    slidesToShow: '.(!empty($attr['cs_item'][0]) ? $attr['cs_item'][0] : '1' ).',
                    slidesToScroll: '.((!empty($attr['cs_item'][0]) && !empty($attr['cs_slide']) && $attr['cs_slide'] == 'default' ) ? $attr['cs_item'][0] : 1 ).',
                    rtl: '.(!empty($attr['cs_rtl']) ? $attr['cs_rtl'] : 'false' ).',
                    lazyLoad: "ondemand",
                    prevArrow: "<button type=\'button\' class=\'slick-prev ax-slider__arrow\'>></button>",
                    nextArrow: "<button type=\'button\' class=\'slick-next ax-slider__arrow\'><</button>",';
                
                    $output .= '
                    /* responsive options */
                    respondTo: "slider",
                    responsive: [{
                        breakpoint: 960,
                        settings: {
                          slidesToShow: '.($attr['cs_item'][1] ?? '1').',
                          slidesToScroll: '.($attr['cs_item'][1] ?? '1').'
                        }
                    }, {
                        breakpoint: 768,
                        settings: {
                          slidesToShow: '.($attr['cs_item'][2] ?? '1').',
                          slidesToScroll: '.($attr['cs_item'][2] ?? '1').'
                        }
                    }, {
                        breakpoint: 600,
                        settings: {
                            slidesToShow: '.($attr['cs_item'][3] ?? '1').',
                            slidesToScroll: '.($attr['cs_item'][3] ?? '1').'
                        }
                    }, {
                        breakpoint: 480,
                        settings: {
                            slidesToShow: '.($attr['cs_item'][4] ?? '1').',
                            slidesToScroll: '.($attr['cs_item'][4] ?? '1').'
                        }
                    }]
                };
                $("#ticker_'.$label.'").slick(slider_options);';
            }
			
            if (empty($attr['carousel'])) {
                $ticker_id = '#ticker_'.$label;
                
                $ticker_lazyload_output = '';
                if (!empty($attr['lazyload'])) {
                    $ticker_lazyload_output = '
                    $(".sb-thumb img").lazyload({
                        effect: "fadeIn",
                        skip_invisible: true,';
                    if (isset($attr['wall_height']) && $attr['wall_height'] != '')
                        $ticker_lazyload_output .= 'container: $("'.$ticker_id.'"),';
                    $ticker_lazyload_output .= '
                        threshold: '.($block_height * 2).',
                        failure_limit: '.$results.'
                    });';

                    $output .= '
                    function sb_tickerlazyload() {
                        var lielem = $("'.$ticker_id.' li:last-child");
                        var lix = lielem.index();
                        for (i = 0; i < 4; i++) {
                            var inum = lix-i;
                            var imgelem = $("'.$ticker_id.' li").eq(inum).find(".sb-thumb img");
                            if (typeof imgelem.attr("data-original") !== "undefined" && imgelem.attr("data-original") !== null)
                                $( "img[data-original=\'"+imgelem.attr("data-original")+"\']" ).trigger("appear");
                        }
                    }';
                }
                
				$output .= '
                var $sbticker = $("'.$ticker_id.'").newsTicker({
                    row_height: '.$block_height.',
                    max_rows: 1,
                    speed: '.($attr['rotate_speed'] ?? '').',
                    duration: '.($attr['duration'] ?? '').',
                    direction: "'.($attr['direction'] ?? '').'",
                    autostart: '.($attr['autostart'] ?? '').',
                    pauseOnHover: '.($attr['pauseonhover'] ?? '').',
                    prevButton: $("#ticker-prev-'.$label.'"),
                    nextButton: $("#ticker-next-'.$label.'"),
                    stopButton: $("#ticker-pause-'.$label.'"),
                    startButton: $("#ticker-play-'.$label.'"),
                    start: function() {
                    	$("#timeline_'.$label.' #ticker-pause-'.$label.'").show();
                        $("#timeline_'.$label.' #ticker-play-'.$label.'").hide();
                    },
                    stop: function() {
                    	$("#timeline_'.$label.' #ticker-pause-'.$label.'").hide();
                        $("#timeline_'.$label.' #ticker-play-'.$label.'").show();
                    },
					movingUp: function() {
						$("'.$ticker_id.'").trigger("scroll");
                    },
                    movingDown: function() {';
                if (isset($attr['lazyload']) && !empty($attr['lazyload'])) {
                    $output .= '
                    sb_tickerlazyload();';
                }
                $output .= '
                    }
                });';
                if (!empty($attr['tabable']) && !empty($attr['autoclose']) ) {
				    $output .= '$sbticker.newsTicker("pause");';
                }
                if (isset($attr['lazyload']) && !empty($attr['lazyload'])) {
                    $output .= $ticker_lazyload_output . '
                    sb_tickerlazyload();';
                }
				
				// Filtering rotating feed
				if ( empty($attr['tabable']) && !empty($attr['filters']) ) {
                $output .= "
                $('#timeline_$label .filter span').on('click',function() {
            		/* fetch the class of the clicked item */
            		var ourClass = $(this).data('filter');
            		
            		/* reset the active class on all the buttons */
            		$('#timeline_$label .filter span').removeClass('active');
            		/* update the active state on our clicked button */
            		$(this).addClass('active');
            		
            		if (ourClass == 'all') {
            			/* show all our items */
            			$('$ticker_id').children('li.sb-item').show();
            		} else {
            			/* hide all elements that don't share ourClass */
            			$('$ticker_id').children('li:not(' + ourClass + ')').fadeOut('fast');
            			/* show all elements that do share ourClass */
            			$('$ticker_id').children('li' + ourClass).fadeIn('fast');
                        
        				setTimeout(function() {
                            $('$ticker_id').trigger('scroll');
        				}, 500);
            		}
            		return false;
            	});";
				}
            }
                
            if ( !empty($attr['slide']) && ! (!empty($attr['tabable']) && isset($attr['position']) && $attr['position'] == 'normal') ) {
                if ( $location == 'left' || $location == 'right' ) {
                    $getsizeof = 'Width';
                    $opener = 'sb-opener';
                    $padding = '';
                } else {
                    $getsizeof = 'Height';
                    $opener = 'sb-heading';
                    $padding = ( !empty($attr['showheader']) || ($location == 'bottom' && empty($attr['tabable'])) ) ? ' - 30' : '';
                }
                $openid = (!empty($attr['tabable'])) ? "#timeline_$label .sb-tabs li" : "#timeline_$label .$opener";
                $output .= "
                /* slide in-out */
                var padding = $('#timeline_$label').outer$getsizeof();
                padding = parseFloat(padding)$padding;";
                $output .= ( !empty($attr['autoclose']) ) ? "$('#timeline_$label').animate({ '$location': '-='+padding+'px' }, 'fast' );" : '';
                $output .= "
                $('$openid').on('click',function(event) {
                    if ( $('#timeline_$label').hasClass('open') ) {
                        if ( $(this).hasClass( 'active' ) ) {
                            $('$openid').removeClass('active');
                            $('#timeline_$label').animate({ '$location': '-='+padding+'px' }, 'slow' ).removeClass('open');
                        } else {
                            $('$openid').removeClass('active');
                            $(this).addClass('active');
                        }
                    } else {
                        $(this).addClass('active');
                        $('#timeline_$label').animate({ '$location': '+='+padding+'px' }, 'slow' ).addClass('open');
                    }
                    event.preventDefault();
                });";
			} else {
                // only for normal tabable
                $openid = "#timeline_$label .sb-tabs li";
                $output .= "
                $('$openid').on('click',function(event) {
                    $('$openid').removeClass('active');
                    if ( $('#timeline_$label').hasClass('open') ) {
                        if ( $(this).hasClass( 'active' ) ) {
                            $('#timeline_$label').removeClass('open');
                        } else {
                            $(this).addClass('active');
                        }
                    } else {
                        $(this).addClass('active');
                        $('#timeline_$label').addClass('open');
                    }
                    event.preventDefault();
                });";
            }
            
            if (isset($ticker_id) && !empty($ticker_id))
                $ticker_id_t = ' '.$ticker_id;
        } elseif ( $is_wall ) {
            if ( ! empty($feeds) ) {
                if (isset($attr['stagger']) && !empty($attr['stagger']))
					$wallExt['stagger'] = 'stagger: '.$attr['stagger'];
                if ( ! empty($wallExt) )
                	$wallExtStr = implode(',', $wallExt);
                $columnWidth = (isset($attr['fixWidth']) && $attr['fixWidth'] == 'false') ? '".sb-isizer"' : $itemwidth;
                $gutter = (isset($attr['fixWidth']) && $attr['fixWidth'] == 'false') ? '".sb-gsizer"' : $gutterX;
                $percentPosition = (isset($attr['fixWidth']) && $attr['fixWidth'] == 'false') ? 'true' : 'false';
                if (isset($attr['fixWidth']) && $attr['fixWidth'] == 'block') {
                $output .= '
                function sb_setwallgrid($wall) {
                	var wallw = $wall.width();
                	if (wallw >= 960 && wallw < 1200) {
                		var ncol = '.$attr['breakpoints'][1].';
                		'.$bpcol[1].'
                	}
                	else if (wallw >= 768 && wallw < 959) {
                		var ncol = '.$attr['breakpoints'][2].';
                		'.$bpcol[2].'
                	}
                	else if (wallw >= 600 && wallw < 767) {
                		var ncol = '.$attr['breakpoints'][3].';
                		'.$bpcol[3].'
                	}
                	else if (wallw >= 480 && wallw < 599) {
                		var ncol = '.$attr['breakpoints'][4].';
                		'.$bpcol[4].'
                	}
                	else if (wallw >= 320 && wallw < 479) {
                		var ncol = '.$attr['breakpoints'][5].';
                		'.$bpcol[5].'
                	}
                	else if (wallw <= 319) {
                		var ncol = '.$attr['breakpoints'][6].';
                		'.$bpcol[6].'
                	} else {
						var ncol = '.$attr['breakpoints'][0].';
						'.$bpcol[0].'
					}
					var twgut = '.$gutterX.' * (ncol-1);
                	var itemw = (wallw - twgut) / ncol;
                	$wall.isotope({
                		masonry: {
                			columnWidth: parseFloat(itemw.toFixed(3)),
                			gutter: '.$gutterX.'
                		}
                	});
                }';
                }
                $output .= '
    			var $wall = $("#timeline_'.$label.$ticker_id_t.'").isotope({
                    itemSelector: ".sb-item",
                    layoutMode: "masonry",
					getSortData: {
                      dateid: function( itemElem ) {
                      	return $( itemElem ).attr("id");
                      }
					},
                    percentPosition: '.$percentPosition.',';
                    if (isset($attr['fixWidth']) && $attr['fixWidth'] == 'false') {
                    	$output .= '
                    masonry: {
                      columnWidth: '.$columnWidth.',
                      gutter: '.$gutter.'
                    },';
                    }
                    $output .= '
                    transitionDuration: '.($attr['transition'] ?? '').',
                    originLeft: '.($attr['originLeft'] ?? '').',
                    '.($wallExtStr ?? '').'
    			});';
    			if (isset($attr['fixWidth']) && $attr['fixWidth'] == 'block')
    				$output .= 'sb_setwallgrid($wall);';
    			$output .= str_replace('.sb-thumb img', '#timeline_'.$label.$ticker_id_t.' .sb-thumb img', $lazyload_output);
				
				$wall_filter = ( isset($attr['default_filter']) && isset($filterItems[$attr['default_filter']]) ) ? '$wall.isotope({ filter: ".sb-'.$attr['default_filter'].'" });' : '';
                $output .= '
                /* layout wall on reload */
                function sb_reloadwall($wall) {
                    $(window).one("transitionend webkitTransitionEnd oTransitionEnd otransitionend MSTransitionEnd", function(e) {
						setTimeout(function() {
							$('.$layout_container.').trigger("resize");
						}, 500);
                        $('.$layout_container.').trigger("scroll");
						$wall.isotope("layout");
						'.$wall_filter.'
                    });
                }
                sb_reloadwall($wall);';

				// Display a Show More buttton to limit items on mobile
				if ( ! empty($attr['showmore']) ) {
					$output .= 'jQuery($wall).SbShowMore({
					btnText: "'.__( "Show More", "social-board-admin" ).'",
						w768: '. ($attr['breakpoints'][3] * 3).',
						w600: '. ($attr['breakpoints'][4] * 2).',
						w480: '. ($attr['breakpoints'][5] * 2).',
						w320: '. ($attr['breakpoints'][6] * 2) .'
					});';
				}
                
                if (empty($attr['lazyload'])) {
                    $output .= '
                    /* layout wall after each image loads */
                    $wall.imagesLoaded().progress( function() {
                        $wall.isotope("layout");
                    });';
                }

                $output .= '
                /* set wall grid on container resize */
                $(window).resize(function() {';
                if (isset($attr['fixWidth']) && $attr['fixWidth'] == 'block')
                    $output .= 'sb_setwallgrid($wall);';
                $output .= '
                    setTimeout(function() {
                        $(window).trigger("scroll");
                    }, 500);
                });';

				if ( !empty($lightbox) || !empty($slideshow) ) {
					$filter_lightbox = '
					if (%s) {
						if ($(this).hasClass("sb-inline")) {
							$(this).addClass("cboxElement");
						} else {
							$(this).find(".sb-container .sb-thumb a.icbox, .sb-container .sb-thumb a.iframe").addClass("cboxElement");
						}
					} else {
						if ($(this).hasClass("sb-inline")) {
							$(this).removeClass("cboxElement");
						} else {
							$(this).find(".sb-container .sb-thumb a").removeClass("cboxElement");
						}
					}';
					$filter_lightbox_all = '
					if ($(this).hasClass("sb-inline")) {
						$(this).addClass("cboxElement");
					} else {
						$(this).find(".sb-container .sb-thumb a.icbox, .sb-container .sb-thumb a.iframe").addClass("cboxElement");
					}';
				}

                // Filter wall by networks
				$output .= '
				$("#sb_'.$label.$ticker_id_t.' .filter-items").on("click", "span", function() {
                    $(".filter-label,.sb-filter").removeClass("active");
                    var filterValue = $(this).addClass("active").attr("data-filter");';
                    if ( empty($attr['filter_ads']) ) {
                        $output .= 'filterValue = (filterValue != "*") ? filterValue + ", .sb-advert" : filterValue;';
					}
					if ( empty($filter_lightbox) ) {
						$filter_lightbox_str = 'filter: filterValue';
					} else {
						$filter_lightbox_str = '
						filter: function() {
							if (filterValue != "*") {
								'.sprintf($filter_lightbox, '$(this).hasClass(filterValue.substring(1))').'
								return ($(this).hasClass(filterValue.substring(1)));
							} else {
								'.$filter_lightbox_all.'
								return filterValue;
							}
						}';
					}
                    $output .= '
					if ( $(this).hasClass( "filter-label" ) ) {
						$wall.isotope({
							'.$filter_lightbox_str.'
						});
	                    $wall.one( "arrangeComplete", function() {
                            $('.$layout_container.').trigger("resize");
	                    });
                    }
    			});';
                
                // Filter wall with a text phrase
				if ( !empty($attr['filter_search']) ) {
                    $output .= '
                $("#sb_'.$label.$ticker_id_t.' .sb-search").keyup(function(){
                    var filterValue = $(this).val();
                    if (filterValue != "") {
                        $wall.isotope({
                            filter: function() {';
							if ( !empty($filter_lightbox) ) {
								$output .= sprintf($filter_lightbox, '$(this).text().search(new RegExp(filterValue, "i")) > 0');
							}
							$output .= '
                                return ($(this).text().search(new RegExp(filterValue, "i")) > 0);
                            }
                        });
                    } else {
                        $wall.isotope({
							filter: function() {';
					if ( !empty($filter_lightbox_all) ) {
						$output .= $filter_lightbox_all;
					}
					$output .= '
								return true;
							}
						});
                    }
                	$wall.one( "arrangeComplete", function() {
                        $('.$layout_container.').trigger("resize");
                	});
                });';
                }
                
                if ( ! empty($this->sboption['filtering_tabs']) ) {
                    $output .= '
                $("#sb_'.$label.$ticker_id_t.' .sb-filter").on("click",function(){
                    var filterTerm = $(this).attr("data-filter");
                    if (filterTerm != "") {
						var filterRegex = /^\.+[a-z]+-\d+-[\s\S]+$/ig;
						if (filterRegex.test(filterTerm)) {
							$wall.isotope({
								filter: filterTerm
							});
						} else {
	                        $wall.isotope({
	                            filter: function() {';
									if ( !empty($filter_lightbox) ) {
										$output .= sprintf($filter_lightbox, '$(this).text().search(new RegExp(filterTerm, "ig")) > 0');
									}
									$output .= '
	                                return ($(this).text().search(new RegExp(filterTerm, "ig")) > 0);
	                            }
	                        });
                        }
                    	$wall.one( "arrangeComplete", function() {
                            $('.$layout_container.').trigger("resize");
                    	});
                    }
                });';
                }
				
                $relayout_scroll = '$wall.isotope("layout");
                $(window).trigger("resize");';
                if (isset($attr['wall_height']) && $attr['wall_height'] != '') {
                    $output .= 'scrollStop(function () {
                        '.$relayout_scroll.'
                    }, "sb_'.$label.'");';
                } else {
                    $output .= 'scrollStop(function () {
                        '.$relayout_scroll.'
                    });';
                }
                $output .= $more_output;
            }
        } elseif ( $is_timeline || $is_grid ) {
            if (!empty($attr['lazyload'])) {
			    $output .= '
				$(".sb-thumb img").lazyload({
					effect: "fadeIn",
					skip_invisible: true
                });';
			}
			
			if ( $is_grid ) {
			// Filter Grid with a text phrase
			if ( !empty($attr['filter_search']) ) {
				$output .= '
				$(document).on("keyup","#sb_'.$label.$ticker_id_t.' #GridSearchFilter",function() {
					GridFiltersFixColumn();
				});';
			}

			if ( !empty($attr['filters']) ) {
				$output .= '
    			/* Filter Grid by networks */
				$("#sb_'.$label.$ticker_id_t.' .filter-items").on("click", "span", function() {
                    $(".filter-label,.sb-filter").removeClass("active");
                    $(this).addClass("active");
                    GridFiltersFixColumn();
    			});';
			}
			
			if ( !empty($attr['filters']) || !empty($attr['filter_search']) ) {
				$output .= '$( document ).ajaxComplete( GridFiltersFixColumn );
				function GridFiltersFixColumn(){
					$("#timeline_'.$label.$ticker_id_t.'").css("width","100%");
					var TwoFold = $("#timeline_'.$label.$ticker_id_t.' .sb-item.sb-twofold").length;
					if(TwoFold){
				
						if($("#timeline_'.$label.$ticker_id_t.' [data-filter-items]").length){
							$("#timeline_'.$label.$ticker_id_t.' [data-filter-items]").remove();
						}
				
						if($("#timeline_'.$label.$ticker_id_t.' .sb-item.sb-filter-status-true").length){
							$("#timeline_'.$label.$ticker_id_t.' .sb-item").removeClass("sb-filter-status-true");
						}
				
						if($("#timeline_'.$label.$ticker_id_t.' .sb-item.sb-filter-search-status-true").length){
							$("#timeline_'.$label.$ticker_id_t.' .sb-item").removeClass("sb-filter-search-status-true");
						}
				
						var _filterActive = $("#sb_'.$label.$ticker_id_t.' .filter-items .active");
						var _filterSearch = $("#sb_'.$label.$ticker_id_t.' #GridSearchFilter");
				
						if(_filterActive == "*" && ! _filterSearch){
							$("#timeline_'.$label.$ticker_id_t.' .sb-column").show();
							return;
						}
				
						$("#timeline_'.$label.$ticker_id_t.' .sb-column").hide();
				
						if(_filterActive.length){
							var filterValue = _filterActive.attr("data-filter");
							if($("#timeline_'.$label.$ticker_id_t.' .sb-item" + filterValue).length){
								$("#timeline_'.$label.$ticker_id_t.' .sb-item" + filterValue).addClass("sb-filter-status-true");
							}
						}
				
						if(_filterSearch.length){
							var _searchValue = _filterSearch.val().toLowerCase();
							var _searchItems = $("#timeline_'.$label.$ticker_id_t.' .sb-item.sb-filter-status-true").addClass("sb-filter-status-true");
							if(_searchItems){
								_searchItems.each(function(i,v){
									if($(this).text().toLowerCase().indexOf(_searchValue) > -1){
										$(this).addClass("sb-filter-search-status-true");
									}
								});
							}
						}
				
						var _filterItems = $("#timeline_'.$label.$ticker_id_t.' .sb-item.sb-filter-status-true.sb-filter-search-status-true");
						var _toggleClass = "sb-twofold";
						var _return = [];
						_filterItems.each(function(i,v){
							if(((i + 1) % 3) == 1){
								$(this).removeClass("sb-twofold").removeClass("sb-solo").addClass(_toggleClass);
								_toggleClass = (_toggleClass == "sb-twofold") ? "sb-solo" : "sb-twofold";
				
								if((_filterItems.length - 1) == i){
									_return.push("<div class=\'sb-column\' data-filter-items>" + v.outerHTML + "</div>");
								}
								else{
									_return.push("<div class=\'sb-column\' data-filter-items>" + v.outerHTML);
								}
							}
							if(((i + 1) % 3) == 2){
								$(this).removeClass("sb-twofold").addClass("sb-solo");
				
								if((_filterItems.length - 1) == i){
									_return.push(v.outerHTML + "</div>");
								}
								else{
									_return.push(v.outerHTML);
								}
							}
							if((((i + 1) % 3) == 0)){
								$(this).removeClass("sb-twofold").removeClass("sb-solo").addClass(_toggleClass);
								_return.push(v.outerHTML + "</div>");
							}
						});
				
						var _out = "";
						$.each(_return, function( index, value ) {
							_out += value;
						});
				
						$("#timeline_'.$label.$ticker_id_t.'").prepend(_out).find(".sb-thumb img").lazyload();
				
					}
					else{
						var _filterActive = $("#sb_'.$label.$ticker_id_t.' .filter-items .active");
						var filterValue = (_filterActive) ? _filterActive.attr("data-filter") : "*";
						var _filterSearch = $("#sb_'.$label.$ticker_id_t.' #GridSearchFilter");
						var _searchValue = (_filterSearch.val().toLowerCase()) ? _filterSearch.val().toLowerCase() : "";
						$("#timeline_'.$label.$ticker_id_t.' .sb-column").hide();

						if(_filterActive){
							if($("#timeline_'.$label.$ticker_id_t.' .sb-item" + filterValue).length){
								$("#timeline_'.$label.$ticker_id_t.' .sb-item" + filterValue).closest(".sb-column").show();
								var _filterItems = $("#timeline_'.$label.$ticker_id_t.' .sb-item" + filterValue);
								_filterItems.each(function(i,v){
									if($(this).text().toLowerCase().indexOf(_searchValue) <= -1){
										$(this).closest(".sb-column").hide();
										$(this).closest(".sb-column").find(".sb-item .sb-container .sb-thumb a").removeClass("cboxElement");
									} else {
										$(this).closest(".sb-column").find(".sb-item .sb-container .sb-thumb a.icbox, .sb-item .sb-container .sb-thumb a.iframe").addClass("cboxElement");
									}
								});
							}
						}
						else{
							var _filterItems = $("#timeline_'.$label.$ticker_id_t.' .sb-item");
							_searchValue.each(function(i,v){
								if($(this).text().toLowerCase().indexOf(_searchValue) > -1){
									$(this).closest(".sb-column").show();
								}
							});
						}
					}
				}';
			}
			}
			
            if (isset($more_output) && !empty($more_output))
				$output .= $more_output;
        }

        // load tabs and rebuild feed ticker
        if (isset($attr['tabable']) && !empty($attr['tabable'])) {
        	$output .= '
            $("#timeline_'.$label.' .sb-tabs").on("click", "li", function() {
                if ( $(this).hasClass( "active" ) ) {
                    feed = $(this).attr("data-feed");
                    tabnonce = $(this).parent().attr("data-nonce");';
                    $output .= "
                    $('#timeline_".$label." .sb-content ul').html('<p class=\"sb-loading\"><i class=\"sb-icon sb-'+feed+'\"></i></p>');";
                    $output .= '
                    $.ajax({
                        type: "post",
                        url: "'.admin_url( 'admin-ajax.php' ).'",
                        data: {action: "sb_tabable", feed: feed, attr: '.$attr_ajax.', nonce: tabnonce},
                        cache: false
                    })
                    .done(function( response ) {
                        $("#timeline_'.$label.$ticker_id_t.'").html(response);
                        $sbticker.newsTicker();
                        ' . $ticker_lazyload_output . $iframe_output . '
                    })
                    .fail(function() {
                        alert( "Problem reading the feed data!" );
                    });
                }
            });';
        }
         
        if ( isset($lightbox) && !empty($lightbox) ) {
            if (isset($attr['lightboxtype']) && $attr['lightboxtype'] == 'slideshow') {
                if ( ! isset($GLOBALS['sb_scripts']['iframe_slideshow']) ) {
                    $output .= $iframe_slideshow;
                    $GLOBALS['sb_scripts']['iframe_slideshow'] = true;
                }
            } else {
                if ( ! isset($GLOBALS['sb_scripts']['iframe_media']) ) {
                    $output .= $iframe_media;
                    $GLOBALS['sb_scripts']['iframe_media'] = true;
                }
            }

            if ( isset($slideshow) ) {
                $colorbox_resize = 'width:"85%"';
                $slicepoint = (!empty($attr['slicepoint'])) ? $attr['slicepoint'] : 300;
				if (0 < $slicepoint) {
                	$output .= '
                    $("div.sb-body .sb-text").expander({
                    slicePoint: '.$slicepoint.',
                    expandText: "'.__( 'read more (+)', 'social-board' ).'",
                    userCollapseText: "'.__( 'read less (-)', 'social-board' ).'"
                    });';
				}
            } else {
                $colorbox_resize = 'maxWidth:"95%", maxHeight:"95%"';
            }
            
            // resize colorbox on screen rotation
            $resize_part1 = '
            $(document).bind("cbox_complete", function(){ $(window).scrollTop($(window).scrollTop()+1); });
            $(window).on("resize", function() {
                if (jQuery("#cboxOverlay").is(":visible")) {
                    var wsize = sb_getwinsize();
                    var cbox = $( "#cboxLoadedContent" );';
                // Slide autosize
                if ( isset($slideshow) ) {
                    $resize_part2 = '
                    var slidespan = $("#cboxLoadedContent .sb-slide .sb-thumb");
                    if (slidespan.length > 0) {
                        var slidethumb = $(".sb-slide .sb-thumb iframe, .sb-slide .sb-thumb img");
                        if ( slidethumb.attr("height") ) {
                            var cwidth = ( cbox.width() < slidethumb.attr("width") ) ? cbox.width() : slidethumb.width();
                            var wwidth = Math.round((wsize.newWidth * 70) / 100);
                            if (cwidth < wwidth && wsize.newHeight > slidethumb.attr("height")) {
                                cwidth = wwidth;
                            }
                            if (slidethumb.attr("width") < wwidth) {
                                var newheight = slidethumb.attr("height") ? slidethumb.attr("height") : wsize.newHeight;
                                cwidth = Math.floor( (slidethumb.attr("width") * newheight ) / slidethumb.attr("height") );
                            } else {
                                if ( $(window).width() > 768 ) {
                                    var newheight = Math.floor( (wwidth * slidethumb.attr("height") ) / slidethumb.attr("width") );
                                    slidethumb.width(wwidth);
                                } else {
                                    var newheight = Math.floor( (wsize.newWidth * slidethumb.attr("height") ) / slidethumb.attr("width") );
                                }
                            }
                            
							if (newheight > wsize.newHeight) {
                                slidethumb.height(wsize.newHeight);
								var setWidth = Math.floor( (cwidth * wsize.newHeight ) / newheight );
								if (slidethumb.width() > setWidth)
                                	slidethumb.width(setWidth);
                            } else {
                                slidethumb.height(newheight);
                                if (slidethumb.width() < cwidth)
                                    slidethumb.width(cwidth);
                            }
                        }
                        
                        if ( $(window).width() >= 768 ) {
                            if (slidespan.children("span").length > 0) {
                                $(".sb-slide .sb-inner .sb-body").innerHeight(500);
                            } else {
                                $(".sb-slide .sb-inner .sb-body").innerHeight( (newheight > wsize.newHeight) ? wsize.newHeight : newheight );
                            }
                        } else {
                            var bheight = wsize.newHeight - newheight;
                            if (bheight < 150) {
                                bheight = 150;
                            }
                            $(".sb-slide .sb-inner .sb-body").css("height", "auto").css("min-height", bheight);
                        }
                    } else {
                        var bheight = (wsize.newHeight < 500) ? wsize.newHeight : 500;
                        $(".sb-slide .sb-inner .sb-body").innerHeight(bheight);
                    }';
                }
                $resize_part3 = '
                    var iframebox = $( "#cboxLoadedContent iframe" );
                    if ( iframebox.length ) {
                        var iframeWidth = iframebox.attr("width");
                        var iframeHeight = iframebox.attr("height");
                        if ( $(window).width() <= 767 ) {
                            var pheight = Math.round( (iframeHeight / iframeWidth) * 95 );
                            jQuery.colorbox.resize({width: "95%", height: pheight+"%"});
                        } else {
                            if ( cbox.children("div.sb-slide").length > 0) {
                                jQuery.colorbox.resize({'.$colorbox_resize.'});
                            } else {
                                if ( iframeHeight > wsize.newHeight ) {
                                    var newWidth = Math.round( (wsize.newHeight * iframeWidth) / iframeHeight);
                                    iframeWidth = newWidth;
                                    iframeHeight = wsize.newHeight;
                                    
                                    if ( iframeWidth > wsize.newWidth ) {
                                        iframeWidth = wsize.newWidth;
                                        iframeHeight = wsize.newHeight;
                                    }
                                }
                                jQuery.colorbox.resize({ width: parseInt(iframeWidth)+10, height: parseInt(iframeHeight)+10 });
                            }
                        }
                    } else {
                        jQuery.colorbox.resize({'.$colorbox_resize.'});
                    }
                }
            });';
				
			if ( isset($slideshow) ) {
				if ( ! isset($GLOBALS['sb_scripts']['resize_slideshow']) ) {
					$output .= $resize_part1.$resize_part2.$resize_part3;
					$GLOBALS['sb_scripts']['resize_slideshow'] = true;
				}
			} else {
				if ( ! isset($GLOBALS['sb_scripts']['resize_media']) && ! isset($GLOBALS['sb_scripts']['resize_slideshow']) ) {
					$output .= $resize_part1.$resize_part3;
					$GLOBALS['sb_scripts']['resize_media'] = true;
				}
			}
        }

			if ( !empty( $attr['full_item_link'] ) ) {
				$output .= 'jQuery(".sboard").FullBlockCoverage({
					link: '. !(empty($attr['links'])).'
				});';
			}
            
			$output .= '
            });
        </script>';
        }
        
        if ( ! $ajax_feed) {
			$output .= !empty($this->forceCrawl) ? "\t<!-- End AX Social Stream Plugin - cache is disabled. -->\n" : "\t<!-- End AX Social Stream Plugin - cache is enabled - duration: " . $attr['cache'] . " minutes -->\n";
		}
		
		$SCRIPT_DEBUG = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG;
		if ( ! $SCRIPT_DEBUG )
			$output = str_replace( array("\r\n","\r","\t","\n"), '', $output );
        
        // Slide-show output
        if ( isset($attr['lightboxtype']) && $attr['lightboxtype'] == 'slideshow' && !empty($ss_output) && empty( $attr['full_item_link']) ) {
            if ( ! $ajax_feed) {
				$output .= '
	    		<div id="sb_slides_'.$label.'" style="display:none">
	                '.$ss_output.'
	    		</div>';
			} else {
				$output .= $ss_output;
			}
        }
    	if ( $echo ) {
			echo $output;
		} else {
			return $output;
		}
    }

    // proper way to enqueue scripts and styles
    public function add_scripts_header() {
        $SCRIPT_DEBUG = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG && file_exists( AXS_DIRNAME . '/public/src' );
        $src = $SCRIPT_DEBUG ? 'src/' : '';

        // Only if in development
        if ( $SCRIPT_DEBUG ) {
            // add legacy css files
            $stylesheets['styles'] = ['styles', 'default', 'flat', 'metro', 'modern', 'modern2'];

            // Add layout based css files
            if ( !empty($this->enabled_layouts['wall']) )
                $stylesheets['wall'] = [];
            if ( !empty($this->enabled_layouts['grid']) )
                $stylesheets['grid'] = ['brick'];
            if ( !empty($this->enabled_layouts['timeline']) )
                $stylesheets['timeline'] = ['timeline'];
            if ( !empty($this->enabled_layouts['feed']) )
                $stylesheets['feed'] = ['styles'];
            if ( !empty($this->enabled_layouts['carousel']) )
                $stylesheets['carousel'] = ['slick', 'slick-theme', 'hero'];

            if ( ! empty($stylesheets) ) {
                foreach ($stylesheets as $layout => $css_files) {
                    foreach ($css_files as $file) {
                        if ( $layout == 'styles' )
                            $layout = '';
                        wp_enqueue_style( 'axss-' . $layout . '-' . $file, plugins_url( 'public/' . $src . $layout . '/css/' . $file . '.css', AXS_FILE ), array(), AXS_VERSION );
                    }
                }
            }
        } else {
            // add legacy bundled css file
            $css_files = ['styles'];

            if ( !empty($this->enabled_layouts) ) {
                foreach ($this->enabled_layouts as $file => $filekey) {
                    if ( $filekey ) {
                        if ( $file != 'wall')
                            $css_files[] = $file;
                    }
                }
            }

            if ( ! empty($css_files) ) {
                foreach ($css_files as $file) {
                    wp_enqueue_style( 'axss-' . $file, plugins_url( 'public/' . $src . 'css/' . $file . '.min.css', AXS_FILE ), array(), AXS_VERSION );
                }
            }
        }
        
        // load fontawesome for hero theme
        if ( !empty($this->enabled_layouts['carousel']) ) {
            wp_enqueue_style( 'axss-carousel-fontawesome', '//use.fontawesome.com/releases/v5.7.2/css/all.css', array(), AXS_VERSION );
        }
        
        // include jquery
        if (empty($GLOBALS['sb_no_jquery']))
        	wp_enqueue_script( 'jquery' );
    }

    public function add_scripts() {
        // add js files based on enabled layouts
        $jquery = ( empty($GLOBALS['sb_no_jquery']) ? array('jquery') : array() );
        $js_assets_position = $this->js_assets_position();

        $SCRIPT_DEBUG = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG && file_exists( AXS_DIRNAME . '/public/src' );
        $src = $SCRIPT_DEBUG ? 'src/' : '';

        // Only if in development
        if ( $SCRIPT_DEBUG ) {
            // add legacy css files
            $scripts['utils'] = ['jquery.colorbox', 'jquery.lazyload', 'jquery.expander', 'jquery.show-more', 'full-block-coverage'];

            // Add layout based css files
            if ( !empty($this->enabled_layouts['wall']) )
                $scripts['wall'] = ['isotope.pkgd', 'imagesloaded.pkgd', 'scrollStop'];
            if ( !empty($this->enabled_layouts['grid']) )
                $scripts['grid'] = [];
            if ( !empty($this->enabled_layouts['timeline']) )
                $scripts['timeline'] = ['timeline'];
            if ( !empty($this->enabled_layouts['feed']) )
                $scripts['feed'] = ['jquery.newsTicker'];
            if ( !empty($this->enabled_layouts['carousel']) )
                $scripts['carousel'] = ['slick'];

            if ( ! empty($scripts) ) {
                foreach ($scripts as $layout => $script_files) {
                    foreach ($script_files as $file) {
                        $layout_str = ($layout == 'utils') ? '' : $layout;
                        wp_enqueue_script( 'axss-' . $layout_str . ( $layout_str ? '-' : '' ) . $file, plugins_url( 'public/' . $src . $layout_str . '/js/' . $file . '.js', AXS_FILE ), $jquery, AXS_VERSION, $js_assets_position['footer'] );
                    }
                }
            }
        } else {
            // add legacy bundled css file
            $script_files = ['utils'];

            if ( !empty($this->enabled_layouts) ) {
                foreach ($this->enabled_layouts as $file => $filekey) {
                    if ( $filekey && $file != 'grid' ) {
                        $script_files[] = $file;
                    }
                }
            }

            if ( ! empty($script_files) ) {
                foreach ($script_files as $file) {
                    wp_enqueue_script( 'axss-' . $file, plugins_url( 'public/' . $src . 'js/' . $file . '.min.js', AXS_FILE ), $jquery, AXS_VERSION, $js_assets_position['footer'] );
                }
            }
        }
    }
	
    // Function to retrieve data from feeds
    public function axs_get_feed( $feed_key, $i, $key2, $feed_value, int $results, $setoption, $sboption, $cache, $sb_label = null ) {
		$feed_value = trim($feed_value['feed']);
		if (isset($this->sboption['limiting']) && $this->sboption['limiting'] == 'custom') {
			$results = ! empty($this->sboption['section_'.$feed_key][$feed_key.'_id_'.$i][$key2]['limit']) ? intval($this->sboption['section_'.$feed_key][$feed_key.'_id_'.$i][$key2]['limit']) : 5;
		}
        switch ( $feed_key ) {
            case 'facebook':
                $pageresults = 9; // the max results that is possible to fetch from facebook API - for group = 9 for page = 30 - we used 9 to support both
                $stepresults = ceil($results / $pageresults);
                
                // If using 3 album ID
                $feedValue = explode('/', $feed_value);
                if ( count($feedValue) == 2 ) {
                    $feed_value = $feedValue[0];
                }

                // Find access token
                if ( !empty($setoption['section_facebook']['facebook_accounts']) ) {
                    $resetAccounts = reset($setoption['section_facebook']['facebook_accounts']);
                    // If is not group feed
					if ($i != 2) {
						// Set default page token
						if ( ! empty($resetAccounts['pages']) ) {
							$resetPages = reset($resetAccounts['pages']);
							$facebook_access_token = $resetPages['access_token'];
						}
					}
                    // If not exist, set default account token
                    if ( !isset($facebook_access_token) ) {
                        $facebook_access_token = $resetAccounts['access_token'];
                    }
                    // Search account's access token by profile ID
                    if ( isset($setoption['section_facebook']['facebook_accounts'][$feed_value]) ) {
                        $facebook_access_token = $setoption['section_facebook']['facebook_accounts'][$feed_value]['access_token'];
                    } else {
                        foreach ($setoption['section_facebook']['facebook_accounts'] as $faccount) {
                            // Search by page ID
                            if ( isset($faccount['pages'][$feed_value]) ) {
                                $facebook_access_token = $faccount['pages'][$feed_value]['access_token'];
                                break;
                            }
                            // If not found, search by page username
                            if ( ! empty($faccount['pages']) ) {
								foreach ($faccount['pages'] as $fpage) {
									if ( isset($fpage['username']) && $fpage['username'] == $feed_value ) {
										$facebook_access_token = $fpage['access_token'];
										break;
									}
								}
							}

							// Search by Group ID
							if ( isset($faccount['groups'][$feed_value]) ) {
								$facebook_access_token = $faccount['groups'][$feed_value]['access_token'];
								break;
							}
							// If not found, search by group username
							if ( !empty($faccount['groups']) ) {
								foreach ($faccount['groups'] as $fpage) {
									if ( isset($fpage['username']) && $fpage['username'] == $feed_value ) {
										$facebook_access_token = $fpage['access_token'];
										break;
									}
								}
                            }
                        }
                    }
                } else {
                    $facebook_access_token = $setoption['section_facebook']['facebook_access_token'] ?? '';
                }
                if ($locale = get_locale() )
                    $locale_str = '&locale='.$locale;
                
                if (isset($sboption['facebook_datetime_range']['from']) && $datetime_from = $sboption['facebook_datetime_range']['from'])
                    $since_str = '&since='.strtotime($datetime_from);
                    
                if (isset($sboption['facebook_datetime_range']['to']) && $datetime_to = $sboption['facebook_datetime_range']['to'])
                    $until_str = '&until='.strtotime($datetime_to);
                
                if ($i == 3 || $i == 4) {
                    if (isset($_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) && $after = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2])
                        $after_str = '&after='.$after;
                } else {
                    if (isset($_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) && $until = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2])
                        $until_str = '&until='.$until;
                }
                // Define fields
                $afields = array(
                    'id', 'created_time', 'updated_time', 'permalink_url',
                    'message', 'story', 'picture', 'full_picture', 'status_type',
                    'from{id,name,picture,link}',
                    'likes.limit(5).summary(true){id,name,username,pic}',
                    'attachments.limit(5){media_type,title,type,description,url,description_tags,media{image,source},target,unshimmed_url,subattachments}'
                );
				if ( !empty( $this->sboption['output']['comments']) && $sboption['facebook_comments'] > 0 ) {
					$afields[] = 'comments.limit(5).summary(true){id,from{id,name,picture,link},message,permalink_url,comments.limit(5).summary(true){id,from{id,name,picture,link},message,permalink_url}}';
				}
                $pagefeed = $sboption['facebook_pagefeed'] ?? 'posts';
                // Touching the fields based on type of the feed
                if ($i == 1) {
                    // Page feed
                    $feedType = $pagefeed;
                } elseif ($i == 2) {
                    // Group feed
                    $feedType = 'feed';
                    unset($afields[10]);
                } elseif ($i == 3) {
                    // Album/Page feed
                    $feedType = 'photos';
                    $afields[] = 'images';
                    if ( count($feedValue) == 2 ) {
                        $feed_value = $feedValue[1];
                    }
                } elseif ($i == 4) {
                    // Page videos feed
                    $feedType = 'videos';
                    $afields[] = 'images';
                    $afields[] = 'title';
                    $afields[] = 'format';
                } elseif ($i == 6) {
                    // User posts feed
                    $feedType = 'feed';
                    array_splice($afields, array_search('likes.limit(5).summary(true){id,name,username,pic}', $afields ), 1);
                }

				$fields = implode(',', $afields);
                $feed_url = 'https://graph.facebook.com/v14.0/' . $feed_value . '/' . $feedType . '?limit=' . $results . ($since_str ?? '') . ($until_str ?? '') . ($after_str ?? '') . ($locale_str ?? '') . '&fields=' . $fields . '&access_token=' . $facebook_access_token;
                $label = 'https://graph.facebook.com/' . $feed_value . '/' . $feedType . '?limit=' . $results;
                // if group feed
                if ($i == 2) {
                    // crawl the feed or read from the cache
                    $get_feed = TRUE;
                    if ( ! $this->forceCrawl ) {
                        if ( $cache->is_cached($label) ) {
                            $content = $cache->get_cache($label);
                            $get_feed = FALSE;
                        }
                    }                
					if ($get_feed) {
                        $feed = array();
                        for ($i = 1; $i <= $stepresults; $i++) {
                            $content = $cache->do_curl($feed_url);
                            $pagefeed = json_decode($content ?? '');
                            if ( ! empty($pagefeed->data) ) {
                                $feed[] = $pagefeed->data;
                                if ( count($pagefeed->data) < $pageresults )
                                    break;
                                $feed_url = $pagefeed->paging->next;
                            }
                        }
               			if ( ! $this->forceCrawl )
                            $cache->set_cache($label, json_encode($feed));
                    } else {
                        $feed = json_decode($content ?? '');
                    }
                }
                else {
                    $content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);
                    if ( $pagefeed = json_decode($content ?? '') ) {
                        if ( isset( $pagefeed->error ) ) {
                            if (!empty($this->attr['debuglog'])) {
                            	if(isset($pagefeed->error->message) && ! empty($pagefeed->error->message)){
									axs_log( 'Facebook error: '.$pagefeed->error->message.' - ' . $feedType);
								}
                            	else{
									axs_log( 'Facebook error: ' . $feedType);
								}
							}
                            $feed[] = null;
                        } else {
                            if ($i == 3 || $i == 4) {
                                $feed[] = $pagefeed;
                            } else {
                                $feed['data'][] = $pagefeed->data ?? '';
                                $feed['next'] = $pagefeed->paging->next ?? '';
                            }
                        }
                    }
                }
            break;
            case 'twitter':
                $twitter_app = trim($setoption['section_twitter']['twitter_app'] ?? '');
                if ($twitter_app == 'yes') {
                    // User our server
                    $feed_value = trim($feed_value);
                    switch($i) {
                        case 1:
                            $rest = 'tweets/'.$feed_value;
                        break;
                        case 3:
                            $rest = 'hashtag/'.urlencode($feed_value);
                        break;
                        case 4:
                            $rest = 'search/'.urlencode($feed_value);
                        break;
                    }
                } else {
				$params = array();
				$params['expansions'] = 'attachments.media_keys,author_id,entities.mentions.username,geo.place_id,in_reply_to_user_id,referenced_tweets.id,referenced_tweets.id.author_id';
				$params['tweet.fields'] = 'attachments,author_id,context_annotations,conversation_id,created_at,entities,geo,id,in_reply_to_user_id,lang,public_metrics,possibly_sensitive,referenced_tweets,reply_settings,source,text';
				$params['user.fields'] = 'created_at,description,entities,id,location,name,pinned_tweet_id,profile_image_url,protected,public_metrics,url,username,verified,verified_type';
				$params['media.fields'] = 'duration_ms,height,media_key,preview_image_url,type,url,width,public_metrics,alt_text,variants';
                switch($i) {
                	case 1:
                        if ( is_numeric($feed_value) ) {
                            $user_id = $feed_value;
                        } else {
							// Get Twitter User ID
							$userrest = 'users/by/username/'.$feed_value;
							$userparams = array(
								'user.fields' => 'created_at,description,entities,id,location,name,pinned_tweet_id,profile_image_url,protected,public_metrics,url,username'
							);
							$user_feed = $this->twitter_get_feed($userrest, $userparams, $cache, $setoption, 0);
							if (isset($user_feed->data)) {
								$user_id = $user_feed->data->id;
							}
						}

						if (isset($user_id)) {
							// Get Twitter User Timeline
							$rest = 'users/'.$user_id.'/tweets';
							$exclude = array();
							// define what type of tweets to filter from the feed
							if (empty($sboption['twitter_feeds']['replies'])) {
								$exclude[] = 'replies';
							}
							if (empty($sboption['twitter_feeds']['retweets'])) {
								$exclude[] = 'retweets';
							}
							if (!empty($exclude)) {
								$params['exclude'] = implode(',', $exclude);
							}
							$feed_min = 5;
						}
                	break;
                	case 2:
						$rest = 'lists/'.$feed_value.'/tweets';
						$feed_min = 1;
                	break;
                	case 3:
                		// The Search API is not complete index of all Tweets, but instead an index of recent Tweets. The index within the last 7 days of Tweets.
                        $rest = 'tweets/search/recent';
                        if (empty($sboption['twitter_feeds']['retweets'])) {
                            $feed_value .= ' -is:retweet';
						}
                        $params['query'] = $feed_value;
						$feed_min = 10;
                	break;
                }
            }
				if (isset($rest)) {
					// Specifies the number of Tweets to try and retrieve. User can not define more than 3 Twitter feeds.
					// Deleted content is removed after the count has been applied.
					$feed_count = count($this->sboption['section_twitter']['twitter_id_'.$i]);
					if ($feed_count == 1) {
						$feed_max = 15;
					} else {
						$feed_max = 10;
					}
                    if (!isset($feed_min)) {
                        $feed_min = 5;
                    }
					if ($results < $feed_min) {
						$params['max_results'] = $feed_min;
					} elseif ($results > $feed_max) {
						$params['max_results'] = $feed_max;
					} else {
						$params['max_results'] = $results;
					}

					if (isset($sboption['twitter_id_range'][0]) && $id_from = $sboption['twitter_id_range'][0]) {
						$params['since_id'] = $id_from;
					}
					if (isset($sboption['twitter_id_range'][1]) && $id_to = $sboption['twitter_id_range'][1]) {
						$params['until_id'] = $id_to;
					}
					if (isset($_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) && $max_id = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) {
						$params['until_id'] = $max_id;
					}
					
					// Fetch the feed
                    if ($twitter_app == 'yes') {
                        $feed = (object) array(
                            'data' => $this->twitter_get_feed($rest, $params, $cache, $setoption, $i)
                        );
                    } else {
					    $feed = $this->twitter_get_feed($rest, $params, $cache, $setoption, $i);
                    }
				}
            break;
            case 'flickr':
                $flickr_api_key = $setoption['section_flickr']['flickr_api_key'] ?? '';
                if (isset($_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) && $nextPage = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2])
                    $pageToken = '&page='.$nextPage;
                if ($i == 1) {
                    $feedType = 'flickr.people.getPublicPhotos';
                    $feedID = '&user_id='.$feed_value;
                } elseif ($i == 2) {
                    $feedType = 'flickr.groups.pools.getPhotos';
                    $feedID = '&group_id='.$feed_value;
                } elseif ($i == 3) {
                    $feedType = 'flickr.photosets.getPhotos';
                    $feedID = '&photoset_id='.$feed_value;
                }
                $feed_url = 'https://api.flickr.com/services/rest/?method='.$feedType.'&api_key='.$flickr_api_key . $feedID . '&per_page=' . $results . ($pageToken ?? '') . '&extras=date_upload,date_taken,owner_name,icon_server,tags,o_dims,views,media,url_c&format=json&nojsoncallback=1';
                $content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);

				$feed = "";
                if(isset($content) && ! empty($content)){
					$feed = json_decode($content);
				}
    		break;
            case 'delicious':
                if ( empty($_SESSION[$sb_label]['loadcrawl']) ) {
                    $feed_url = "https://feeds.del.icio.us/v2/json/" . $feed_value . '?count=' . $results;
                    $content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);

					$feed = "";
					if(isset($content) && ! empty($content)){
						$feed = json_decode($content);
					}
                }
            break;
    		case 'pinterest':
                if ( empty($_SESSION[$sb_label]['loadcrawl']) ) {
                    // get json data
                    $json_uri = ($i == 1) ? 'users' : 'boards';
                    $feed_url = "https://api.pinterest.com/v3/pidgets/$json_uri/" . $feed_value . "/pins/";
                    $content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);

					$feed[0] = "";
					if ( isset($content) && ! empty($content) ) {
						$feed[0] = json_decode($content);
					}

                    if (isset($feed[0]->status) && $feed[0]->status == 'success') {
                        // get rss data
                        $rss_uri = ($i == 1) ? '/feed.rss' : '.rss';
                        $feed_url = "https://www.pinterest.com/" . $feed_value . "$rss_uri";
                        $content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);

						$feed[1] = "";
						if ( isset($content) && ! empty($content) ) {
							libxml_use_internal_errors(true);
							$feed[1] = simplexml_load_string(trim($content));
							libxml_clear_errors();
						}
                    } else {
                    	if ( ! empty($feed[0]->message) ) {
                        	axs_log( 'Pinterest error: '.$feed[0]->message.' - ' . $feed_url);
						}
                        $feed = null;
                    }
                }
    		break;
			case 'instagram':
				$next_max_id = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2] ?? '';
				$feed_token = $this->instagram_access_token($feed_value);
				if ($i == 1) {
					if ($feed_account = $this->sbinstagram->GetAccount($feed_value) ) {
						$feed_value2 = $feed_account["id"];
						$feed_token = $feed_account["access_token"];
						$business_account = true;
					} else {
						$feed_value2 = $this->sbinstagram->GetId($feed_value);
						if ($feed_value2) {
							$feed_token = $this->sbinstagram->GetAccessToken($feed_value2);
						}
					}
				}
				if ( !isset($feed_value2) ) {
					$feed_value2 = $feed_value;
				}

				// Just to be used as cache label
				$feed_url = 'instagram-' . $i . '-' . urlencode($feed_value) . '-' . $results;
				$max_str = 'max_id';
                if ($i == 1) {
                    if ( isset($business_account) ) {
						// User feed over Business API
						$fields = "id,permalink,shortcode,timestamp,media_product_type,media_type,media_url,thumbnail_url,children.limit(5){id,media_url,media_type,thumbnail_url},like_count,comments_count,caption,is_comment_enabled,comments{id,username,name,text,timestamp},owner{id,username,name,profile_picture_url}";
						$feed_url = 'https://graph.facebook.com/v14.0/' . urlencode($feed_value2) . '/media?fields=' . $fields . '&limit=' . $results . '&access_token=' . $feed_token;
						$max_str = 'after';
                    }
                } elseif ($i == 2) {
					// Hashtag feed over Business API
					if ( ! empty($this->setoption['section_instagram']['instagram_bussiness_accounts']) ) {
						$instagram_account = reset($this->setoption['section_instagram']['instagram_bussiness_accounts']);
						if ( ! empty($instagram_account['accounts']) ) {
							$instagram_account = reset($instagram_account['accounts']);
						}
						$feed_token = $instagram_account['access_token'];
						$user_id = $instagram_account['id'];
						$business_account = true;
					}
					// Find hashtag ID
					if ( ! empty($user_id) ) {
						$tag_feed_url = 'https://graph.facebook.com/v14.0/ig_hashtag_search?user_id=' . $user_id . '&q=' . urlencode($feed_value) . '&access_token=' . $feed_token;
						$is_cached = $cache->is_cached($tag_feed_url);
						$user_content = ( ! $this->forceCrawl) ? $cache->get_data($tag_feed_url, $tag_feed_url) : $cache->do_curl($tag_feed_url);
						if (isset($user_content) && ! empty($user_content)) {
							$feed_ig_hashtag = json_decode($user_content);
							if ( ! empty($feed_ig_hashtag->error) ) {
								if ( ! empty($feed_ig_hashtag->error->message) && ! $is_cached ) {
									// No Luck, use private API
									$private_api = true;

									if (isset($cache->debug_log)) {
										axs_log( 'Instagram error: '.$feed_ig_hashtag->error->message.' - ' . $tag_feed_url );
									}
								}
							}
						}

						$fields = 'id,caption,permalink,media_type,media_url,comments_count,like_count,children{id,media_type,media_url,thumbnail_url},timestamp';
						if ( !isset($private_api) ) {
							$hashtag_id = ! empty($feed_ig_hashtag->data[0]->id) ? $feed_ig_hashtag->data[0]->id : '';
							$feed_url = 'https://graph.facebook.com/v14.0/' . urlencode($hashtag_id) . '/recent_media?user_id=' . $user_id . '&fields=' . $fields . '&limit=' . $results . '&access_token=' . $feed_token;
							$max_str = 'after';
						}

						if ( empty($hashtag_id) ) {
							$key = $setoption['section_licensing']['license_key'];
							$domain = preg_replace('/((^https?:\/\/)?(www\.)?)|(\/$)/', '', strtolower(trim(get_site_url(), '/')));
							$rest = 'hashtag';
							$params = array(
								'user_id' => $user_id,
								'q' => $feed_value,
								'fields' => $fields,
								'limit' => $results
							);
							$params = urlencode(serialize($params));
							$feed_url = 'https://api.axentmedia.com/instagram-basic-service/?key=' . $key . '&domain=' . $domain . '&endpoint=' . $rest . '&access_token=' . $feed_token . '&params=' . $params;
						}
					}
                }

                if (isset($feed_url) && !empty($feed_url)) {
                    if ($next_max_id) {
                        $feed_url .= '&'. $max_str .'='.$next_max_id;
					}
                    
                    // Use official API
					$get_feed = true;
					if ( !$this->forceCrawl ) {
						if ( $cache->is_cached($feed_url) ) {
							$content = $cache->get_cache($feed_url);
							$get_feed = false;
						}
					}

					if ( $get_feed && isset($private_api) ) {
						$is_cached = $cache->is_cached($feed_url);
						$content = ( !$this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);
						$feed = "";
						if (isset($content) && !empty($content)) {
							$feed = json_decode($content);
							if ( isset( $feed->error ) OR isset( $feed->account_id ) ) {
								if (isset($cache->debug_log) && !$is_cached) {
									if ( isset( $feed->detail ) ) {
										axs_log( 'Instagram - '.$feed->detail.' - ' . $rest);
									} else {
										axs_log( 'Instagram error: '.$feed->error.' - ' . $rest);
									}
								}
								$feed = null;
							} else {
								$feed = $this->sbinstagram->GetData($feed);
							}
						}
					} else {
						$feed = "";
						if (isset($content) && !empty($content)) {
							$feed = json_decode($content);
						}

						if ( !empty($feed) && isset($business_account) && isset($feed->data) ) {
							$feed = $this->sbinstagram->GetData($feed);
						}
					}
                }

                if ( !isset($feed) || empty($feed) || empty($feed->data) ) {
					// If user feed by Instagram Display API
					if ($i == 1) {
						if ( !isset($business_account) ) {
							$user_content = $this->sbinstagram->GetContent($feed_value2, (int)$results, $next_max_id);
							if ( !empty($user_content) ) {
								$feed = $user_content;
							}
						}
					}

					// If feed by Business API
					if ( isset($business_account) && isset($feed_url) ) {
						$is_cached = $cache->is_cached($feed_url);
						$account_content = ( !$this->forceCrawl) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);

						$feed = "";
						if (isset($account_content) && !empty($account_content)) {
							$feed = json_decode($account_content);
						}
						
						if ( !empty($feed->data) ) {
							$feed = $this->sbinstagram->GetData($feed);
						} else {
							if ( !empty($feed->error) ) {
								if ( !empty($feed->error->message) && !$is_cached ) {
									if (isset($cache->debug_log)) {
										axs_log( 'Instagram error: '.$feed->error->message.' - ' . $feed_url );
									}
								}
								$feed = null;
							}
						}
					}
				}
    		break;
			case 'tiktok':
				if ($i == 1) {
					$cursor = @$_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2];
					$access_token = $this->sbtiktok->get_token($feed_value);
					if ( !empty($access_token) ) {
						$feed_url = 'https://open.tiktokapis.com/v2/video/list/?fields=id,title,create_time,cover_image_url,share_url,video_description,duration,height,width,embed_html,embed_link,like_count,comment_count,share_count,view_count';
						$headers = array(
							'Authorization: Bearer ' . $access_token,
							'Content-Type: application/json'
						);
						$post_fields = array(
							'open_id' => $feed_value,
							'max_count' => ((int) $results > 20) ? 20 : (int) $results
						);

						$label = $feed_url.$post_fields['open_id'];
						if (!empty($cursor)) {
							$post_fields['cursor'] = (int) $cursor;
							$label .= $post_fields['cursor'];
						}

						$get_content = ( !$this->forceCrawl && $cache->is_cached($label)) ? $cache->get_cache($label) : $this->cache->do_curl($feed_url, true, $headers, true, $post_fields);
						if ( !empty($get_content) && is_string($get_content) ) {
							$content = json_decode($get_content);
							if ($content) {
								if (isset($content->data->videos)) {
									if (!$this->forceCrawl) {
										$cache->set_cache($label, $get_content);
									}
									$feed = (object) array(
										'user' => $this->sbtiktok->get_user_data($feed_value),
										'cursore' => @$content->data->cursor,
										'data' => $content->data->videos
									);
								}
							}
						}
					}
					if (empty($feed)) {
						$feed = null;
					}
				}
			break;
    		case 'youtube':
                $google_api_key = $setoption['section_google']['google_api_key'] ?? '';
                if (isset($_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2])
					&& $nextPageToken = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) {
                    $pageToken = '&pageToken='.$nextPageToken;
				}
                switch($i) {
                	case 1:
                    case 4:
                        $channel_filter = ($i == 1) ? 'forUsername' : 'id';
                        $user_url = 'https://www.googleapis.com/youtube/v3/channels?'.$channel_filter.'=' . $feed_value .'&key=' . $google_api_key . '&part=snippet,contentDetails';
                        $user_content = ( ! $this->forceCrawl ) ? $cache->get_data($user_url, $user_url, true) : @$cache->do_curl($user_url);
                        if ($user_content) {

							$user_feed = "";
							if (isset($user_content) && ! empty($user_content)) {
								$user_feed = json_decode($user_content);
							}
							
                            if (isset($user_feed->items[0]) && !empty($user_feed->items[0])) {
                                $feed_url = 'https://www.googleapis.com/youtube/v3/playlistItems?playlistId=' . $user_feed->items[0]->contentDetails->relatedPlaylists->uploads . '&part=snippet,contentDetails,status';
							}
                        }
                    break;
                    case 2:
                        $feed_url = 'https://www.googleapis.com/youtube/v3/playlistItems?playlistId=' . $feed_value . '&part=snippet,contentDetails,status';
                    break;
                    case 3:
                        $feed_url = 'https://www.googleapis.com/youtube/v3/search?q=' . rawurlencode($feed_value) . '&part=snippet';
                    break;
                }
                if ($results > 50) $results = 50;
                if (isset($feed_url) && !empty($feed_url)) {
                    $feed_url .= '&maxResults=' . $results . ($pageToken ?? '') . '&key=' . $google_api_key;
                    $content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);

					$feed = "";
					if(isset($content) && ! empty($content)){
						$feed = json_decode($content);
					}
					
					if (is_object($feed) && !empty($user_feed))
						$feed->userInfo = $user_feed->items[0]->snippet;
                }
                if (isset($feed->error) && !empty($feed->error)) {
                    $feed = null;
                }
    		break;
    		case 'vimeo':
                $vimeo_access_token = $setoption['section_vimeo']['vimeo_access_token'] ?? '';
                $feedtype = 'videos';
                $feed_url = 'https://api.vimeo.com/users/' . $feed_value . '/' . $feedtype . "?per_page=$results&access_token=$vimeo_access_token";
                if (isset($_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) && $nextPage = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) {
                    $feed_url .= '&page='.$nextPage;
				}
                $content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);
				$feed = "";
				if (isset($content) && ! empty($content)) {
					$feed = json_decode($content);
				}
    		break;
    		case 'tumblr':
                $tumblr_api_key = $setoption['section_tumblr']['tumblr_api_key'] ?? '';
                $feed_url = "https://api.tumblr.com/v2/blog/" . $feed_value . ".tumblr.com/posts?api_key={$tumblr_api_key}&limit=$results";
                if (isset($_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) && $posts_start = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2])
                    $feed_url .= '&offset='.$posts_start;
                    
                $content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);

				$feed = "";
				if(isset($content) && ! empty($content)){
					$feed = json_decode($content);
				}
            break;
    		case 'deviantart':
                if ( empty($_SESSION[$sb_label]['loadcrawl']) ) {
					$feed_url = "https://backend.deviantart.com/rss.xml?type=deviation&q=by%3A" . $feed_value . "+sort%3Atime+meta%3Aall";
					$content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);
					$feed = "";
					if(isset($content) && ! empty($content)){
						libxml_use_internal_errors(true);
						$feed = simplexml_load_string(trim($content));
						libxml_clear_errors();
					}
                }
    		break;
            case 'rss':
                $is_cached = $cache->is_cached($feed_value);
				$content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_value, $feed_value) : $cache->do_curl($feed_value);
				$feed = "";
				if ( isset($content) && ! empty($content) ) {
					$content = preg_replace('/[^\x9\xa\x20-\xD7FF\xE000-\xFFFD]/', '', $content);
					libxml_use_internal_errors(true);
					$feed = simplexml_load_string( trim($content) );
					if ($feed === false) {
						$rsserrors = libxml_get_errors();
						if ( !$is_cached && !empty($rsserrors) ) {
							$rsserrormsg = '';
							foreach ($rsserrors as $rsserror) {
								$rsserrormsg .= trim( strip_tags($rsserror->message) ) . "\n";
							}
							$this->axs_log('RSS error: '.$rsserrormsg.' - ' . $feed_value);
						}
						libxml_clear_errors();
					}
				}
            break;
            case 'soundcloud':
				if ($i == 3) {
					// track search
					$feed_url = "https://api.soundcloud.com/tracks?q={$feed_value}&access=playable,preview,blocked&linked_partitioning=true";
				} else {
					// Retrieve object id (user id by username, playlist id by url)
					if ($i == 1) {
						$object_resurl = 'https://soundcloud.com/'.$feed_value;
					} elseif ($i == 2) {
						$object_value = explode("/", $feed_value);
						$object_resurl = "https://soundcloud.com/{$object_value[0]}/sets/{$object_value[1]}";
					}
					$object_permalink = urldecode($object_resurl);
					$object_url = "https://api.soundcloud.com/resolve?url={$object_permalink}";
					$is_cached = $cache->is_cached($object_url);
					$feed_headers = ( ! $is_cached || $this->forceCrawl) ? $this->soundcloud_create_headers($setoption) : null;
					$object_content = ( ! $this->forceCrawl ) ? $cache->get_data($object_url, $object_url, false, $feed_headers) : @$cache->do_curl($object_url, true, $feed_headers);
					if ( ! empty($object_content) ) {
						$object_data = json_decode($object_content);
						if ( ! empty($object_data->id) ) {
							if ($i == 1) {
								$feed_url = "https://api.soundcloud.com/users/{$object_data->id}/tracks?access=playable,preview,blocked&linked_partitioning=true";
							} else {
								$feed_url = "https://api.soundcloud.com/playlists/{$object_data->id}/tracks?access=playable,preview,blocked&linked_partitioning=true";
							}
						}

						if ( isset($object_data->errors) ) {
							$scerror = $object_data;
							$scerror_url = $object_url;
						}
					}
				}

				if ( isset($feed_url) ) {
					$feed_url .= "&limit={$results}";

					if (isset($_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) && $tracks_start = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2])
                    	$feed_url .= '&offset='.$tracks_start;

					$is_cached = $cache->is_cached($feed_url);
					if ( ! isset($feed_headers) )
						$feed_headers = ( ! $is_cached || $this->forceCrawl) ? $this->soundcloud_create_headers($setoption) : null;
					$content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url, false, $feed_headers) : $cache->do_curl($feed_url, true, $feed_headers);

					$feed = "";
					if(isset($content) && ! empty($content)){
						$feed = json_decode($content);
					}
				}

				if ( isset($feed->errors) ) {
					$scerror = $feed;
					$scerror_url = $feed_url;
				}

				if ( ! empty($scerror) ) {
					if ( ! $is_cached ) {
						$scerrors = $scerror->errors;
						if ( ! empty($scerrors[0]->error_message) )
							$scerroritem = $scerrors[0]->error_message;
						else
							$scerroritem = $scerror->status;
						$this->axs_log( 'SoundCloud error: '.$scerroritem.' - ' . $scerror_url );
					}
					$feed = null;
				}
            break;
            case 'vk':
                $pagefeed = $sboption['vk_pagefeed'] ?? 'all';
                $wall_by = ($i == 1) ? 'domain' : 'owner_id';
                $vk_service_token = $setoption['section_vk']['vk_service_token'] ?? '';
                $feed_url = "https://api.vk.com/method/wall.get?v=5.131&{$wall_by}={$feed_value}&count={$results}&extended=1&lang=en&filter={$pagefeed}&access_token={$vk_service_token}";
                if($i == 3){
                	$feed_v = explode("/",$feed_value);
					$feed_value = $feed_v[0] ?? '';
					$feed_query = $feed_v[1] ?? '';
					$wall_by = (is_int($feed_value)) ? 'owner_id' : 'domain';
					$feed_url = "https://api.vk.com/method/wall.search?v=5.131&{$wall_by}={$feed_value}&count={$results}&extended=1&lang=en&query={$feed_query}&access_token={$vk_service_token}";
				}
                if (isset($this->attr['https']) && !empty($this->attr['https']))
                    $feed_url .= '&https=1';
				if (isset($_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2]) && $offset = $_SESSION[$sb_label]['loadcrawl'][$feed_key.$i.$key2])
                    $feed_url .= '&offset='.$offset;
                else
                    $offset = 0;
                $content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);

				$feed = "";
                if(isset($content) && ! empty($content)){
					$content = mb_convert_encoding($content, "UTF-8", "auto");
					$feed = json_decode($content);
				}
                
                if (is_object($feed))
                    $feed->offset = $offset;
            break;
			case 'wordpress':
				$arg = array(
					"numberposts" => $results,
					"feed_value" => $feed_value
				);
				if ($i == 1) {
					$arg["action"] = "category";
				}
				if ($i == 2) {
					$arg["action"] = "tag";
				}
                $feed = $this->sbwordpress->GetResponse($arg);
			break;
    	}
        
    	return isset($feed) ? $feed : '';
    }
    
    /**
     * [social_board id="" type=""]
     */
    public static function social_board_shortcode( $atts ) {
    	if($atts["type"] == "masonry") $atts["type"] = "wall";
        if (isset($atts['network']) && !empty($atts['network'])) {
            $network2 = explode('{|}', $atts['network']);
            foreach ($network2 as $net2) {
                $network3 = explode('{/}', $net2);
                $network4 = explode('{;}', $network3[1]);
                
                $net5 = array();
                foreach ($network4 as $net4) {
                    $network5 = explode('{:}', $net4);
                    if (stristr($network5[0], '_id_') === FALSE) {
                        $network6 = $network5[1];
                    } else {
                        $network6 = explode('{,}', $network5[1]);
                    }
                    $net5[$network5[0]] = $network6;
                }
                
                $net3[$network3[0]] = $net5;
            }
            $atts['network'] = $net3;
        }
        $echo = (!empty($atts['echo'])) ? true : false;
        $sb = new SocialBoard();
        return $sb->init( $atts, $echo );
    }
    
    // create time string for sorting and applying pinning options
    private function make_timestr($time, $link) {
        $timestr = ( is_numeric($time) ) ? $time : strtotime($time);
        if ( ! empty($this->sboption['pins']) ) {
            $pinscount = strlen(count($this->sboption['pins']) );
            $dkey = array_search($link, $this->sboption['pins']);
            if ($dkey !== false) {
                $dkey = str_pad($dkey, $pinscount, 0, STR_PAD_LEFT);
                $timestr = "9{$dkey}";
            }
        }
		$linkstr = sprintf("%u", crc32($link) );
        return $timestr.'-'.$linkstr;
    }
    
    // applying board items removal
    private function make_remove($link) {
        if ( ! empty($this->sboption['remove']) ) {
            if ( in_array($link, $this->sboption['remove']) )
                return false;
        }
        return true;
    }

	private function make_filters( $datas = array() ) {
		if ( !is_array( $datas ) ) {
			return true;
		}

		if ( !empty( $this->sboption['filter_out'] ) )
			$filter_out = array_filter( $this->sboption['filter_out'] );
		if ( !empty( $this->sboption['filter_in'] ) )
			$filter_in = array_filter( $this->sboption['filter_in'] );

		foreach ( $datas as $data ) {
			if ( !empty( $filter_out ) ) {
				foreach ( $filter_out as $filter ) {
					if ( is_string( $data ) && !empty( $data ) ) {
						if ( strpos( mb_strtolower( $data ), mb_strtolower( $filter ) ) !== false ) {
							return false;
						}
					}
				}
			}

			if ( !empty( $filter_in ) ) {
				foreach ( $filter_in as $filter ) {
					if ( is_string( $data ) && !empty( $data ) ) {
						if ( strpos( mb_strtolower( $data ), mb_strtolower( $filter ) ) !== false ) {
							return true;
						}
					}
				}
				return false;
			}
		}

		return true;
	}
    
    /**
     * Word Limiter
     *
     * Limits a string to X number of words.
     *
     * @param   $end_char   the end character. Usually an ellipsis
     */
    function word_limiter($text, $url = '', $comment = false) {
        $limit = $comment ? ($this->attr['commentwords'] ?? '') : ($this->attr['words'] ?? '');
        $end_char = __( '...', 'social-board' );
        
        $str = trim( flame_strip_tags( $text ) );
        $str1 = trim( flame_strip_tags( $text, array('<a>') ) );
        $str1 = trim( preg_replace('#<([^ >]+)[^>]*>([[:space:]]|&nbsp;)*</\1>#', '', $str1) ); // remove all empty HTML tags
    	if ($str == '') {
    		return '';
    	}

		$words = explode(" ", $str);
        if (count($words) < $limit) {
			if ( empty($str1) ) {
				return '';
			}
            return ($str1 == $str) ? $this->append_links($str1) : trim( preg_replace('/\s*\n+/', "<br>", $str1) );
        }

        if ($limit) {
			$str = implode(" ", array_splice($words, 0, $limit));
            $str = $this->append_links($str);
        }

    	if (isset($this->attr['readmore']) && !empty($this->attr['readmore']) && $url) {
            $end_char = ' <a href="' . $url . '"'.$this->target.' style="font-size: large;">' . $end_char . '</a>';
		}

        return $str.$end_char;
    }

    // Title Limiter (limits the title of each item to X number of words)
    function title_limiter($str, $url = '') {
        $end_char = '...';
        $limit = $this->attr['titles'] ?? 15;
        $str = strip_tags( $str );

    	if (trim($str) == '')
    	{
    		return $str;
    	}
        
		$words = explode(" ", $str);
        if (count($words) < $limit) {
            return $str;
        }

    	preg_match('/^\s*+(?:\S++\s*+){1,'.(int) $limit.'}/', $str, $matches);

        if (strlen($str) == strlen($matches[0]))
    	{
    		$end_char = '';
    	}

        return rtrim($matches[0]).$end_char;
    }
    
    function append_links($str) {
		if ( empty($str) ) {
			return '';
		}
        // make the urls hyper links
        $regex = '#\bhttps?://[^\s()<>]+(?:\([\w\d]+\)|([^[:punct:]\s]|/))#';
        $str = preg_replace_callback($regex, array(&$this, 'links_callback'), $str);
        return trim(preg_replace('/\s*\n+/', "<br>", $str) );
    }
    
    function links_callback($matches) {
        return (!empty($matches[0])) ? '<a href="'.$matches[0].'"'.$this->target.'>'.$matches[0].'</a>' : '';
    }
    
    function format_text($str) {
        $str = trim( flame_strip_tags( $str ) );
    	if ($str == '') {
    		return $str;
    	}
        
        $str = $this->append_links( $str );
        return $str;
    }
    
    // Embed url
    function get_embed($embedUrl, $enforce_custom_code = false) {
        $embedCache = new AXS_HtmlEmbedCache;
        $embedCache->debug_log = $this->attr['debuglog'] ?? '';
        $embedCache->timeout = $this->attr['timeout'] ?? '';
        $embedCache->cache_path = AXS_DIRNAME . '/cache/';
        if ($embedResult = $embedCache->getData($embedUrl) ) {
            // If no provider found, create a html block
            if ( ! empty($embedResult->code) && ! $enforce_custom_code ) {
                $embed = '<div class="sb-html-embed">';
                $embed .= $embedResult->code;
                $embed .= '</div>';
            } else {
                if ($embedResult->title || $embedResult->description) {
                    $embed = '<a class="sb-html-embed" href="' . $embedUrl . '" target="_blank">';
                    $embed .= '
                    <div class="sb-embed-user">';
                    if ( ! empty($embedResult->providerIcon) ) {
                        $embed .= '<div class="sb-embed-uthumb"><img class="sb-img" alt="' . ($embedResult->providerName ?? '') . '" src="' . $embedResult->providerIcon . '"></div>';
                    }
                    $author = $embedResult->authorName ?? ($embedResult->providerName ?? '');
                    $embed .= '<div class="sb-embed-uinfo">' . $author . '</div>
                    </div>';
                    if ( ! empty($embedResult->image) )
                        $embed .= '<div class="sb-thumb"><img class="sb-img sb-html-embed-image" src="' . $embedResult->image . '" alt="' . $author . '" /></div>';
                    if ( ! empty($embedResult->title) )
                        $embed .= '<span class="sb-title">' . $embedResult->title . '</span>';
                    if ( ! empty($embedResult->description) )
                        $embed .= '<span class="sb-text">' . $embedResult->description . '</span>';
                    $embed .= '</a>';
                }
            }

            return $embed;
        }
    }

    function twitter_add_links($text, $embed_urls = null, $referenced_tweet = null) {
        $text = $this->add_links('twitter', $text);
        
        // Convert shared links to Embed code
        if ( ! empty($embed_urls) ) {
            $embedList = array();
			foreach ($embed_urls as $url) {
				if (isset($url->status) && $url->status == '200') {
					$urls[] = $url->url;
					$exurls[] = $url->expanded_url;
					$infourls[] = $url;
				}
            }
			if ( empty($urls) ) {
				return $text;
			}

            $regexp = "<a\s[^>]*href=(\"??)([^\" >]*?)\\1[^>]*>(.*)<\/a>";
            if (preg_match_all("/$regexp/siU", $text, $matches, PREG_SET_ORDER) ) {
                foreach($matches as $match) {
                    $embed_key = array_search($match[3], $urls);
                    if ( is_numeric( $embed_key) )
                    if ( !empty($exurls[$embed_key]) ) {
						if (!empty($referenced_tweet)) {
							$embedList[$embed_key] = array(
								'surl' => $match[0],
								'url' => $exurls[$embed_key],
								'description' => $referenced_tweet->text,
								'user' => $referenced_tweet->user
							);
						} else {
							$embed_image = !empty($infourls[$embed_key]->images) ? $infourls[$embed_key]->images[1]->url : '';
							$embedList[$embed_key] = array(
								'surl' => $match[0],
								'url' => $exurls[$embed_key],
								'title' => $infourls[$embed_key]->title,
								'description' => $infourls[$embed_key]->description,
								'image' => $embed_image
							);
						}
                    }
                }
            }

            foreach ($embedList as $embedItem) {
				$embed = $this->twitter_embed($embedItem);
				$text = str_replace($embedItem['surl'], $embed, $text);
            }
        }

        return $text;
    }

	function twitter_embed($item) {
		$embed = '<a class="sb-html-embed" href="' . $item['url'] . '" target="_blank">';
		if ( ! empty($item['user']) ) {
			$embed .= '
			<div class="sb-embed-user">';
			$embed .= '<div class="sb-embed-uthumb"><img class="sb-img" alt="' . ($item['user']->username ?? '') . '" src="' . $item['user']->profile_image_url . '"></div>';
			$embed .= '<div class="sb-embed-uinfo">' . $item['user']->name . '</div>
			</div>';
		}
		if ( ! empty($item['image']) ) {
			$embed .= '<div class="sb-thumb"><img class="sb-img sb-html-embed-image" src="' . $item['image'] . '" alt="" /></div>';
		}
		if ( ! empty($item['title']) ) {
			$embed .= '<span class="sb-title">' . $item['title'] . '</span>';
		}
		if ( ! empty($item['description']) ) {
			$embed .= '<span class="sb-text">' . $item['description'] . '</span>';
		}
		$embed .= '</a>';

		return $embed;
	}
    
    function twitter_get_feed($rest, $params, $cache, $setoption, $i) {
        $url_scr_service = 'https://api.axentmedia.com/twitter-scr-service';
        $twitter_app = trim($setoption['section_twitter']['twitter_app'] ?? '');
        if ($twitter_app == 'yes') {
			$key = $setoption['section_licensing']['license_key'];
			$domain = preg_replace('/((^https?:\/\/)?(www\.)?)|(\/$)/', '', strtolower(trim(get_site_url(), '/')));
			$oauth_access_token = trim($setoption['section_twitter']['twitter_access_token'] ?? '');
			$oauth_access_token_secret = trim($setoption['section_twitter']['twitter_access_token_secret'] ?? '');
			$params = urlencode(serialize($params));
            $feed_url = $url_scr_service.'/?key='.$key.'&domain='.$domain.'&endpoint='.$rest.'&params='.$params;
			// $feed_url = 'https://api.axentmedia.com/twitter-basic-service/?key='.$key.'&domain='.$domain.'&endpoint='.$rest.'&access_token='.$oauth_access_token.'&access_token_secret='.$oauth_access_token_secret.'&params='.$params;
			$is_cached = $cache->is_cached($feed_url);
			$content = ( ! $this->forceCrawl ) ? $cache->get_data($feed_url, $feed_url) : $cache->do_curl($feed_url);
			$feed = "";
			if (isset($content) && !empty($content)) {
				$feed = json_decode($content);
				if ( isset( $feed->error ) OR isset( $feed->account_id ) ) {
					if (isset($this->attr['debuglog']) && !empty($this->attr['debuglog']) && !$is_cached) {
						if ( isset( $feed->detail ) ) {
							axs_log( 'Twitter - '.$feed->detail.' - ' . $rest);
						} else {
							axs_log( 'Twitter error: '.$feed->error.' - ' . $rest);
						}
					}
					$feed = null;
				}
			}
        } else {
            $consumer_key = trim($setoption['section_twitter']['twitter_api_key'] ?? '');
            $consumer_secret = trim($setoption['section_twitter']['twitter_api_secret'] ?? '');
			$oauth_access_token = trim($setoption['section_twitter']['twitter_access_token'] ?? '');
			$oauth_access_token_secret = trim($setoption['section_twitter']['twitter_access_token_secret'] ?? '');
			$get_feed = TRUE;
			$label = 'https://api.twitter.com/2/'.$rest.'/'.serialize($params);
			if ( ! $this->forceCrawl ) {
				if ( $cache->is_cached($label) ) {
					$content = $cache->get_cache($label);
					$get_feed = FALSE;
				}
			}
			if ($get_feed) {
				$auth = new TwitterOAuth($consumer_key, $consumer_secret, $oauth_access_token, $oauth_access_token_secret);
				$auth->setApiVersion('2');
				$auth->setTimeouts($this->attr['timeout'], $this->attr['timeout']);
				$auth->setDecodeJsonAsArray(false);
				if ( ! empty($setoption['proxysetting']['proxy']) ) {
					$auth->setProxy([
						'CURLOPT_PROXY' => trim($setoption['proxysetting']['proxy'] ?? ''),
						'CURLOPT_PROXYPORT' => trim($setoption['proxysetting']['proxy_port'] ?? ''),
						'CURLOPT_PROXYUSERPWD' => trim($setoption['proxysetting']['proxy_userpass'] ?? '')
					]);
				}
				$content = $auth->get($rest, $params);
				if ( ! $content ) {
					if (isset($this->attr['debuglog']) && !empty($this->attr['debuglog'])) {
						axs_log( 'Twitter error code: '. $auth->getLastHttpCode() . ' - Please check your API credentials or feed settings.' );
					}
				} else {
					$feed = $content;
					if ( isset( $feed->errors ) ) {
						foreach( $feed->errors as $key => $val ) {
							if (isset($this->attr['debuglog']) && !empty($this->attr['debuglog'])) {
								axs_log( 'Twitter error: '.$val->message.' - ' . $rest);
							}
						}
						$feed = null;
					}
				}
				if ( ! $this->forceCrawl ) {
					$cache->set_cache($label, json_encode($content) );
				}
			} else {
				$feed = json_decode($content ?? '');
			}
		}

        return $feed ?? '';
    }
    
    function vk_get_photo($photo,$image_type = "x", $type = "") {
		$key_type = "type";
		if (is_numeric($image_type)){
			switch ($image_type){
				case "75":
					$image_type = "s";
					break;
				case "130":
					$image_type = "m";
					break;
				case "604":
					$image_type = "x";
					break;
				case "807":
					$image_type = "y";
					break;
				case "1280":
					$image_type = "z";
					break;
				case "2560":
					$image_type = "w";
					break;
				default:
					$image_type = "x";
					break;
			}
		}

		if($type == "video"){
			$key_type = "height";
			switch ($image_type){
				case "s":
					$image_type = "96";
					break;
				case "m":
					$image_type = "120";
					break;
				case "x":
					$image_type = "240";
					break;
				case "y":
					$image_type = "450";
					break;
				case "z":
					$image_type = "720";
					break;
				case "w":
					$image_type = "585";
					break;
				default:
					$image_type = "450";
					break;
			}
		}

		if(isset($photo) && !empty($photo)){
			foreach ($photo as $k => $v){
				if($v->{$key_type} == $image_type){
					return $v;
				}
				if((count($photo) - 1) == $k){
					return $v;
				}
			}
		}
		
    }
    
    // Find a Youtube video link in a string and convert it into Embed Code
	function youtube_get_embedurl($url) {
		preg_match("#(http://www.youtube.com)?/(v/([-|~_0-9A-Za-z]+)|watch\?v\=([-|~_0-9A-Za-z]+)&?.*?)#i", $url, $matches);
		foreach ($matches as $matche) {
			if ( ! empty($matche) && strpos($matche, '/' ) === FALSE && strpos($matche, '=' ) === FALSE) {
				$vidID = $matche;
				break;
			}
		}
	    return 'https://www.youtube.com/embed/' . $vidID ;
    }

    function facebook_get_comments($comment, $link, $sublevel = false) {
        $data = '';
        $comment_message = (!empty($this->attr['commentwords'])) ? $this->word_limiter( nl2br($comment->message), $link , true ) : nl2br($comment->message);
        $nocommentimg = (empty($comment->from) ) ? ' sb-nocommentimg' : '';
        $data .= '<span class="sb-meta sb-mention' . $nocommentimg . ($sublevel ? ' sb-subcomment' : '') . '">';
        if ( isset($comment->from) && ! empty($comment->from) ) {
            $commentPicture = isset($comment->from->picture->data) ? $comment->from->picture->data : $comment->from->picture;
            $commenterLink = isset($comment->from->link) ? $comment->from->link : 'https://www.facebook.com/'.$comment->from->id;
            $comment_picture = $commentPicture->url ?? $this->protocol.'://graph.facebook.com/' . $comment->from->id . '/picture?type=square';
            $data .= '<img class="sb-img sb-commentimg" src="'.$comment_picture.'" alt="' . ($comment->from->name ?? '') . '" /><a href="' . ($comment->permalink_url ?? $commenterLink) . '"' . $this->target . '>' . ($comment->from->name ?? '') . '</a> ';
        }
        $data .= $comment_message . '</span>';

        return $data;
    }
    
    // Check if the string is a real Instagram access token
    function instagram_is_access_token($token) {
        $tokenArr = explode('.', $token);
        if ( count($tokenArr) == 3 ) {
            if ( is_numeric($tokenArr[0]) && ! is_numeric($tokenArr[2]) && strlen($tokenArr[2]) == 32 )
                return true;
        }
        return false;
    }

    // Find Instagram access token
    function instagram_access_token($feed_value) {
        $feed_token = '';
        $instagram_access_token = ! empty($this->setoption['section_instagram']['instagram_accounts'])
            ? reset($this->setoption['section_instagram']['instagram_accounts'])['access_token'] : ($this->setoption['section_instagram']['instagram_access_token'] ?? '');
        if ($this->instagram_is_access_token($feed_value) ) {
            $feed_token = $feed_value;
        } elseif ($feed_value == 'self') {
            $feed_token = $instagram_access_token;
        } elseif ( ! empty($this->setoption['section_instagram']['instagram_accounts']) ) {
            // If access token found for this account by ID
            if ( isset($this->setoption['section_instagram']['instagram_accounts'][$feed_value]) ) {
                $feed_token = $this->setoption['section_instagram']['instagram_accounts'][$feed_value]['access_token'];
            } else {
                // If not found, search by username
                foreach ($this->setoption['section_instagram']['instagram_accounts'] as $iaccount) {
                    if ( isset($iaccount['username']) && $iaccount['username'] == $feed_value ) {
                        $feed_token = $iaccount['access_token'];
                        break;
                    }
                }
            }
        }

        return $feed_token;
    }

	function soundcloud_create_headers($setoption) {
		$soundcloud_access_token = $setoption['section_soundcloud']['soundcloud_access_token'] ?? '';
		$is_expired = time() > ($setoption['section_soundcloud']['soundcloud_updated_at'] + $setoption['section_soundcloud']['soundcloud_expires_in']);
		if ($is_expired) {
			$body = array(
				'refresh' => 'refresh',
				'token'   => $setoption['section_soundcloud']['soundcloud_refresh_token']
			);
			if ($setoption['section_soundcloud']['soundcloud_app'] == 'no') {
				$body['api_key']    = $setoption['section_soundcloud']['soundcloud_api_key'];
				$body['api_secret'] = $setoption['section_soundcloud']['soundcloud_api_secret'];
			}
			$get_token = wp_remote_post( 'https://axentmedia.com/soundcloud-access-token/', array(
				'method'  => 'POST',
				'body' 	  => $body,
				'timeout' => 30
				)
			);

			if ( is_wp_error( $get_token ) ) {
				$error_message = $get_token->get_error_message();
				$this->axs_log( 'SoundCloud error: '.$error_message.' - wp_error' );
			} else {
				$response_code = wp_remote_retrieve_response_code( $get_token );
				if ( ! $response_code == 200 ) {
					$this->axs_log( 'SoundCloud error: '.$get_token['response']['message'].' - response_code' );
				} else {
					$res_body = json_decode($get_token['body']);
					if ( isset($res_body->access_token) ) {
						$soundcloud_access_token = $res_body->access_token;
						// Update token
						$setoptions = get_option( "AXS_Settings_Page" );
						$setoptions['section_soundcloud']["soundcloud_access_token"] = $res_body->access_token;
						$setoptions['section_soundcloud']["soundcloud_refresh_token"] = $res_body->refresh_token;
						$setoptions['section_soundcloud']["soundcloud_expires_in"] = $res_body->expires_in;
						$setoptions['section_soundcloud']["soundcloud_updated_at"] = time();
						update_option( "AXS_Settings_Page", $setoptions );
					} else {
						$this->axs_log( 'SoundCloud error: '.$res_body->message.' - res_body' );
					}
				}
			}
		}

		// Create headers
		return array(
			"Authorization: OAuth {$soundcloud_access_token}",
			"Accept: application/json; charset=utf-8"
		);
	}

    function instagram_parse_comments($comments_data, $link = '') {
        $comments_out = '';
        $count = 0;
        $comments_count = $this->sboption['section_instagram']['instagram_comments'] ?? 0;
        if ( $comments_count && isset($comments_data->data) && is_array($comments_data->data) ) {
			foreach ($comments_data->data as $comment) {
				if ( ! empty($comment->text) ) {
					$count++;
					$comment_message = (!empty($this->attr['commentwords'])) ? $this->word_limiter($comment->text, $link, true) : $comment->text;
					$nocommentimg = (empty($comment->from->profile_picture) ) ? ' sb-nocommentimg' : '';
					$comment_date = (!empty($comment->created_time)) ? 'title="'.axs_friendly_date( $comment->created_time, 'datetime' ).'"' : '';
					$comments_out .= '<span class="sb-meta sb-mention'.$nocommentimg.'"'.$comment_date.'>';
					if ( isset($comment->from) && ! empty($comment->from) ) {
						if (isset($comment->from->profile_picture) && !empty($comment->from->profile_picture))
							$comments_out .= '<img class="sb-img sb-commentimg" src="' . $comment->from->profile_picture . '" alt="' . htmlspecialchars ( $comment_message) . '">';
						$name = $comment->from->full_name ?? $comment->from->username;
						$comments_out .= '<a href="https://instagram.com/' . $comment->from->username . '"'.$this->target.'>' . $name . '</a> ';
					}
					else if(isset($comment->username) && !empty($comment->username) ){
						$comments_out .= '<a href="https://instagram.com/' . $comment->username . '"'.$this->target.'>' . $comment->username . '</a> ';
					}
					$comments_out .= $comment_message . '</span>';
					if ( $count >= $comments_count ) break;
				}
			}
		}
        return $comments_out;
    }

    function add_links($endpoint, $text) {
		if ( empty($text) ) {
			return '';
		}

        $endpoints = array(
            'twitter' => array(
                'https://twitter.com/$1',
                'https://twitter.com/search?q=%23$1'
            ),
            'instagram' => array(
                'https://www.instagram.com/$1/',
                'https://www.instagram.com/explore/tags/$1/'
            ),
            'facebook' => array(
                'https://www.facebook.com/$1',
                'https://www.facebook.com/hashtag/$1/'
            )
        );
        // Add links to all @ mentions
        $text = preg_replace('/@+([a-zA-Z0-9_]+)/', '<a href="'.$endpoints[$endpoint][0].'"'.$this->target.'>$0</a>', $text);
        // Add links to all hash tags
        $text = preg_replace('/#+([a-zA-Z0-9_]+)/', '<a href="'.$endpoints[$endpoint][1].'"'.$this->target.'>$0</a>', $text);
        
        return $text;
    }

    function youtube_parse_comments($comments_data, $link = '') {
        $comments_out = '';
        $count = 0;
        $comments_count = $this->sboption['section_youtube']['youtube_comments'] ?? 0;
        if ( ! isset($this->sboption['section_youtube']['youtube_comments']) )
            $comments_count = 3;
        if ($comments_count)
        foreach ( $comments_data->items as $comment ) {
            $count++;
            $snippet = $comment->snippet->topLevelComment->snippet;
            $comment_message = (!empty($this->attr['commentwords'])) ? $this->word_limiter(nl2br($snippet->textOriginal), $link, true) : nl2br($snippet->textDisplay);
            $nocommentimg = (empty($snippet->authorDisplayName) ) ? ' sb-nocommentimg' : '';
            $comment_date = $snippet->updatedAt ?? $snippet->publishedAt ?? '';
            $comments_out .= '<span class="sb-meta sb-mention'.$nocommentimg.'" title="'.axs_friendly_date( $comment_date, 'datetime' ).'">';
            if ( isset($snippet->authorDisplayName) && ! empty($snippet->authorDisplayName) ) {
                $comments_out .= '<img class="sb-img sb-commentimg" src="'.$snippet->authorProfileImageUrl.'" alt="' . htmlspecialchars ( $comment_message) . '" /><a href="' . $snippet->authorChannelUrl . '"'.$this->target.'>' . $snippet->authorDisplayName . '</a> ';
            }
            $comments_out .= $comment_message . '</span>';
            if ( $count >= $comments_count ) break;
        }
        return $comments_out;
    }
    
    // create the ajax callback for comments
    function axs_fetchcomments() {
        if ( ! session_id() ) {
            session_start();
        }
        if ( ! wp_verify_nonce( $_REQUEST['nonce'], "fetchcomments")) {
            exit("No naughty business please!");
        }

        if ( ! empty($_SERVER['HTTP_X_REQUESTED_WITH'])
            && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            
            $sb = new SocialBoard();
            $sb->fetchcomments = true;
            $sb->init( $_REQUEST['attr'] );

            if ($_REQUEST['id'] && $_REQUEST['feed']) {
                $id = $_REQUEST['id'];
                $feed = explode('-', $_REQUEST['feed']);
				$feedclass = trim($feed[0]);
                // fetch comments from API
                switch ($feedclass) {
                    case 'instagram':
						// only if Instagram user feed
						if ( $feed[1] == 1 ) {
							$feed_value = $sb->sboption['section_instagram']['instagram_id_' . $feed[1]][$feed[2]];
							if ( is_array($feed_value) ) {
								$feed_value = $feed_value['feed'];
							}
							
							if ( $feed_account = $sb->sbinstagram->GetAccount( $feed_value ) ) {
								$feed_value = $feed_account['id'];
								$feed_token = $feed_account['access_token'];
								if ( !empty( $feed_token ) ) {
									$feed_url = 'https://graph.facebook.com/v14.0/' . $id . '/comments?fields=id,text,timestamp,like_count,media,replies,user{id,username,name,profile_picture_url},username&access_token=' . $feed_token;
								}
							}
						}
                    break;
                    case 'youtube':
                        $google_api_key = $sb->setoption['section_google']['google_api_key'] ?? '';
                        $feed_url = 'https://www.googleapis.com/youtube/v3/commentThreads?videoId=' . $id . '&part=snippet,replies&maxResults=' . $sb->attr['results'] . '&key=' . $google_api_key;
                    break;
				}
				
                if ( ! empty($feed_url) ) {
                    // if other public instagram user account without a $feed_token

					$content = ( !$sb->forceCrawl ) ? $sb->cache->get_data( $feed_url, $feed_url ) : $sb->cache->do_curl( $feed_url );
					$comments_feed =  ( isset( $content ) && !empty( $content )) ? json_decode( $content ) : '';
					
                    if ( ! empty($comments_feed) ) {
                        $feedfunc = $feedclass.'_parse_comments';
                        echo $sb->$feedfunc($comments_feed, $_REQUEST['link']);
                    }
                }
            }
        } else {
            header("Location: ".$_SERVER["HTTP_REFERER"]);
        }
        
        die();
    }

    // get js assets position
    function js_assets_position() {
		$js_assets_position = $this->setoption['section_layouts']['js_assets_position'] ?? '';
        if ( ! $js_assets_position )
            $js_assets_position = array('footer' => true, 'order' => 10);
        return $js_assets_position;
	}
	
	function axs_log($message) {
		if (isset($this->attr['debuglog']) && !empty($this->attr['debuglog'])) {
			axs_log( $message );
		}
	}
} // end class

// create the ajax callback for tabable widget
add_action("wp_ajax_sb_tabable", "axs_tabable");
add_action("wp_ajax_nopriv_sb_tabable", "axs_tabable");
function axs_tabable() {
    if ( ! session_id() ) {
        session_start();
    }
    if ( ! wp_verify_nonce( $_REQUEST['nonce'], "tabable")) {
        exit("No naughty business please!");
    }

    if ( ! empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        $sb = new SocialBoard();
        $sb->init( $_REQUEST['attr'], true, null, array( array($_REQUEST['feed'] => 1) ) );
    } else {
        header("Location: ".$_SERVER["HTTP_REFERER"]);
    }
    
    die();
}

// create the ajax callback for load more
add_action("wp_ajax_sb_loadmore", "axs_loadmore");
add_action("wp_ajax_nopriv_sb_loadmore", "axs_loadmore");
function axs_loadmore() {
    if ( ! session_id() ) {
        session_start();
    }
    if ( ! wp_verify_nonce( $_REQUEST['nonce'], "loadmore")) {
        exit("No naughty business please!");
    }
    
    if ( ! empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        $sb = new SocialBoard();
        $sb->init( $_REQUEST['attr'], true, null, 'all', $_SESSION[$_REQUEST['label']]['loadmore'] );
    } else {
        header("Location: ".$_SERVER["HTTP_REFERER"]);
    }
    
    die();
}

// create the ajax callback for live update
add_action("wp_ajax_sb_liveupdate", "axs_liveupdate");
add_action("wp_ajax_nopriv_sb_liveupdate", "axs_liveupdate");
function axs_liveupdate() {
    if ( ! session_id() ) {
        session_start();
    }
    if ( ! wp_verify_nonce( $_REQUEST['nonce'], "liveupdate")) {
        exit("No naughty business please!");
    }

    if ( ! empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        $args = array('liveresults' => $_REQUEST['results'] ?? '');
        $sb = new SocialBoard();
        $sb->init( $_REQUEST['attr'], true, $args, 'all', array() );
    } else {
        header("Location: ".$_SERVER["HTTP_REFERER"]);
    }
    
    die();
}

// Friendly dates (i.e. "2 days ago")
function axs_friendly_date( $date, $format = 'friendly' ) {
	// Make sure the server is using the right time zone
	if ( get_option( 'timezone_string' ) && strpos( get_option( 'timezone_string' ), 'UTC' ) === false )
		date_default_timezone_set( get_option( 'timezone_string' ) );

	// Get the time difference in seconds
	$post_time = ( is_numeric($date) ) ? $date : strtotime( $date );
    if ( strlen($post_time) > 10 ) $post_time = substr( $post_time, 0, -3 );

    $full_time = date_i18n( get_option( 'date_format' ).' '.get_option( 'time_format' ), $post_time );

    if ($format == 'friendly') {
        
        $current_time = time();
        $time_difference = $current_time - $post_time;
        
        // Seconds per...
        $minute = 60;
        $hour = 3600;
        $day = 86400;
        $week = $day * 7;
        $month = $day * 31;
        $year = $day * 366;
        
        // if over 3 years
        if ( $time_difference > $year * 3 ) {
            $friendly_date = __( 'a long while ago', 'social-board' );
        }
        
        // if over 2 years
        else if ( $time_difference > $year * 2 ) {
            $friendly_date = __( 'over 2 years ago', 'social-board' );
        }
        
        // if over 1 year
        else if ( $time_difference > $year ) {
            $friendly_date = __( 'over a year ago', 'social-board' );
        }
        
        // if over 11 months
        else if ( $time_difference >= $month * 11 ) {
            $friendly_date = __( 'about a year ago', 'social-board' );
        }
        
        // if over 2 months
        else if ( $time_difference >= $month * 2 ) {
            $months = (int) $time_difference / $month;
            $friendly_date = sprintf( __( '%d months ago', 'social-board' ), $months );
        }
        
        // if over 4 weeks ago
        else if ( $time_difference > $week * 4 ) {
            $friendly_date = __( 'last month', 'social-board' );
        }
        
        // if over 3 weeks ago
        else if ( $time_difference > $week * 3 ) {
            $friendly_date = __( '3 weeks ago', 'social-board' );
        }
        
        // if over 2 weeks ago
        else if ( $time_difference > $week * 2 ) {
            $friendly_date = __( '2 weeks ago', 'social-board' );
        }
        
        // if equal to or more than a week ago
        else if ( $time_difference >= $day * 7 ) {
            $friendly_date = __( 'last week', 'social-board' );
        }
        
        // if equal to or more than 2 days ago
        else if ( $time_difference >= $day * 2 ) {
            $days = (int) $time_difference / $day;
            $friendly_date = sprintf( __( '%d days ago', 'social-board' ), $days );
        }
        
        // if equal to or more than 1 day ago
        else if ( $time_difference >= $day ) {
            $friendly_date = __( 'yesterday', 'social-board' );
        }
        
        // 2 or more hours ago
        else if ( $time_difference >= $hour * 2 ) {
            $hours = (int) $time_difference / $hour;
            $friendly_date = sprintf( __( '%d hours ago', 'social-board' ), $hours );
        }
        
        // 1 hour ago
        else if ( $time_difference >= $hour ) {
            $friendly_date = __( 'an hour ago', 'social-board' );
        }
        
        // 2–59 minutes ago
        else if ( $time_difference >= $minute * 2 ) {
            $minutes = (int) $time_difference / $minute;
            $friendly_date = sprintf( __( '%d minutes ago', 'social-board' ), $minutes );
        }
        
        else {
            $friendly_date = __( 'just now', 'social-board' );
        }

        $dateout = '<time title="' . $full_time . '" datetime="' . date( 'c', $post_time ) . '">' . ucfirst( $friendly_date ) . '</time>';
    } elseif ($format == 'date') {
        $dateout = '<time title="' . $full_time . '" datetime="' . date( 'c', $post_time ) . '">' . date_i18n( get_option( 'date_format' ), $post_time ) . '</time>';
    } elseif ($format == 'datetime') {
        $dateout = $full_time;
    }
	
	return $dateout;
}

// i18n dates
function axs_i18n_date( $date, $format ) {
    $post_time = ( is_numeric($date) ) ? $date : strtotime( $date );
    if ( strlen($post_time) > 10 ) $post_time = substr( $post_time, 0, -3 );
    return date_i18n( $format, $post_time );
}

function countFormatter($digit) {
    if ($digit >= 1000000000) {
        return round($digit/ 1000000000, 1). 'G';
    }
    if ($digit >= 1000000) {
        return round($digit/ 1000000, 1).'M';
    }
    if ($digit >= 1000) {
        return round($digit/ 1000, 1). 'K';
	}
    return $digit;
}

function axs_explode( $output = array() ) {
    if ( ! empty($output) ) {
        $outputArr = explode(',', str_replace(' ', '', $output) );
        foreach ($outputArr as $val)
            $out[$val] = true;
        
        return $out;
    }
    return false;
}

function axs_options( $post_id ) {
    if ( ! is_numeric($post_id) ) {
        $args = array(
            'meta_key' => 'classname',
            'meta_value' => $post_id,
            'post_type' => 'sb_themes',
            'posts_per_page' => 999999
        );
        $posts = get_posts($args);
        $post_id = $posts[0]->ID;
    }
    
    // 1. To retrieve the meta box data - get_post_meta( $post->ID ) will return an array of all the meta field values.
    // or if you know the field id of the value you want, you can do $value = get_post_meta( $post->ID, $field_id, true );
    $postData = array();
    // This way, array will be unserialized; easier to view.
    foreach( ( array ) get_post_custom_keys( $post_id ) as $sKey ) {
        $postData[ $sKey ] = get_post_meta( $post_id, $sKey, true );
    }
    
    return $postData;
}

// display different items according to user selections in admin widget
function axs_admin_inline_script() {
    if ( wp_script_is( 'jquery', 'done' ) ) {
    
    if ( $sb_widget = get_option('widget_sb_widget') ){
    ?>
    <script type="text/javascript">
    var sb_widget_updated = function($) {
        <?php foreach ($sb_widget as $widgetKey => $widgetItem) {
        if ( is_int($widgetKey) ) {
            ?>
            var widgetElement = '[name="widget-sb_widget[<?php echo $widgetKey; ?>]';
            <?php
            if ($widgetItem['position'] == 'normal'){ ?>
                $(widgetElement+'[width]"]').val('');
                $(widgetElement+'[location]"]').closest("tr").hide();
                $(widgetElement+'[slide]"]').closest("tr").hide();
                $(widgetElement+'[autoclose]"]').closest("tr").hide();
            <?php } ?>
            var widget = $( '[id$=sb_widget-<?php echo $widgetKey; ?>]' );
            sb_widget_update(null, widget);
        <?php }
        } ?>
    }
    
    jQuery(document).ready(function($) {
        sb_widget_updated($);
    });
    
    function sb_widget_update(e, widget) {
        // selective
        widget.on('change', 'input[name$="[position]"]', function() {
            if ( jQuery(this).val() == 'normal' ) {
                widget.find('input[name$="[width]"]').val('');
                widget.find('select[name$="[location]"]').closest("tr").fadeOut('fast');
                widget.find('input[name$="[slide]"]').closest("tr").fadeOut('fast');
                widget.find('input[name$="[autoclose]"]').closest("tr").fadeOut('fast');
            } else {
                widget.find('input[name$="[width]"]').val('280');
                widget.find('select[name$="[location]"]').closest("tr").fadeIn('fast');
                widget.find('input[name$="[tabable]"]').closest("tr").fadeIn('fast');
                widget.find('input[name$="[slide]"]').closest("tr").fadeIn('fast');
                widget.find('input[name$="[autoclose]"]').closest("tr").fadeIn('fast');
            }
        });
        
        widget.on('change', 'select[name$="[location]"]', function() {
            var ischecked = true;
            if ( jQuery( this ).val() == 'sb-left' || jQuery( this ).val() == 'sb-right' ) {
                var ischecked = false;
            }
            widget.find('input[name$="[showheader]"]').prop('checked', ischecked);
        });
        widget.on('change', 'input[name$="[slide]"]', function() {
            var ischecked = jQuery( this ).prop('checked');
            widget.find('input[name$="[autoclose]"]').prop('disabled', ! ischecked);
        });
        widget.on('change', 'input[name$="[carousel]"]', function() {
            var ischecked = jQuery( this ).prop('checked');
            widget.find('input[name$="[tabable]"]').prop('disabled', ischecked);
            widget.find('input[name$="[override_settings]"]').val( (ischecked) ? 'cs_item=1&cs_rows=1' : '' );
        });
    }

    // "widget" represents jQuery object of the affected widget's DOM element
    jQuery(document).on('widget-updated', sb_widget_update);
    jQuery(document).on('widget-added', sb_widget_update);
    </script>
    <?php } ?>
    
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        // clear cache
        $(".sb-clearcache").on("click", function() {
          var nonce = $(this).attr("data-nonce");
          var log = $(this).attr("data-log");
          var msgid = (log) ? 'clearlog' : 'clearcache';
          $('#sb-'+msgid+'-msg').html('<span class="sb-loading"></span>');
          $.ajax({
            type: "post",
            url: "<?php echo admin_url( 'admin-ajax.php' ); ?>",
            data: {action: "sb_clearcache", nonce: nonce, log: log},
            cache: false
            })
            .done(function( response ) {
                $("#sb-"+msgid+"-msg").html(response);
            })
            .fail(function() {
                alert( "Problem cleaning the cache!" );
          });
        });
    });
    </script>
    <?php
    }
}
add_filter('admin_footer', 'axs_admin_inline_script');

// create the ajax callback for clear cache
add_action("wp_ajax_sb_clearcache", "axs_clearcache");
add_action("wp_ajax_nopriv_sb_clearcache", "axs_clearcache");
function axs_clearcache() {
    if ( ! wp_verify_nonce( $_REQUEST['nonce'], "clearcache")) {
        exit("No naughty business please!");
    }

    if ( ! empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        // delete cached files
        if (isset($_REQUEST['log']) && !empty($_REQUEST['log'])) {
            $file = AXS_DIRNAME . '/sb.log';
            @unlink($file);
        } else {
            $path = "/cache/";
            $cachefiles = @glob(AXS_DIRNAME . $path ."*");
            // Delete everything except index.html through the loop
            $cachefiles = array_values( preg_grep( '/^((?!index.html).)*$/', $cachefiles ) );
            if ( ! empty($cachefiles) ) {
                foreach($cachefiles as $file) {
                    @unlink($file);
                }
            }
        }
        $cache = (isset($_REQUEST['log']) && !empty($_REQUEST['log'])) ? 'Log' : 'Cache';
        echo '<span style="color:green;">'.$cache.' cleared successfully.</span>';
    }
    else {
        header("Location: ".$_SERVER["HTTP_REFERER"]);
    }
    
    die();
}

// Plugin activation
function axs_activate() {
    require_once AXS_DIRNAME . "/admin/SB_Install.php";
    $version = get_option("sb_version");
    if ( ! $version ) {
        add_option("sb_version", AXS_VERSION, '', 'no');
        axs_data_insert();
    } elseif ( version_compare($version, AXS_VERSION, '<') ) {
		require_once AXS_DIRNAME . "/library/class_upgrade.php";
		new \AXS_AX_SOCIAL\UPGRADE\AXS_Upgrade();
		
        update_option("sb_version", AXS_VERSION);
        axs_data_update();
    }
}
register_activation_hook(__FILE__, 'axs_activate');

if (!isset($_GET['action']) || $_GET['action'] != 'deactivate') {
    add_action('admin_init', 'axs_activate');
}

// hex to rgb numerical converter for color styling
function axs_hex2rgb($hex, $str = true) {
   $hex = str_replace("#", "", $hex);

   if(strlen($hex) == 3) {
      $r = hexdec(substr($hex,0,1).substr($hex,0,1));
      $g = hexdec(substr($hex,1,1).substr($hex,1,1));
      $b = hexdec(substr($hex,2,1).substr($hex,2,1));
   } else {
      $r = hexdec(substr($hex,0,2));
      $g = hexdec(substr($hex,2,2));
      $b = hexdec(substr($hex,4,2));
   } 
   // returns the rgb values separated by commas OR returns an array with the rgb values
   $rgb = ($str) ? "$r, $g, $b" : array($r, $g, $b);
   return $rgb;
}

// Shuffle associative and non-associative array
function axs_shuffle_assoc($list) {
  if (!is_array($list)) return $list;

  $keys = array_keys($list);
  shuffle($keys);
  $random = array();
  foreach ($keys as $key)
    $random[$key] = $list[$key];

  return $random;
}

function flame_strip_tags( $html, $allowed_tags = array() ) {
	if ( empty($html) ) {
		return '';
	}
	if ( is_object($html) ) {
		$html = (array) $html;
	}
    $allowed_tags = array_change_key_case($allowed_tags, CASE_LOWER);
    $rhtml = preg_replace_callback( '/<\/?([^>\s]+)[^>]*>/i', function ($matches) use (&$allowed_tags) {
        return in_array( strtolower( $matches[1] ), $allowed_tags ) ? $matches[0] : '';
    }, $html);
    return is_string($rhtml) ? $rhtml : '';
}

// get all URLs from string
function axs_geturls($string) {
    $regex = '#\bhttps?://[^\s()<>]+(?:\([\w\d]+\)|([^[:punct:]\s]|/))#';
    preg_match_all($regex, $string, $matches);
    return $matches[0] ?? '';
}

function axs_getsrc($html) {
    preg_match_all('/<img[^>]+>/i', $html, $rawimagearray, PREG_SET_ORDER);
    if ( isset($rawimagearray[0][0]) ) {
        preg_match('@src="([^"]+)"@', $rawimagearray[0][0], $match);
        $img['src'] = @array_pop($match);
        preg_match('@width="([^"]+)"@', $rawimagearray[0][0], $matchwidth);
        $img['width'] = @array_pop($matchwidth);
        
        return (isset($img['width']) && $img['width'] < 10) ? false : $img;
    }
}

function axs_getimgproxy($img) {
	$img_enc = urlencode($img);
	$token = md5(($img_enc ?? '').($_SERVER['SERVER_ADDR'] ?? '').($_SERVER['SERVER_ADMIN'] ?? '').($_SERVER['SERVER_NAME'] ?? '').($_SERVER['SERVER_PORT'] ?? '').($_SERVER['SERVER_PROTOCOL'] ?? '').strip_tags($_SERVER['SERVER_SIGNATURE'] ?? '').($_SERVER['SERVER_SOFTWARE'] ?? '').($_SERVER['DOCUMENT_ROOT'] ?? ''));
	$imgStr = 'refresh=3600&token='.$token.'&src='.($img_enc ?? '');
	return esc_url( add_query_arg( 'sbimg', base64_encode($imgStr), site_url('index.php') ) );
}

function axs_log($value) {
    // Create if not exists
    if ( ! file_exists(AXS_LOGFILE) ) {
        $fp = fopen(AXS_LOGFILE, 'w');
        fclose($fp);
    }
    else{
		$file = array_filter(array_map("trim", file(AXS_LOGFILE)));
		if(count($file) > 300){
			$file = array_slice($file, -300);
			$out = array();
			foreach ($file as $k => $v){
				$out[] = $v;
				if($k > 0 && is_int(( $k + 1 ) / 2)){
					$out[] = PHP_EOL ;
				}
			}
			file_put_contents(AXS_LOGFILE, implode(PHP_EOL , $out));
		}
	}
    
    SB_AdminPageFramework_Debug::log( $value, AXS_LOGFILE );
}

// modify widget
add_filter('widget_display_callback', 'axs_widget_custom_display', 10, 3);
function axs_widget_custom_display($instance, $widget, $args) {
    if ($widget instanceof SB_AdminPageFramework_Widget_Factory) {
    	$instance['sb_type'] = 'feed';
    	
	    if (isset($instance['override_settings']) && !empty($instance['override_settings'])) {
	        parse_str($instance['override_settings'], $aOverride);
	        $instance = array_merge($aOverride, $instance);
	        unset($instance['override_settings']);
	    }

	    // remove widget header if required
	    $instance['label'] = $instance['title'] ?? '';
	    if ( (isset($instance['position']) && $instance['position'] == 'sticky') || (isset($instance['showheader']) && !empty($instance['showheader'])) ) {
	        unset($instance['title']);
	    }
	    
	    if ( isset($instance['position']) && $instance['position'] == 'sticky' ) {
	        // remove from sidebar if is sticky
	        $sb = new SocialBoard();
	        $sb->attr = $instance;
	        $sb->args = $args;
	        add_action( 'wp_footer', array(&$sb, 'run') );
	        
	        return false;
	    }
	    else
	        return $instance;
    }
    else
		return $instance;
}

// function for using in template files
function social_board( $atts ) {
	if($atts["type"] == "masonry") $atts["type"] = "wall";
    $sb = new SocialBoard();
    return $sb->init( $atts, false );
}

// register the plugin for the first time
$sb = new SocialBoard();
$sb->start();

// remove possible filters
add_filter( 'the_content', 'axs_disable_wpautop', 0 );
function axs_disable_wpautop( $content ) {
	'sb_posts' === get_post_type() && remove_filter( 'the_content', 'wpautop' );
	return $content;
}

// image display proxy
add_action( 'template_redirect', 'axs_image_proxy');
function axs_image_proxy() {
    global $wp_query;
    
    $sbimg = $wp_query->get( 'sbimg' );
    if ( ! empty( $sbimg ) ) {
        // load image proxy
        include( AXS_DIRNAME . '/library/ImageCache.php' );
        $imgcache = new AXS_ImageCache;
        $imgcache->imageCacheDir = AXS_DIRNAME . '/cache/';
        
        // convert string
        parse_str(base64_decode($sbimg), $imgArr);
        $imgcache->cacheImage($imgArr);
    }
}

function axs_custom_rewrite_tag() {
    add_rewrite_tag('%sbimg%', '([^&]+)');
}
add_action('init', 'axs_custom_rewrite_tag', 10, 0);

// get enabled layouts
function axs_enabled_layouts() {
    $setoption = SB_AdminPageFramework::getOption( 'AXS_Settings_Page' ) ? SB_AdminPageFramework::getOption( 'AXS_Settings_Page' ) : SB_AdminPageFramework::getOption( 'SB_Settings_Page' );
    $enabled_layouts = $setoption['section_layouts']['enabled_layouts'] ?? '';
	if ( ! $enabled_layouts ) {
        $enabled_layouts = array(
            'wall' => true,
            'grid' => false,
            'timeline' => false,
            'feed' => false,
            'carousel' => false
        );
	}
    return $enabled_layouts;
}

function axs_throw_error( $message ) {
    return '<p class="sboard-nodata"><strong>AX Social Stream: </strong> ' . $message . '</p>';
}

// integrations
add_action('plugins_loaded', 'axs_integration');
function axs_integration() {
    // Cornerstone
    if ( class_exists( 'Cornerstone_Plugin' ) ) {
        include( AXS_DIRNAME . '/integration/cornerstone.php' );
    }
    
    // Visual Composer
    if ( defined( 'WPB_VC_VERSION' ) ) {
        include( AXS_DIRNAME . '/integration/visualcomposer.php' );
	}

    // Elementor
    if ( defined( 'ELEMENTOR_VERSION' ) ) {
        include( AXS_DIRNAME . '/integration/elementor/elementor.php' );
	}
}

// Plugin Setup Manager 
if ( file_exists( __DIR__ . '/inc/plugin-setup.php' ) ) {
    require_once __DIR__ . '/inc/plugin-setup.php';
}
