<?php
/**
 * Alpha Circle Progressbar Element
 *
 * @author  D-THEMES
 * @package Alpha Core Framework
 * @subpackage Core
 * @since 1.3.0
 */

use Elementor\Plugin;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Utils;
use Elementor\Core\Schemes\Color;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Image_Size;

class Alpha_Circle_Progressbar_Elementor_Widget extends \Elementor\Widget_Base {

	public function get_name() {
		return ALPHA_NAME . '_widget_circle_progressbar';
	}

	public function get_title() {
		return esc_html__( 'Circle Progressbar', 'alpha-core' );
	}

	public function get_icon() {
		return 'alpha-elementor-widget-icon eicon-counter-circle';
	}

	public function get_categories() {
		return array( 'alpha_widget' );
	}

	public function get_keywords() {
		return array( 'progress bar', 'chart' );
	}

	protected function is_dynamic_content(): bool {
		return false;
	}

	public function get_style_depends() {
		return array( 'alpha-circle-progressbar' );
	}

	public function get_script_depends() {
		return array( 'easypiechart', 'alpha-circle-progressbar' );
	}

	public function has_widget_inner_wrapper(): bool {
		return ! Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
	}

	protected function register_controls() {

		$this->start_controls_section(
			'section_circular_bar',
			array(
				'label' => esc_html__( 'Circular Bar', 'alpha-core' ),
			)
		);

		$this->add_control(
			'title',
			array(
				'label'   => esc_html__( 'Title', 'alpha-core' ),
				'type'    => Controls_Manager::TEXT,
				'dynamic' => array(
					'active' => true,
				),
			)
		);

		$this->add_control(
			'value',
			array(
				'label'   => esc_html__( 'Progressbar Value', 'alpha-core' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 70,
				'dynamic' => array(
					'active' => true,
				),
			)
		);

		$this->add_control(
			'units',
			array(
				'type'        => Controls_Manager::TEXT,
				'label'       => esc_html__( 'Units', 'alpha-core' ),
				'default'     => '%',
				'description' => esc_html__( 'Enter measurement units (Example: %, px, points, etc).', 'alpha-core' ),
			)
		);

		$this->add_control(
			'view',
			array(
				'type'    => Controls_Manager::SELECT,
				'label'   => esc_html__( 'View Mode', 'alpha-core' ),
				'options' => array(
					''           => esc_html__( 'Title and Value', 'alpha-core' ),
					'only-title' => esc_html__( 'Only Title', 'alpha-core' ),
					'only-value' => esc_html__( 'Only Value', 'alpha-core' ),
					'only-icon'  => esc_html__( 'Only Icon', 'alpha-core' ),
				),
			)
		);

		$this->add_control(
			'icon_cl',
			array(
				'type'             => Controls_Manager::ICONS,
				'label'            => esc_html__( 'Select Icon', 'alpha-core' ),
				'fa4compatibility' => 'icon',
				'default'          => array(
					'value'   => ALPHA_ICON_PREFIX . '-icon-verification',
					'library' => 'alpha-icons',
				),
				'condition'        => array(
					'view' => 'only-icon',
				),
			)
		);

		$this->add_control(
			'icon_size',
			array(
				'label'     => esc_html__( 'Icon Size (px)', 'alpha-core' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'step' => 1,
						'min'  => 1,
						'max'  => 50,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .circular-bar i'   => 'font-size: {{SIZE}}px;',
					'{{WRAPPER}} .circular-bar svg' => 'width: {{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}}',
				),
				'condition' => array(
					'view' => 'only-icon',
				),
			)
		);

		$this->add_control(
			'icon_color',
			array(
				'type'      => Controls_Manager::COLOR,
				'label'     => esc_html__( 'Icon Color', 'alpha-core' ),
				'selectors' => array(
					'.elementor-element-{{ID}} .circular-bar i' => 'color: {{VALUE}};',
				),
				'condition' => array(
					'view'              => 'only-icon',
					'icon_cl[library]!' => 'svg',
				),
			)
		);

		$this->add_control(
			'icon_svg_stroke',
			array(
				'label'     => esc_html__( 'Stroke Color', 'alpha-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'.elementor-element-{{ID}} .circular-bar svg' => 'stroke: {{VALUE}};',
				),
				'condition' => array(
					'view'             => 'only-icon',
					'icon_cl[library]' => 'svg',
				),
			)
		);

		$this->add_control(
			'icon_svg_fill',
			array(
				'label'     => esc_html__( 'Fill Color', 'alpha-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'.elementor-element-{{ID}} .circular-bar svg' => 'fill: {{VALUE}};',
				),
				'condition' => array(
					'view'             => 'only-icon',
					'icon_cl[library]' => 'svg',
				),
			)
		);

		$this->add_control(
			'linecap',
			array(
				'type'        => Controls_Manager::SELECT,
				'label'       => esc_html__( 'Bar Shape', 'alpha-core' ),
				'description' => esc_html__( 'Choose how the corner of the bar line looks like.', 'alpha-core' ),
				'default'     => 'square',
				'options'     => array(
					'square' => esc_html__( 'Square', 'alpha-core' ),
					'round'  => esc_html__( 'Round', 'alpha-core' ),
				),
			)
		);

		$this->add_control(
			'speed',
			array(
				'type'    => Controls_Manager::NUMBER,
				'label'   => esc_html__( 'Animation Speed (ms)', 'alpha-core' ),
				'default' => 2500,
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'general_style',
			array(
				'label' => esc_html__( 'General', 'alpha-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'content_pos',
			array(
				'label'       => esc_html__( 'Content Position (%)', 'alpha-core' ),
				'description' => esc_html__( 'Type a certain number for content position.', 'alpha-core' ),
				'type'        => Controls_Manager::SLIDER,
				'range'       => array(
					'px' => array(
						'step' => 1,
						'min'  => 1,
						'max'  => 200,
					),
				),
				'selectors'   => array(
					'.elementor-element-{{ID}} .circular-bar .bar-content' => 'top: {{SIZE}}%;',
				),
			)
		);

		$this->add_responsive_control(
			'content_spacing',
			array(
				'label'       => esc_html__( 'Content Spacing', 'alpha-core' ),
				'description' => esc_html__( 'Type a certain number for spacing between title and value.', 'alpha-core' ),
				'type'        => Controls_Manager::SLIDER,
				'default'     => array(
					'size' => 10,
				),
				'size_units'  => array(
					'px',
					'rem',
				),
				'range'       => array(
					'px' => array(
						'step' => 1,
						'min'  => 1,
						'max'  => 50,
					),
				),
				'condition'   => array(
					'view' => '',
				),
				'selectors'   => array(
					'.elementor-element-{{ID}} .circular-bar strong' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'size',
			array(
				'type'    => Controls_Manager::NUMBER,
				'label'   => esc_html__( 'Circle Size', 'alpha-core' ),
				'default' => 175,
				'min'     => 10,
				'max'     => 500,
			)
		);

		$this->add_control(
			'barcolor',
			array(
				'type'        => Controls_Manager::COLOR,
				'label'       => esc_html__( 'Bar color', 'alpha-core' ),
				'description' => esc_html__( 'Select progressbar color. Please clear this if you want to use the default color.', 'alpha-core' ),
			)
		);

		$this->add_control(
			'trackcolor',
			array(
				'type'        => Controls_Manager::COLOR,
				'label'       => esc_html__( 'Track Color', 'alpha-core' ),
				'default'     => '#D4DAE1',
				'description' => esc_html__( 'Choose the color of the track. Please clear this if you want to use the default color.', 'alpha-core' ),
			)
		);

		$this->add_control(
			'line',
			array(
				'type'    => Controls_Manager::NUMBER,
				'label'   => esc_html__( 'Line Width (px)', 'alpha-core' ),
				'default' => 5,
				'min'     => 1,
				'max'     => 50,
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_style',
			array(
				'label'     => esc_html__( 'Title', 'alpha-core' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'view' => array( '', 'only-title' ),
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'selector' => '.elementor-element-{{ID}} .circular-bar strong',
			)
		);

		$this->add_control(
			'title_color',
			array(
				'label'       => esc_html__( 'Title Color', 'alpha-core' ),
				'description' => esc_html__( 'Control the title color.', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'selectors'   => array(
					'.elementor-element-{{ID}} .circular-bar strong' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'value_style',
			array(
				'label'     => esc_html__( 'Value', 'alpha-core' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'view' => array( '', 'only-value' ),
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'value_typography',
				'selector' => '.elementor-element-{{ID}} .circular-bar label',
			)
		);

		$this->add_control(
			'value_color',
			array(
				'label'       => esc_html__( 'Value Color', 'alpha-core' ),
				'description' => esc_html__( 'Control the value color.', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'selectors'   => array(
					'.elementor-element-{{ID}} .circular-bar label' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$atts         = $this->get_settings_for_display();
		$atts['self'] = $this;
		if ( isset( $atts['icon_cl'] ) && isset( $atts['icon_cl']['value'] ) ) {
			$atts['icon'] = $atts['icon_cl']['value'];
		}
		require alpha_core_e_templates_path( 'widgets/circle-progressbar/render-circle-progressbar-elementor' );
	}

	protected function content_template() {
		?>
		<#
			let options = {};
			options['trackColor']          = settings.trackcolor;
			options['barColor']            = settings.barcolor ? settings.barcolor : '<?php echo esc_js( function_exists( 'alpha_get_option' ) ? alpha_get_option( 'primary_color' ) : '#fd7800' ); ?>';
			options['lineCap']             = settings.linecap;
			options['lineWidth']           = settings.line;
			options['size']                = settings.size;
			options['animate']             = {};
			options['animate']['duration'] = settings.speed;

			view.addRenderAttribute( 'wrapper', 'class', 'circular-bar center' );
			if ( settings.type ) {
				view.addRenderAttribute( 'wrapper', 'class', settings.type );
			}
			if ( settings.view ) {
				view.addRenderAttribute( 'wrapper', 'class', settings.view );
			}
			view.addRenderAttribute( 'title', 'class', 'bar-title' );
			view.addInlineEditingAttributes( 'title' );
			view.addRenderAttribute( 'innerWrapper', 'data-percent', settings.value );
			view.addRenderAttribute( 'innerWrapper', 'data-plugin-options', JSON.stringify( options ) );
			view.addRenderAttribute( 'innerWrapper', 'style', 'height:' + Number( settings.size ) + 'px;' );
		#>
		<div {{{ view.getRenderAttributeString( 'wrapper' ) }}}>
			<div class="circular-bar-chart" {{{ view.getRenderAttributeString( 'innerWrapper' ) }}}>
				<div class="bar-content">
			<#
				if ( 'only-icon' == settings.view ) {
					if( 'svg' == settings.icon_cl.library ) {
						var svgHtml = elementor.helpers.renderIcon( view, settings.icon_cl, { 'aria-hidden': true } );
					#>
						{{{ svgHtml.value }}}
					<#} else {
						view.addRenderAttribute( 'icon', 'class', settings.icon_cl.value );
					#>
						<i {{{ view.getRenderAttributeString( 'icon' ) }}}></i>
					<#
					}
			#>
			<# } else if ( 'only-title' == settings.view ) { #>
				<# if ( settings.title ) { #>
					<strong {{{ view.getRenderAttributeString( 'title' ) }}}>{{{ settings.title }}}</strong>
				<# } #>
			<# } else { #>
				<# if ( settings.title && 'only-value' !== settings.view ) { #>
					<strong {{{ view.getRenderAttributeString( 'title' ) }}}>{{{ settings.title }}}</strong>
				<# } #>
				<label><span class="percent">0</span>{{{ settings.units }}}</label>
				</div>
			<# } #>
			</div>
		</div>
		<?php
	}
}
