/**
 * Guest File Upload - AJAX Handler
 */
jQuery(document).ready(function($) {
    var uploadedFiles = {};
    
    // File upload handler for all guest file inputs
    $(document).on('change', 'input[type="file"][name^="guest_"]', function() {
        var fileInput = this;
        var file = fileInput.files[0];
        var fieldName = fileInput.name;
        var $parent = $(fileInput).closest('.customer-form-row');
        
        // Clear previous messages
        $parent.find('.file-error, .file-success').remove();
        
        if (!file) {
            delete uploadedFiles[fieldName];
            return;
        }
        
        // Validate file size
        if (file.size > togo_checkout.max_file_size) {
            $parent.append('<span class="file-error">File is too large. Maximum size is 5MB.</span>');
            fileInput.value = '';
            delete uploadedFiles[fieldName];
            return;
        }
        
        // Validate file type
        if (togo_checkout.allowed_types.indexOf(file.type) === -1) {
            $parent.append('<span class="file-error">Invalid format. Only PDF, JPG, PNG are allowed.</span>');
            fileInput.value = '';
            delete uploadedFiles[fieldName];
            return;
        }
        
        // Show uploading
        $parent.append('<span class="file-uploading">Uploading...</span>');
        $('#place_order').prop('disabled', true);
        
        // Prepare form data
        var formData = new FormData();
        formData.append(fieldName, file);
        formData.append('action', 'upload_guest_files');
        formData.append('nonce', togo_checkout.nonce);
        
        // AJAX upload
        $.ajax({
            url: togo_checkout.ajax_url,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                $parent.find('.file-uploading').remove();
                $('#place_order').prop('disabled', false);
                
                if (response.success && response.data[fieldName] && response.data[fieldName].success) {
                    uploadedFiles[fieldName] = response.data[fieldName];
                    $parent.append('<span class="file-success">✓ ' + file.name + ' (' + Math.round(file.size/1024) + 'KB)</span>');
                } else {
                    var errorMsg = response.data[fieldName] ? response.data[fieldName].error : 'Upload failed';
                    $parent.append('<span class="file-error">Error: ' + errorMsg + '</span>');
                    fileInput.value = '';
                    delete uploadedFiles[fieldName];
                }
            },
            error: function() {
                $parent.find('.file-uploading').remove();
                $parent.append('<span class="file-error">Connection error. Please try again.</span>');
                fileInput.value = '';
                delete uploadedFiles[fieldName];
            }
        });
    });
    
    // Before form submit, validate guest information and files
    $('form.checkout').on('checkout_place_order', function() {
        var errors = [];
        
        // Check guest information fields dynamically
        $('.guest-section').each(function() {
            var guestNumber = $(this).data('guest');
            var $guestSection = $(this);
            
            // Validate all input fields in this guest section
            $guestSection.find('input[required], select[required], textarea[required]').each(function() {
                var $field = $(this);
                var fieldName = $field.attr('name');
                var fieldValue = $field.val();
                var fieldLabel = $field.closest('.customer-form-row').find('label').text().replace('*', '').trim();
                var fieldType = $field.attr('type') || $field.prop('tagName').toLowerCase();
                
                // Required field validation
                if (!fieldValue || fieldValue.trim() === '') {
                    errors.push('Guest ' + guestNumber + ': ' + fieldLabel + ' is required');
                    return;
                }
                
                // Type-specific validation
                switch (fieldType) {
                    case 'text':
                        if (fieldValue.trim().length < 2) {
                            errors.push('Guest ' + guestNumber + ': ' + fieldLabel + ' must be at least 2 characters');
                        }
                        break;
                        
                    case 'number':
                        var numValue = parseInt(fieldValue);
                        if (isNaN(numValue) || numValue < 1 || numValue > 120) {
                            errors.push('Guest ' + guestNumber + ': ' + fieldLabel + ' must be a valid number (1-120)');
                        }
                        break;
                        
                    case 'email':
                        var emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                        if (!emailPattern.test(fieldValue)) {
                            errors.push('Guest ' + guestNumber + ': ' + fieldLabel + ' must be a valid email address');
                        }
                        break;
                        
                    case 'tel':
                        var phonePattern = /^[\d\s\+\-\(\)]{8,}$/;
                        if (!phonePattern.test(fieldValue.replace(/\s/g, ''))) {
                            errors.push('Guest ' + guestNumber + ': ' + fieldLabel + ' must be a valid phone number');
                        }
                        break;
                        
                    case 'textarea':
                        if (fieldValue.trim().length < 10) {
                            errors.push('Guest ' + guestNumber + ': ' + fieldLabel + ' must be at least 10 characters');
                        }
                        break;
                }
            });
        });
        
        // Check required file uploads dynamically
        $('input[type="file"][data-required="true"]').each(function() {
            var fieldName = this.name;
            var $field = $(this);
            var fieldLabel = $field.closest('.customer-form-row').find('label').text().replace('*', '').trim();
            
            // Extract guest number from field name
            var guestNumberMatch = fieldName.match(/_(\d+)$/);
            var guestNumber = guestNumberMatch ? guestNumberMatch[1] : '';
            
            if (!uploadedFiles[fieldName]) {
                errors.push('Guest ' + guestNumber + ': Please upload ' + fieldLabel);
            }
        });
        
        if (errors.length > 0) {
            alert(errors.join('\n') + '\n' + 'Please check your information and try again.');
            return false;
        }
        
        return true;
    });
});