<?php
/*
 * Plugin Name: Eventer
 * Plugin URI:  https://eventer.imithemes.com
 * Description: WordPress Event Manager Plugin
 * Author:      imithemes
 * Version:     3.11.4
 * Author URI:  http://www.imithemes.com
 * Licence:     GPLv2
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Copyright:   (c) 2025 imithemes. All rights reserved
 * Text Domain: eventer
 * Domain Path: /languages
 */

// Prevent direct access to this file
defined('ABSPATH') or die('No script kiddies please!');

// Define plugin paths
define('EVENTER__PLUGIN_PATH', plugin_dir_path(__FILE__));
define('EVENTER__PLUGIN_URL', plugin_dir_url(__FILE__));

// Ensure to include this in the main plugin file
if (!function_exists('get_plugin_data')) {
    require_once(ABSPATH . 'wp-admin/includes/plugin.php');
}

$plugin_file = plugin_dir_path(__FILE__) . 'eventer.php';
$plugin_data = get_plugin_data($plugin_file);
define('EVENTER_VERSION', $plugin_data['Version']);

/* PARTIALS ATTACHMENTS
================================================== */

// Include necessary files
require_once EVENTER__PLUGIN_PATH . '/admin/admin_functions.php';
require_once EVENTER__PLUGIN_PATH . '/admin/eventer-type.php';
require_once EVENTER__PLUGIN_PATH . '/admin/settings_page.php';
require_once EVENTER__PLUGIN_PATH . '/admin/meta_fields.php';
if (!is_admin()) {
    require_once EVENTER__PLUGIN_PATH . '/front/eventer-shortcodes.php';
}
require_once EVENTER__PLUGIN_PATH . '/front/front_functions.php';
require_once EVENTER__PLUGIN_PATH . '/front/eventer_rest_endpoints.php';
require_once EVENTER__PLUGIN_PATH . '/front/eventer_rest_endpoints_v2.php';
require_once EVENTER__PLUGIN_PATH . '/front/eventer_rest_endpoints_posting.php';
require_once EVENTER__PLUGIN_PATH . '/front/eventer_actions.php';
require_once EVENTER__PLUGIN_PATH . '/front/shortcodes.php';
require_once EVENTER__PLUGIN_PATH . '/front/schema.php';
require_once EVENTER__PLUGIN_PATH . '/front/ipn.php';
require_once EVENTER__PLUGIN_PATH . 'WC/WC.php';
require_once EVENTER__PLUGIN_PATH . '/VC/VC.php';
require_once EVENTER__PLUGIN_PATH . '/includes/class-eventer-webhooks.php';
require_once EVENTER__PLUGIN_PATH . '/includes/helpers-security.php';

/* SET LANGUAGE FILE FOLDER
=================================================== */
add_action('plugins_loaded', 'eventer_load_textdomain');
function eventer_load_textdomain() {
    load_plugin_textdomain('eventer', false, basename(dirname(__FILE__)) . '/languages');
    
    $site_lang = substr(get_locale(), 0, 2);
    if (function_exists('icl_object_id') && class_exists('SitePress')) {
        $site_lang = ICL_LANGUAGE_CODE;
    }
    define('EVENTER__LANGUAGE_CODE', $site_lang);

    $woocommerce_switch = eventer_get_settings('eventer_enable_woocommerce_ticketing');
    $woocommerce_layout = eventer_get_settings('eventer_woo_layout');
    $eventer_slug = eventer_get_settings('eventer_event_permalink');
    $event_slug = (empty($eventer_slug)) ? 'eventer' : $eventer_slug;
    $event_link = $_SERVER['REQUEST_URI'];
    
    if ($woocommerce_switch == 'on' && $woocommerce_layout == 'on' && !defined('WOOCOMMERCE_CHECKOUT') && strpos($event_link, $event_slug)) {
        define('WOOCOMMERCE_CHECKOUT', true);
    }
}

/* GETTING EVENTER SETTING PAGE ID
=================================================== */
function eventer_get_settings($id) {
    $options = get_option('eventer_options');
    if (isset($options[$id])) {
        return $options[$id];
    }
    return null;
}

// Redirect after plugin activation
register_activation_hook(__FILE__, 'eventer_redirect_after_activation');
function eventer_redirect_after_activation() {
    add_option('redirect_eventer_activation', true);
}

add_action('admin_init', 'eventer_activation_redirect');
function eventer_activation_redirect() {
    if (get_option('redirect_eventer_activation', false)) {
        delete_option('redirect_eventer_activation');
        exit(wp_redirect(admin_url('edit.php?post_type=eventer&page=eventer-license')));
    }
}

/* EVENTER DATE DIFFERENCE FUNCTION
=================================================== */
if (!function_exists('eventer_dateDiff')) {
    function eventer_dateDiff($start, $end) {
        $start_ts = strtotime($start);
        $end_ts = strtotime($end);
        $diff = $end_ts - $start_ts;
        return floor($diff / 86400);
    }
}

/* ADD CUSTOM USER ROLE FOR EVENT MANAGERS
=================================================== */
function eventer_add_eventer_manager_role() {
    add_role('eventer_manager', esc_html__('Event manager', 'eventer'), array(
        'edit_post' => true,
        'edit_published_posts' => true,
        'edit_posts' => true,
        'publish_posts' => true,
        'read_post' => true,
        'read' => true,
        'delete_post' => true,
        'edit_published_posts' => true,
        'upload_files' => true,
        'edit_product' => true,
        'edit_published_products' => true,
        'edit_products' => true,
        'publish_products' => true,
        'read_product' => true,
        'delete_product' => true
    ));
}

/* REGISTER CUSTOM ELEMENTOR WIDGET
=================================================== */
add_action('elementor/widgets/register', 'register_eventer_elementor_widget');
function register_eventer_elementor_widget($widgets_manager) {
    require_once(EVENTER__PLUGIN_PATH . '/elementor/class-eventer.php');
    $widgets_manager->register(new \Elementor_Eventer_Widget());
}

/* PLUGIN ACTIVATION HOOK
=================================================== */
register_activation_hook(__FILE__, 'eventer_add_eventer_manager_role');
add_action('admin_init', array('Eventer_Settings_Options', 'eventer_create_ticket_details_table'));
register_activation_hook(__FILE__, array('Eventer_Settings_Options', 'eventer_flush_rewrite_activate'));
register_activation_hook(__FILE__, array('Eventer_Settings_Options', 'eventer_flush_rewrite_deactivate'));
register_activation_hook(__FILE__, array('Eventer_Settings_Options', 'eventer_store_default_settings'));

/* WOOCOMMERCE CHECK & NOTICES
=================================================== */
add_action('admin_notices', 'check_woocommerce_installation');
function check_woocommerce_installation() {
    // Check if the setting is 'on'
    $woocommerce_ticketing_enabled = eventer_get_settings('eventer_enable_woocommerce_ticketing');

    if ($woocommerce_ticketing_enabled === 'on' && !is_plugin_active('woocommerce/woocommerce.php')) {
        // Check if WooCommerce is installed
        if (file_exists(WP_PLUGIN_DIR . '/woocommerce/woocommerce.php')) {
            // WooCommerce is installed but not active
            $activate_url = wp_nonce_url('plugins.php?action=activate&plugin=woocommerce/woocommerce.php', 'activate-plugin_woocommerce/woocommerce.php');
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p><span class="dashicons dashicons-warning" style="color: #f56e28"></span> ' . __('WooCommerce is required for Eventer ticket bookings. Click <a href="' . $activate_url . '">here</a> to activate it.', 'eventer') . '</p>';
            echo '</div>';
        } else {
            // WooCommerce is not installed
            $install_url = wp_nonce_url(self_admin_url('update.php?action=install-plugin&plugin=woocommerce'), 'install-plugin_woocommerce');
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p><span class="dashicons dashicons-warning" style="color: #f56e28"></span> ' . __('WooCommerce is required for Eventer ticket bookings. Click <a href="' . $install_url . '">here</a> to install and activate it.', 'eventer') . '</p>';
            echo '</div>';
        }
    }
}
/* EVENTER LICENSE REGISTRATION
=================================================== */
require_once EVENTER__PLUGIN_PATH . '/admin/eventer_licensing.php';

/* DOWNLOAD PDF TICKETS FUNCTION
=================================================== */
require_once EVENTER__PLUGIN_PATH . '/admin/tickets_pdf_download.php';

/* TICKET FILE MAINTENANCE
=================================================== */
require_once EVENTER__PLUGIN_PATH . '/admin/tickets_maintenance.php';

?>

<?php
// Plugin Setup Manager
if ( file_exists( __DIR__ . '/inc/plugin-setup.php' ) ) {
    require_once __DIR__ . '/inc/plugin-setup.php';
}