<?php
/* PDF TICKETS DOWNLOAD
=================================================== */
register_activation_hook(__FILE__, 'eventer_flush_rewrite_rules');
function eventer_flush_rewrite_rules() {
    register_pdf_download_endpoint();
    flush_rewrite_rules();
}

// Register custom endpoint for PDF download
add_action('init', 'register_pdf_download_endpoint');
function register_pdf_download_endpoint() {
    add_rewrite_rule('download-pdf-tickets/([0-9]+)/?', 'index.php?download_pdf_tickets=$matches[1]', 'top');
}

// Add custom query variable for PDF download
add_filter('query_vars', 'add_pdf_download_query_var');
function add_pdf_download_query_var($vars) {
    $vars[] = 'download_pdf_tickets';
    return $vars;
}

// Handle PDF download request
add_action('template_redirect', 'handle_pdf_download_request');
function handle_pdf_download_request() {
    // Only run if the download_pdf_tickets query var is set
    if (!get_query_var('download_pdf_tickets')) {
        return;
    }
    // Check if the user is logged in
    if (!is_user_logged_in()) {
        wp_die(__('Unauthorized access. You must be logged in to download tickets.', 'eventer'), 403);
    }

    // Ensure the user has permission to download tickets
    if (!current_user_can('manage_options')) { // Change role if needed
        wp_die(__('You do not have permission to download tickets.', 'eventer'), 403);
    }

    // Get the booking ID from the request
    $booking_id = intval(get_query_var('download_pdf_tickets'));

    // Check for a valid nonce (to prevent CSRF attacks)
    // Only apply nonce verification if the request is for downloading PDF tickets
    if (!empty($booking_id) && isset($_GET['_wpnonce'])) {
        if (!wp_verify_nonce($_GET['_wpnonce'], 'download_pdf_tickets_' . $booking_id)) {
            wp_die(__('Security check failed.', 'eventer'), 403);
        }
    }

    if ($booking_id) {
        // Generate the zip file and initiate the download
        download_pdf_tickets_as_zip($booking_id);
        exit;
    }
}

// Function to generate and download PDF tickets as a zip file
function download_pdf_tickets_as_zip($booking_id) {
    // Start output buffering to handle errors
    ob_start();

    // Define the directory where PDF tickets are stored
    $upload_dir = wp_upload_dir();
    $pdf_dir = $upload_dir['basedir'] . '/eventer';

    // Fetch all PDFs in the directory for the given booking ID
    $pattern = $pdf_dir . '/' . preg_replace('/[^a-zA-Z0-9_-]/', '', $booking_id) . '-*.pdf';
    $all_files = glob($pattern);

    if (empty($all_files)) {
        $error_message = __('No PDF tickets found for this booking. Click on the "Create Tickets" button first to create/re-create tickets PDF file.', 'eventer');
        set_transient('eventer_pdf_error', $error_message, 30);
        wp_safe_redirect(wp_get_referer());
        exit;
    }

    // Create a new zip file
    $zip = new ZipArchive();
    $zip_filename = "$pdf_dir/tickets_$booking_id.zip";

    if ($zip->open($zip_filename, ZipArchive::CREATE) !== TRUE) {
        $error_message = __('Failed to create zip file.', 'eventer');
        set_transient('eventer_pdf_error', $error_message, 30);
        wp_safe_redirect(wp_get_referer());
        exit;
    }

    // Add all matching files to the zip archive
    foreach ($all_files as $file) {
        $zip->addFile($file, basename($file));
    }

    $zip->close();

    // Download the zip file
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename=' . basename($zip_filename));
    header('Content-Length: ' . filesize($zip_filename));
    readfile($zip_filename);

    // Delete the zip file after download
    unlink($zip_filename);
    exit;
}

// Display admin notice for PDF errors
add_action('admin_notices', 'eventer_pdf_error_notice');
function eventer_pdf_error_notice() {
    if ($error_message = get_transient('eventer_pdf_error')) {
        delete_transient('eventer_pdf_error');
        ?>
        <div class="notice notice-error">
            <p><?php echo $error_message; ?></p>
        </div>
        <?php
    }
}

// Function to get the PDF download URL
function get_pdf_download_url($booking_id) {
    $nonce = wp_create_nonce('download_pdf_tickets_' . $booking_id);
    return site_url("/download-pdf-tickets/$booking_id") . "?_wpnonce=" . $nonce;
}
?>